#include "permissions.h"

#include <drive/backend/actions/flag.h>
#include <drive/backend/actions/info_access.h>
#include <drive/backend/actions/override_settings.h>
#include <drive/backend/actions/tag.h>
#include <drive/backend/cars/car.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/logging/events.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/offers/actions/abstract.h>
#include <drive/backend/promo_codes/common/action.h>

IDynamicActionChecker::ECheckResult TUserPermissions::CheckActionAcceptable(const TUserAction& action, const NDrive::IServer& server) const {
    IDynamicActionChecker::TPtr checkerImpl = IDynamicActionChecker::TFactory::Construct(action.GetType());
    if (!checkerImpl) {
        return IDynamicActionChecker::ECheckResult::EOk;
    }
    return checkerImpl->CheckAction(GetUserId(), *this, action, server);
}

const TTagsFilter* TUserPermissions::GetFilterByObjectAction(const TObjectAccessAction::EAction action) const {
    auto it = FiltersByObjectActions.find(action);
    if (it == FiltersByObjectActions.end()) {
        return nullptr;
    }
    return &it->second;
}

TVector<TDBTag> TUserPermissions::FilterObserveTags(const TVector<TDBTag>& tags) const {
    TVector<TDBTag> result;
    const TSet<TString>* tagNamesObserve = GetTagNamesByActionPtr(TTagAction::ETagAction::Observe);
    if (!tagNamesObserve) {
        return result;
    }
    for (auto&& i : tags) {
        if (tagNamesObserve->contains(i->GetName())) {
            result.emplace_back(i);
        }
    }
    return result;
}

bool TUserPermissions::CheckObjectSelector(const TTaggedObject& obj, const TObjectAccessAction::EAction action) const {
    auto it = ObjectSelectorIntervals.find(action);
    if (it == ObjectSelectorIntervals.end() || it->second.empty()) {
        return true;
    }
    auto hash = TObjectAccessAction::CalcHash(obj.GetId());
    auto timestamp = ModelingNow();
    for (auto&& i : it->second) {
        if (i->IsMatching(obj.GetTags()) && !i->Check(hash, timestamp)) {
            return false;
        }
    }
    return true;
}

bool TUserPermissions::CheckObjectHashWithFilter(const TString& id, const TObjectAccessAction::EAction action) const {
    auto it = ObjectSelectorIntervalsWithFilters.find(action);
    if (it == ObjectSelectorIntervalsWithFilters.end() || it->second.empty()) {
        return true;
    }
    auto hash = TObjectAccessAction::CalcHash(id);
    auto timestamp = ModelingNow();
    for (auto&& i : it->second) {
        if (i->Check(hash, timestamp)) {
            return true;
        }
    }
    return false;
}

bool TUserPermissions::CheckObjectTagAction(const TTagAction::ETagAction action, const TConstDBTag& tag, const TTaggedObject& object, const TObjectEvents<TConstDBTag>& history) const {
    if (tag->GetVisibilityFeatures(*this) & (ui64)action) {
        return true;
    }
    bool result = false;
    auto actionsIt = ConditionalTagActions.find(action);
    if (actionsIt == ConditionalTagActions.end()) {
        return result;
    }
    for (const auto& i : actionsIt->second) {
        if (i->Match(tag->GetName())) {
            switch (i->CheckObjectStatus(tag, *this, object.GetTags(), history)) {
                case TTagAction::ECheckResult::Deny:
                    return false;
                case TTagAction::ECheckResult::Accept:
                    result = true;
                    break;
                case TTagAction::ECheckResult::Ignore:
                    break;
                case TTagAction::ECheckResult::Problem:
                    return false;
            }
        }
    }
    return result;
}

TString TUserPermissions::ExplainInvisibility(TUnvisibilityInfoSet info) {
    TStringBuilder result;
    for (auto&& [value, name] : GetEnumNames<EUnvisibilityInfo>()) {
        if (info & value) {
            if (!result.empty()) {
                result << ',';
            }
            result << name;
        }
    }
    return result;
}

TUserPermissions::EVisibility TUserPermissions::GetVisibility(const TTaggedObject& object, const NEntityTagsManager::EEntityType entityType, TUnvisibilityInfoSet* unvisibilityInfo /*= nullptr*/, bool forceShowObjectWithoutTags /*= false*/) const {
    if (forceShowObjectWithoutTags && object.GetTags().empty()) {
        return EVisibility::Visible;
    }
    if (unvisibilityInfo) {
        *unvisibilityInfo = 0;
    }
    if (object.GetTags().empty()) {
        bool defaultVisibility = false;
        switch (entityType) {
            case NEntityTagsManager::EEntityType::Car:
                defaultVisibility = false;
                break;
            case NEntityTagsManager::EEntityType::Area:
            case NEntityTagsManager::EEntityType::Trace:
            case NEntityTagsManager::EEntityType::Account:
                defaultVisibility = true;
                break;
            case NEntityTagsManager::EEntityType::User:
                defaultVisibility = ((GetUserReportTraits() & (NUserReport::TReportTraits)NUserReport::EReportTraits::ReportHideNoTags) == 0);
            default:
                break;
        }
        if (defaultVisibility) {
            return EVisibility::Visible;
        } else {
            if (unvisibilityInfo) {
                *unvisibilityInfo = EUnvisibilityInfo::Unavailable | EUnvisibilityInfo::UnavailableByDefault;
            }
            return EVisibility::NoVisible;
        }
    }

    if (!CheckObjectSelector(object, TObjectAccessAction::EAction::Observe)) {
        bool selected = false;
        for (auto&& i : object.GetTags()) {
            if (i->GetPerformer() == GetUserId()) {
                selected = true;
                if (!CheckObjectTagAction(TTagAction::ETagAction::ObserveObjectsOnPerform, i, object, {})) {
                    return EVisibility::SelectedStrict;
                }
            }
        }
        if (selected) {
            return EVisibility::Selected;
        }
        if (unvisibilityInfo) {
            *unvisibilityInfo = EUnvisibilityInfo::Unavailable | EUnvisibilityInfo::UnavailableByObjectSelector;
        }
        return EVisibility::NoVisible;
    }

    bool provideByFilter;
    {
        const TTagsFilter* f = GetFilterByObjectAction(TObjectAccessAction::EAction::Observe);
        provideByFilter = f && f->IsMatching(object.GetTags());
        if (provideByFilter && !CheckObjectHashWithFilter(object.GetId(), TObjectAccessAction::EAction::Observe)) {
            if (unvisibilityInfo) {
                *unvisibilityInfo = EUnvisibilityInfo::Unavailable | EUnvisibilityInfo::UnavailableByObjectSelectorWithFilter;
            }
            return EVisibility::NoVisible;
        }
    }

    i32 maxPriority = -Max<i32>();
    bool visibleMaxPriority = provideByFilter;
    bool visibleOnBusy = false;
    bool selected = false;
    bool strict = false;
    TUnvisibilityInfoSet skip = 0;
    for (auto&& tag : object.GetTags()) {
        TObjectEvents<TConstDBTag> history;
        const i32 currentPriority = tag->GetTagPriority(-Max<i32>());
        if (currentPriority > maxPriority) {
            maxPriority = currentPriority;
            visibleMaxPriority = provideByFilter;
        }
        visibleOnBusy = visibleOnBusy || CheckObjectTagAction(TTagAction::ETagAction::ObserveBusyObject, tag, object, history);
        if (currentPriority == maxPriority) {
            visibleMaxPriority = visibleMaxPriority || CheckObjectTagAction(TTagAction::ETagAction::ObserveObject, tag, object, history) || CheckObjectTagAction(TTagAction::ETagAction::ForceObserveObject, tag, object, history);
        }
        if (tag->GetPerformer() == GetUserId()) {
            selected = true;
            if (!CheckObjectTagAction(TTagAction::ETagAction::ObserveObjectsOnPerform, tag, object, history)) {
                strict = true;
                break;
            }
            if (CheckObjectTagAction(TTagAction::ETagAction::HideSelfPerformObject, tag, object, history)) {
                skip |= EUnvisibilityInfo::Unavailable;
                skip |= EUnvisibilityInfo::UnavailableByHideSelfPerformObject;
                if (unvisibilityInfo) {
                    *unvisibilityInfo = skip;
                }
                return EVisibility::NoVisible;
            }
        } else if (!!tag->GetPerformer() && !CheckObjectTagAction(TTagAction::ETagAction::ObserveOnPerformByAnother, tag, object, history)) {
            skip |= EUnvisibilityInfo::Busy;
        }
        if (CheckObjectTagAction(TTagAction::ETagAction::HideObject, tag, object, history) && !CheckObjectTagAction(TTagAction::ETagAction::ForceObserveObject, tag, object, history)) {
            skip |= EUnvisibilityInfo::Unavailable;
            skip |= EUnvisibilityInfo::UnavailableByHideObject;
        }
    }
    if (!visibleMaxPriority) {
        skip |= EUnvisibilityInfo::Unavailable;
        skip |= EUnvisibilityInfo::UnavailableByMaxPriority;
    }
    if (unvisibilityInfo) {
        *unvisibilityInfo = skip;
    }

    if (selected) {
        return strict ? EVisibility::SelectedStrict : EVisibility::Selected;
    }
    if (visibleOnBusy && skip == EUnvisibilityInfo::Busy) {
        return EVisibility::Visible;
    }

    return skip ? EVisibility::NoVisible : EVisibility::Visible;
}

TUserPermissions::EPermormability TUserPermissions::GetPerformable(const TTaggedObject& object, const TSet<TString>& tagIds, const NEntityTagsManager::EMultiplePerformersPolicy manyPerformersPolicy) const {
    TMap<i32, TVector<TDBTag>> tagsByPriority;

    if (!CheckObjectSelector(object, TObjectAccessAction::EAction::Perform)) {
        return EPermormability::NoPerformAbilities;
    }

    const TTagsFilter* f = GetFilterByObjectAction(TObjectAccessAction::EAction::Perform);
    const bool matchFlag = f && f->IsMatching(object);
    if ((f && !matchFlag) || (!f && HasFiltersByObjectActions(TObjectAccessAction::EAction::Perform))) {
        return EPermormability::NoPerformAbilities;
    }
    if (matchFlag && !CheckObjectHashWithFilter(object.GetId(), TObjectAccessAction::EAction::Perform)) {
        return EPermormability::NoPerformAbilities;
    }

    auto performGroups = object.GetPerformGroups();
    auto performers = object.GetPerformers();

    TUnperformableInfoSet unperformableInfo = 0;
    bool busyIsAvailable = true;
    for (auto&& tag : object.GetTags()) {
        tagsByPriority[tag->GetTagPriority(-1)].emplace_back(tag);
        if (tagIds.contains(tag.GetTagId())) {
            const TTagAction::TTagActions tagActions = GetActionsByTagIdx(tag->GetDescriptionIndex());
            busyIsAvailable = busyIsAvailable && (
                performers.contains(GetUserId()) ||
                (tag->GetMultiPerformingAbility().GetOrElse(manyPerformersPolicy) == NEntityTagsManager::EMultiplePerformersPolicy::Allow) ||
                (performGroups[tag->GetPerformGroup()].empty() && (tagActions & (ui64)TTagAction::ETagAction::ObserveBusyObject))
            );
            const TTagAction::TTagActions mask = tag->IsBusyFor(GetUserId()) ? (ui64)TTagAction::ETagAction::ForcePerform : ((ui64)TTagAction::ETagAction::Perform | (ui64)TTagAction::ETagAction::ForcePerform);
            if ((tagActions & mask) == 0) {
                return EPermormability::NoPerformAbilities;
            }
        } else if (tag->IsBusyFor(GetUserId())) {
            unperformableInfo |= (TUnperformableInfoSet)EUnperformableInfo::Busy;
        }
    }

    if (manyPerformersPolicy == NEntityTagsManager::EMultiplePerformersPolicy::Deny && !busyIsAvailable && (unperformableInfo & (TUnperformableInfoSet)EUnperformableInfo::Busy)) {
        return EPermormability::NoPerformAbilities;
    }

    if (tagsByPriority.size()) {
        for (auto&& maxPriorityTag : tagsByPriority.rbegin()->second) {
            const TTagAction::TTagActions tagActions = GetActionsByTagIdx(maxPriorityTag->GetDescriptionIndex());
            if (tagActions & ((ui64)TTagAction::ETagAction::Perform | (ui64)TTagAction::ETagAction::ForcePerform)) {
                return EPermormability::Performable;
            }
        }
    }
    return EPermormability::NoPerformableByPriorities;
}

TUserPermissions::TUserPermissions(
    const TDriveUserData& userData,
    const TUserPermissionsFeatures& userFeatures,
    const TVector<TDBAction>& actionsActual,
    const TVector<TDBAction>& enableActions,
    const TVector<TUserRoleInfo>& userRoles,
    const TMap<TTagAction::ETagAction, TSet<TTagDescription::TConstPtr>>& tagsByActions,
    const TMap<TString, TMap<TString, TTagEvolutionAction>>& evolutions,
    const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts
)
    : TDriveUserData(userData)
    , IUserOption::TContext(GetUserId())
    , Timestamp(Now())
    , ActionsActual(actionsActual)
    , EnableActions(enableActions)
    , UserFeatures(userFeatures)
    , TagsByActions(tagsByActions)
    , Evolutions(evolutions)
    , RolesByUser(userRoles)
    , DeviceReportTraits(NDeviceReport::NoTraits)
    , DeviceSearchTraits(NDeviceReport::NoTraits)
    , UserReportTraits(NUserReport::NoTraits)
    , UserSearchTraits(NUserReport::NoTraits)
    , CarRegistryReportTraits(0)
    , MessageVisibilityTraits(0)
{
    auto evlog = NDrive::GetThreadEventLogger();
    if (evlog) {
        evlog->AddEvent(NJson::TMapBuilder
            ("event", "CreateUserPermissions")
            ("user_id", userData.GetUserId())
            ("actual_actions_count", actionsActual.size())
            ("enable_actions_count", enableActions.size())
            ("user_roles_count", userRoles.size())
        );
    }

    for (const auto& account : accounts) {
        if (account && account->IsActual(Now())) {
            if (auto parent = account->GetParent()) {
                ActualAccounts.emplace(parent->GetId(), parent);
            }
            ActualAccounts.emplace(account->GetId(), account);
        }
    }

    ActionsByTags.resize(500, 0);
    for (auto&& i : TagsByActions) {
        TSet<TString> tagNames;
        for (auto&& tag : i.second) {
            if (ActionsByTags.size() <= tag->GetIndex()) {
                ActionsByTags.resize(Max<ui32>(ActionsByTags.size() * 2, tag->GetIndex() + 100), 0);
            }
            ActionsByTags[tag->GetIndex()] |= (ui32)i.first;
            tagNames.emplace(tag->GetName());
        }
        TagsNamesByActions.emplace(i.first, std::move(tagNames));
    }

    if (evlog) {
        evlog->AddEvent("ProcessEnableActions");
    };
    for (auto&& action : EnableActions) {
        if (action.GetAs<IPromoCodeUsage>()) {
            const TSet<TString> types = action.GetAs<IPromoCodeUsage>()->GetAvailableTypes();
            AvailablePromoCodeTypes.insert(types.begin(), types.end());
        }
    }

    if (evlog) {
        evlog->AddEvent("ProcessActualActions");
    };
    for (auto&& action : ActionsActual) {
        if (!action || !action->GetEnabled()) {
            continue;
        }
        if (action.GetAs<TOverrideSettingsAction>()) {
            for (auto&& a : action.GetAs<TOverrideSettingsAction>()->GetSettings()) {
                OverrideSettings.emplace(a.first, a.second);
            }
        }
        if (action.GetAs<TObjectAccessAction>()) {
            for (auto&& a : action.GetAs<TObjectAccessAction>()->GetActions()) {
                HasFiltersByObjectActionsFlags.emplace(a);
            }
        }
        if (const TTagAction* tagAction = action.GetAs<TTagAction>()) {
            if (tagAction->HasCondition()) {
                for (const auto& action : tagAction->GetTagActions()) {
                    ConditionalTagActions[action].emplace_back(tagAction);
                }
            }
        }

        if (action.GetAs<TObjectSelectorUsageAction>()) {
            const TObjectSelectorUsageAction* tagAction = action.GetAs<TObjectSelectorUsageAction>();
            for (auto&& action : tagAction->GetActions()) {
                ObjectSelectorIntervals[action].emplace_back(tagAction);
            }
        } else if (action.GetAs<TObjectAccessAction>()) {
            for (auto&& a : action.GetAs<TObjectAccessAction>()->GetActions()) {
                FiltersByObjectActions[a].Merge(action.GetAs<TObjectAccessAction>()->GetFilter());
                ObjectSelectorIntervalsWithFilters[a].emplace_back(action.GetAs<TObjectAccessAction>());
            }
        }
        if (action.GetAs<ICommonOfferBuilderAction>()) {
            OfferBuilders.emplace_back(action.Impl());
        }
        if (action.GetAs<IOfferCorrectorAction>()) {
            OfferCorrections.emplace_back(action.Impl());
        }
        if (auto infoAccessAction = action.GetAs<TInfoAccessAction>()) {
            CarRegistryReportTraits |= infoAccessAction->GetCarRegistryTraits();
            DeviceReportTraits |= infoAccessAction->GetDeviceTraits();
            DeviceSearchTraits |= infoAccessAction->GetDeviceSearchTraits();
            UserReportTraits |= infoAccessAction->GetUserTraits();
            UserSearchTraits |= infoAccessAction->GetUserSearchTraits();
            MessageVisibilityTraits |= infoAccessAction->GetMessageVisibilityTraits();
            const TSet<TString>& availableDocuments = infoAccessAction->GetAvailableDocuments();
            AvailableDocuments.insert(availableDocuments.begin(), availableDocuments.end());
        }
        if (action.GetAs<TAdministrativeAction>()) {
            AdministrativeActions.push_back(*action.GetAs<TAdministrativeAction>());
        }
        if (action.GetAs<TLockAction>()) {
            LockActions.push_back(*action.GetAs<TLockAction>());
        }
        if (action.GetAs<TFilterAction>()) {
            FilterActions.emplace(*action.GetAs<TFilterAction>());
        }
        if (action.GetAs<TUserOptionAction>()) {
            UserOptionActions.emplace_back(action.GetAs<TUserOptionAction>());
        }
        if (auto flags = action.GetAs<TFlagsAction>()) {
            for (auto&&[key, value] : flags->GetFlags()) {
                Flags[key] = value;
            }
        }
    }

    if (evlog) {
        evlog->AddEvent("ProcessUserOptionActions");
    };
    for (auto&& i : UserOptionActions) {
        UserOptions.emplace_back(i->Build(*this));
    }

    if (evlog) {
        evlog->AddEvent("CreatedUserPermissions");
    };
}

TMaybe<double> TUserPermissions::GetAdministrativeLimit(const TString& instance) const {
    TMaybe<double> result;
    for (auto&& i : AdministrativeActions) {
        auto limit = i.GetLimit(instance);
        if (!limit) {
            continue;
        }
        if (result) {
            result = std::max(*result, *limit);
        } else {
            result = limit;
        }
    }
    return result;
}

TMaybe<TSet<TString>> TUserPermissions::GetAdministrativeInstances(const TAdministrativeAction::EAction action, const TAdministrativeAction::EEntity entity, const TString& handlerName) const {
    TSet<TString> result;
    for (auto&& i : AdministrativeActions) {
        auto instances = i.GetInstances(action, entity, handlerName);
        if (!instances.Defined()) {
            return Nothing();
        }
        result.insert(instances->begin(), instances->end());
    }
    return result;
}

bool TUserPermissions::CheckAdministrativeActions(const TAdministrativeAction::EAction action, const TAdministrativeAction::EEntity entity, const TString& handlerName /*= {}*/, const TString& instance /*= {}*/, const TMaybe<TSet<TString>>& instanceTags) const {
    for (auto&& i : AdministrativeActions) {
        if (i.CheckAction(action, entity, handlerName, instance, instanceTags)) {
            return true;
        }
    }
    return false;
}

TTagAction::TTagActions TUserPermissions::GetActionsByTagIdx(const ui32 idx) const {
    if (idx < ActionsByTags.size()) {
        return ActionsByTags[idx];
    } else {
        return 0;
    }
}

TSet<TString> TUserPermissions::GetTagNamesByAction(const TTagAction::ETagAction action, NEntityTagsManager::EEntityType entityType) const {
    TSet<TString> result;
    auto tags = GetTagsByAction(action);
    for (auto&& i : tags) {
        if (entityType != NEntityTagsManager::EEntityType::Undefined) {
            if (!i->HasEntityTypes()) {
                continue;
            }
            if (!i->GetEntityTypesRef().contains(entityType)) {
                continue;
            }
        }
        result.emplace(i->GetName());
    }
    return result;
}

const TSet<TString>* TUserPermissions::GetTagNamesByActionPtr(const TTagAction::ETagAction action) const {
    auto it = TagsNamesByActions.find(action);
    if (it == TagsNamesByActions.end()) {
        return nullptr;
    } else {
        return &it->second;
    }
}

const TSet<TTagDescription::TConstPtr>& TUserPermissions::GetTagsByAction(const TTagAction::ETagAction action) const {
    auto it = TagsByActions.find(action);
    if (it == TagsByActions.end()) {
        return Default<TSet<TTagDescription::TConstPtr>>();
    }
    return it->second;
}

const TTagEvolutionAction* TUserPermissions::GetEvolutionPtr(const TString& from, const TString& to) const {
    auto it = Evolutions.find(from);
    if (it == Evolutions.end()) {
        return nullptr;
    } else {
        auto itTo = it->second.find(to);
        if (itTo == it->second.end()) {
            return nullptr;
        }
        return &itTo->second;
    }
}

const TMap<TString, TMap<TString, TTagEvolutionAction>>* TUserPermissions::GetEvolutions() const {
    return &Evolutions;
}

const TMap<TString, TTagEvolutionAction>* TUserPermissions::GetEvolutions(const TString& from) const {
    TVector<TTagEvolutionAction> result;
    auto it = Evolutions.find(from);
    if (it == Evolutions.end()) {
        return nullptr;
    } else {
        return &it->second;
    }
}

NJson::TJsonValue TUserPermissions::GetFlagsReport() const {
    NJson::TJsonValue result = NJson::JSON_NULL;
    for (auto&&[key, value] : Flags) {
        result[key] = value;
    }
    return result;
}

bool TUserPermissions::HasAction(const TString& actionName) const {
    for (auto&& i : ActionsActual) {
        if (i->GetName() == actionName) {
            return true;
        }
    }
    return false;
}

ui32 TUserPermissions::LockedResourcesLimit() const {
    ui32 maxLockedResourcesCount = 1;
    for (auto&& i : LockActions) {
        if (!i.GetLockedTags()) {
            maxLockedResourcesCount = Max<ui32>(maxLockedResourcesCount, i.GetLockedResourcesLimit());
        }
    }
    return maxLockedResourcesCount;
}

bool TUserPermissions::LockedTagsLimitExceeded(const TMap<TString, TVector<TDBTag>>& performedObjectTags, TString& lockingGroup) const {
    TVector<TString> tagNames;
    for (auto&& [objectId, tags] : performedObjectTags) {
        for (auto&& tag : tags) {
            tagNames.push_back(tag->GetName());
        }
    }

    for (auto&& i : LockActions) {
        if (!i.GetLockedTags()) {
            continue;
        }
        ui32 matchedTags = 0;
        for (auto&& tagName : tagNames) {
            if (i.Match(tagName)) {
                matchedTags += 1;
                if (matchedTags > i.GetLockedResourcesLimit()) {
                    lockingGroup = i.GetLockedTags();
                    return true;
                }
            }
        }
    }

    return false;
}

ui32 TUserPermissions::GetRemainingPerformLoad(const TString& tagId, const TVector<TString>& performedTagNames) const {
    ui32 result = Max<ui32>() / 3;

    ui32 maxLockedResourcesCount = LockedResourcesLimit();

    for (auto&& i : LockActions) {
        if (!i.GetLockedTags() || !i.Match(tagId)) {
            continue;
        }
        ui32 matchedTags = 0;
        for (auto&& tagName : performedTagNames) {
            if (i.Match(tagName)) {
                matchedTags += 1;
                if (matchedTags >= i.GetLockedResourcesLimit() || matchedTags >= maxLockedResourcesCount) {
                    return 0;
                }
            }
        }
        result = Min(result, Min(i.GetLockedResourcesLimit(), maxLockedResourcesCount) - matchedTags);
        if (!result) {
            break;
        }
    }

    return result;
}

bool TUserPermissions::IsRegistered() const {
    return !GetUserFeatures().GetIsFallbackUser() && GetPublicStatus() == NDrive::UserStatusActive;
}

bool TUserPermissions::GetValueStr(TStringBuf key, TString& result, TInstant actuality) const {
    auto it = OverrideSettings.find(key);
    if (it != OverrideSettings.end()) {
        result = it->second;
        return true;
    }
    if (NDrive::HasServer()) {
        const auto& settings = NDrive::GetServer().GetSettings();
        return settings.GetValueStr(key, result, actuality);
    }
    return false;
}
