#pragma once

#include "action.h"
#include "config.h"

#include <drive/backend/actions/administrative.h>
#include <drive/backend/actions/evolution.h>
#include <drive/backend/actions/filter.h>
#include <drive/backend/actions/lock.h>
#include <drive/backend/actions/tag.h>
#include <drive/backend/actions/user_option.h>
#include <drive/backend/users/user.h>
#include <drive/backend/billing/interfaces/account.h>

class ICommonOffer;
class IOfferReport;

namespace NAttachmentReport {
    using TStructureReportTraits = ui32;
}

namespace NDeviceReport {
    using TReportTraits = ui64;
}

class IDynamicActionChecker {
public:
    using TFactory = NObjectFactory::TObjectFactory<IDynamicActionChecker, TString>;
    using TPtr = TAtomicSharedPtr<IDynamicActionChecker>;

    enum class ECheckResult {
        EOk,
        EUserFails,
        EServiceFails
    };

    class TContext {
    public:
        virtual ~TContext() = default;

        virtual const TUserActions& GetOfferBuilders() const = 0;
    };

public:
    virtual ~IDynamicActionChecker() = default;

    virtual ECheckResult CheckAction(const TString& userId, const TContext& context, const TUserAction& action, const NDrive::IServer& server) const = 0;
};

class TUserPermissions
    : public TDriveUserData
    , public IDynamicActionChecker::TContext
    , public IUserOption::TContext
    , protected NDrive::ISettingGetter
{
public:
    using TPtr = TIntrusiveConstPtr<TUserPermissions>;
    using TConstPtr = TPtr;
    using TMutablePtr = TIntrusivePtr<TUserPermissions>;
    using TConditionalActions = TMap<TTagAction::ETagAction, TVector<const TTagAction*>>;
    using TUserAccounts = TMap<ui64, NDrive::NBilling::IBillingAccount::TPtr>;

public:
    R_READONLY(TInstant, Timestamp);
    R_READONLY(TVector<TDBAction>, ActionsActual);
    R_READONLY(TVector<TDBAction>, EnableActions);
    R_READONLY(TSet<TString>, AvailableDocuments);
    R_READONLY(TConditionalActions, ConditionalTagActions);
    R_READONLY(TVector<IUserOption::TPtr>, UserOptions);
    R_READONLY(TUserAccounts, ActualAccounts);
    R_READONLY(TUserPermissionsFeatures, UserFeatures);

private:
    TMap<TObjectAccessAction::EAction, TVector<const TObjectAccessAction*>> ObjectSelectorIntervals;
    TMap<TObjectAccessAction::EAction, TVector<const TObjectAccessAction*>> ObjectSelectorIntervalsWithFilters;
    TMap<TObjectAccessAction::EAction, TTagsFilter> FiltersByObjectActions;
    TSet<TObjectAccessAction::EAction> HasFiltersByObjectActionsFlags;
    TMap<TTagAction::ETagAction, TSet<TTagDescription::TConstPtr>> TagsByActions;
    TMap<TTagAction::ETagAction, TSet<TString>> TagsNamesByActions;
    TVector<TTagAction::TTagActions> ActionsByTags;
    TMap<TString, TMap<TString, TTagEvolutionAction>> Evolutions;

    TVector<TUserRoleInfo> RolesByUser;
    TMap<TString, TString> OverrideSettings;
    TSet<TString> AvailablePromoCodeTypes;
    TVector<TAdministrativeAction> AdministrativeActions;
    TVector<TLockAction> LockActions;
    TSet<TFilterAction> FilterActions;
    TVector<const TUserOptionAction*> UserOptionActions;
    TMap<TString, NJson::TJsonValue> Flags;
    TUserActions OfferCorrections;
    TUserActions OfferBuilders;
    NDeviceReport::TReportTraits DeviceReportTraits;
    NDeviceReport::TReportTraits DeviceSearchTraits;
    NUserReport::TReportTraits UserReportTraits;
    NUserReport::TReportTraits UserSearchTraits;
    NAttachmentReport::TStructureReportTraits CarRegistryReportTraits;
    ui32 MessageVisibilityTraits;

private:
    bool CheckObjectSelector(const TTaggedObject& obj, const TObjectAccessAction::EAction action) const;
    bool CheckObjectHashWithFilter(const TString& id, const TObjectAccessAction::EAction action) const;

protected:
    TUserPermissions(
        const TDriveUserData& userData,
        const TUserPermissionsFeatures& userFeatures,
        const TVector<TDBAction>& actionsActual,
        const TVector<TDBAction>& enableActions,
        const TVector<TUserRoleInfo>& userRoles,
        const TMap<TTagAction::ETagAction, TSet<TTagDescription::TConstPtr>>& tagsByActions,
        const TMap<TString, TMap<TString, TTagEvolutionAction>>& evolutions,
        const TVector<NDrive::NBilling::IBillingAccount::TPtr>& accounts
    );

public:
    template <class... TArgs>
    static TMutablePtr Create(TArgs... args) {
        return new TUserPermissions(std::forward<TArgs>(args)...);
    }

public:
    TUserPermissions(const TUserPermissions& other) = delete;
    TUserPermissions& operator=(const TUserPermissions& other) = delete;

    using NDrive::ISettingGetter::Ref;
    using NDrive::ISettingGetter::UnRef;

    NDrive::TSettingGetterConstPtr SettingGetter() const {
        return const_cast<NDrive::ISettingGetter*>(static_cast<const NDrive::ISettingGetter*>(this));
    }
    TConstPtr Self() const {
        return const_cast<TUserPermissions*>(this);
    }

    const TFilterAction* GetFilterAction(const TString& actionId) const {
        for (auto&& i : FilterActions) {
            if (i.GetName() == actionId) {
                return &i;
            }
        }
        return nullptr;
    }

    template <class T>
    const T* GetOption() const {
        for (auto&& i : UserOptions) {
            const T* opt = dynamic_cast<const T*>(i.Get());
            if (opt) {
                return opt;
            }
        }
        return nullptr;
    }

    template <class T>
    const IUserOption& GetOptionSafe() const {
        for (auto&& i : UserOptions) {
            const T* opt = dynamic_cast<const T*>(i.Get());
            if (opt) {
                return *opt;
            }
        }
        return Default<TFakeUserOption>();
    }

    const IUserOption* GetOptionSafe(const TString& id) const {
        for (auto&& i : UserOptions) {
            if (i->GetOptionId() == id) {
                return i.Get();
            }
        }
        return nullptr;
    }

    template <class T>
    TVector<const IUserOption*> GetOptionsSafe() const {
        TVector<const IUserOption*> result;
        for (auto&& i : UserOptions) {
            const T* opt = dynamic_cast<const T*>(i.Get());
            if (opt) {
                result.emplace_back(opt);
            }
        }
        return result;
    }

    TUserAction::TConstPtr GetOfferBuilder(const TString& name) const {
        for (auto&& i : OfferBuilders) {
            if (i->GetName() == name) {
                return i;
            }
        }
        return nullptr;
    }

    const TSet<TString>& GetAvailablePromoCodeTypes() const {
        return AvailablePromoCodeTypes;
    }

    const NDrive::ISettingGetter& GetSettings() const {
        return *this;
    }

    template <class T>
    TMaybe<T> GetSetting(const ISettings& settings, const TString& key) const {
        Y_UNUSED(settings);
        return GetSettings().GetValue<T>(key);
    }
    template <class T>
    TMaybe<T> GetSetting(const TString& key) const {
        return GetSettings().GetValue<T>(key);
    }

    template <class T, class U>
    T GetSetting(const ISettings& settings, const TString& key, U&& defaultValue) const {
        return GetSetting<T>(settings, key).GetOrElse(std::forward<U>(defaultValue));
    }
    template <class T, class U>
    T GetSetting(const TString& key, U&& defaultValue) const {
        const auto& settings = NDrive::GetServer().GetSettings();
        return GetSetting<T>(settings, key, std::forward<U>(defaultValue));
    }

    template <class T>
    static TMaybe<T> GetSetting(const TString& key, const ISettings& settings, TConstPtr self) {
        if (self) {
            return self->GetSetting<T>(settings, key);
        } else {
            return settings.GetValue<T>(key);
        }
    }
    template <class T>
    static TMaybe<T> GetSetting(const TString& key, TConstPtr self) {
        const auto& settings = NDrive::GetServer().GetSettings();
        return GetSetting<T>(key, settings, self);
    }

    bool HasFiltersByObjectActions(const TObjectAccessAction::EAction action) const {
        return HasFiltersByObjectActionsFlags.contains(action);
    }

    IDynamicActionChecker::ECheckResult CheckActionAcceptable(const TUserAction& action, const NDrive::IServer& server) const;

    NDeviceReport::TReportTraits GetDeviceReportTraits() const {
        return DeviceReportTraits;
    }

    NDeviceReport::TReportTraits GetDeviceSearchTraits() const {
        return DeviceSearchTraits;
    }

    NUserReport::TReportTraits GetUserReportTraits() const {
        return UserReportTraits;
    }

    NUserReport::TReportTraits GetUserSearchTraits() const {
        return UserSearchTraits;
    }

    ui32 GetMessageVisibilityTraits() const {
        return MessageVisibilityTraits;
    }

    NAttachmentReport::TStructureReportTraits GetCarRegistryReportTraits() const {
        return CarRegistryReportTraits;
    }

    const TTagsFilter* GetFilterByObjectAction(const TObjectAccessAction::EAction action) const;
    TVector<TDBTag> FilterObserveTags(const TVector<TDBTag>& tags) const;

    enum class EVisibility {
        Visible = 0,
        SelectedStrict = 1,
        Selected = 2,
        NoVisible = 3
    };

    using TUnvisibilityInfoSet = ui64;
    enum EUnvisibilityInfo: TUnvisibilityInfoSet {
        Busy                                    = 1 << 0,
        Unavailable                             = 1 << 1,
        UnavailableByDefault                    = 1 << 2,
        UnavailableByObjectSelector             = 1 << 3,
        UnavailableByObjectSelectorWithFilter   = 1 << 4,
        UnavailableByHideSelfPerformObject      = 1 << 5,
        UnavailableByHideObject                 = 1 << 6,
        UnavailableByMaxPriority                = 1 << 7,
    };
    static TString ExplainInvisibility(TUnvisibilityInfoSet info);

    bool CheckObjectTagAction(const TTagAction::ETagAction action, const TConstDBTag& tag, const TTaggedObject& object, const TObjectEvents<TConstDBTag>& history) const;
    EVisibility GetVisibility(const TTaggedObject& object, const NEntityTagsManager::EEntityType entityType, TUnvisibilityInfoSet* unvisibilityInfo = nullptr,  bool forceShowObjectsWithoutTags = false) const;

    enum class EUnperformableInfo: ui32 {
        Busy = 1,
        Unavailable = 1 << 1
    };

    using TUnperformableInfoSet = ui32;

    enum class EPermormability {
        Performable = 0,
        NoPerformableByPriorities = 1,
        NoPerformAbilities = 2,
    };

    EPermormability GetPerformable(const TTaggedObject& object, const TSet<TString>& tagIds, const NEntityTagsManager::EMultiplePerformersPolicy manyPerformersPolicy) const;

    const TUserActions& GetOfferCorrections() const {
        return OfferCorrections;
    }

    const TUserActions& GetOfferBuilders() const override {
        return OfferBuilders;
    }

    ui32 LockedResourcesLimit() const;

    bool LockedTagsLimitExceeded(const TMap<TString, TVector<TDBTag>>& performedObjectTags, TString& lockingGroup) const;
    ui32 GetRemainingPerformLoad(const TString& tagId, const TVector<TString>& performedTagNames) const;

    bool IsRegistered() const;

    template <class TContainer>
    TContainer FilterTagNames(const TContainer& container, const TTagAction::ETagAction action) const {
        const TSet<TString>* actionTags = GetTagNamesByActionPtr(action);
        if (actionTags == nullptr) {
            return TContainer();
        }
        const auto predicat = [&actionTags](const TString& value) -> bool {
            return !actionTags->contains(value);
        };
        TContainer result = container;
        result.erase(std::remove_if(result.begin(), result.end(), predicat), result.end());
        return result;
    }

    TSet<TString> GetTagNamesByAction(const TTagAction::ETagAction action, NEntityTagsManager::EEntityType entityType = NEntityTagsManager::EEntityType::Undefined) const;
    TTagAction::TTagActions GetActionsByTagIdx(const ui32 idx) const;
    const TSet<TString>* GetTagNamesByActionPtr(const TTagAction::ETagAction action) const;
    const TSet<TTagDescription::TConstPtr>& GetTagsByAction(const TTagAction::ETagAction action) const;
    bool HasAction(const TString& actionName) const;

    const TTagEvolutionAction* GetEvolutionPtr(const TString& from, const TString& to) const;
    const TMap<TString, TTagEvolutionAction>* GetEvolutions(const TString& from) const;
    const TMap<TString, TMap<TString, TTagEvolutionAction>>* GetEvolutions() const;

    NJson::TJsonValue GetFlagsReport() const;

    TMaybe<double> GetAdministrativeLimit(const TString& instance) const;
    TMaybe<TSet<TString>> GetAdministrativeInstances(const TAdministrativeAction::EAction action, const TAdministrativeAction::EEntity entity, const TString& handlerName = {}) const;
    bool CheckAdministrativeActions(const TAdministrativeAction::EAction action, const TAdministrativeAction::EEntity entity, const TString& handlerName = {}, const TString& instance = {}, const TMaybe<TSet<TString>>& instanceTags = {}) const;

    const TVector<TUserRoleInfo>& GetRolesByUser() const {
        return RolesByUser;
    }

    const TSet<TFilterAction>& GetFilterActions() const {
        return FilterActions;
    }

    TMaybe<TString> GetMobilePaymentMethod(const ISettings& settings) const {
        return GetSetting<bool>(settings, "show_mobile_payment_method").GetOrElse(false) ? GetUserFeatures().OptionalMobilePay() : TMaybe<TString>();
    }

    bool UseYandexPaymentMethod(const ISettings& settings) const {
        return GetSetting<bool>(settings, "show_yandex_payment_method").GetOrElse(true) && GetUserFeatures().GetIsPlusUser();
    }

protected:
    bool GetValueStr(TStringBuf key, TString& result, TInstant actuality = TInstant::Zero()) const override;
};
