#include "roles.h"
#include "roles_impl.h"

#include <drive/backend/actions/evolution.h>
#include <drive/backend/actions/tag.h>
#include <drive/backend/logging/evlog.h>
#include <drive/backend/tags/tags_manager.h>

#include <library/cpp/json/json_reader.h>

#include <util/system/yield.h>

NJson::TJsonValue TLinkedRoleActionHeader::BuildJsonReport() const {
    NJson::TJsonValue result = SerializeToTableRecord().SerializeToJson();
    result["role_action_meta"] = SerializeMetaToJson();
    return result;
}

bool TLinkedRoleActionHeader::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return TBaseDecoder::DeserializeFromJson(*this, jsonInfo);
}

bool TLinkedRoleActionHeader::DeserializeFromTableRow(const NStorage::TTableRecord& row) {
    return TDecoder::DeserializeFromTableRecord(*this, row);
}

NStorage::TTableRecord TLinkedRoleActionHeader::SerializeToTableRecord() const {
    NStorage::TTableRecord row;
    row.Set("action_id", GetSlaveObjectId());
    row.Set("role_id", GetRoleId());
    row.Set("role_action_meta", SerializeMetaToJson().GetStringRobust());
    return row;
}

NJson::TJsonValue TLinkedRoleRoleHeader::BuildJsonReport() const {
    NJson::TJsonValue result = SerializeToTableRecord().SerializeToJson();
    result["link_meta"] = SerializeMetaToJson();
    return result;
}

bool TLinkedRoleRoleHeader::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return TBaseDecoder::DeserializeFromJson(*this, jsonInfo);
}

bool TLinkedRoleRoleHeader::DeserializeFromTableRow(const NStorage::TTableRecord& row) {
    return TDecoder::DeserializeFromTableRecord(*this, row);
}

NStorage::TTableRecord TLinkedRoleRoleHeader::SerializeToTableRecord() const {
    NStorage::TTableRecord row;
    row.Set("slave_role_id", GetSlaveObjectId());
    row.Set("role_id", GetRoleId());
    row.Set("link_meta", SerializeMetaToJson().GetStringRobust());
    return row;
}

TActionsDB::TActionsDB(const ITagsHistoryContext& context, const THistoryConfig& config, const TPropositionsManagerConfig& propositionsConfig)
    : TBase("drive_tag_actions", context, config)
    , Propositions(context, propositionsConfig)
    , TagsContext(context)
    , TagActionMatchCacheRebuildThreadPool(IThreadPool::TParams()
        .SetThreadName("tag_action_match_cache_rebuild")
    )
{
    TagActionMatchCacheRebuildThreadPool.Start(1);
}

TActionsDB::TActionTags TActionsDB::GetActionTags(const TString& name, TInstant since) const {
    auto tagActionMatchCache = GetTagActionMatchCache(since);
    return GetActionTags(name, tagActionMatchCache.Get());
}

TActionsDB::TActionTags TActionsDB::GetActionTags(const TString& name, const TTagActionMatchCache* tagActionMatchCache) const {
    if (!tagActionMatchCache) {
        return {};
    }
    auto it = tagActionMatchCache->ActionTags.find(name);
    if (it == tagActionMatchCache->ActionTags.end()) {
        return {};
    } else {
        return it->second;
    }
}

TActionsDB::TEvolutionTags TActionsDB::GetEvolutionTags(const TString& name, TInstant since) const {
    auto tagActionMatchCache = GetTagActionMatchCache(since);
    return GetEvolutionTags(name, tagActionMatchCache.Get());
}

TActionsDB::TEvolutionTags TActionsDB::GetEvolutionTags(const TString& name, const TTagActionMatchCache* tagActionMatchCache) const {
    if (!tagActionMatchCache) {
        return {};
    }
    auto it = tagActionMatchCache->EvolutionTags.find(name);
    if (it == tagActionMatchCache->EvolutionTags.end()) {
        return {};
    } else {
        return it->second;
    }
}

bool TActionsDB::ApplyPatchJson(NJson::TJsonValue& result, const TJsonDiff& patch) {
    const NJson::TJsonValue::TMapType* mapInfo = nullptr;
    if (!result.GetMapPointer(&mapInfo)) {
        Y_ASSERT(false);
        return false;
    }
    for (auto&& i : patch) {
        if (i.second.GetOldValue() == "__empty__" && !result.Has(i.first)) {
            if (i.second.GetNewValue().GetString() != "__delete__") {
                result.InsertValue(i.first, i.second.GetNewValue());
            }
        } else if (result.Has(i.first) && i.second.GetOldValue() == result[i.first]) {
            if (i.second.GetNewValue().GetString() == "__delete__") {
                result.EraseValue(i.first);
            } else {
                result[i.first] = i.second.GetNewValue();
            }
        }
    }
    return true;
}

void BuildTagActionMatchCache(const TDBAction& action, const ITagsMeta::TTagDescriptionsByName& registeredTags, TActionsDB::TTagActionMatchCache& cache) {
    cache.ActionTags[action->GetName()].clear();
    cache.EvolutionTags[action->GetName()].clear();
    if (const TTagAction* tagAction = action.GetAs<TTagAction>()) {
        if (tagAction->HasCondition()) {
            return;
        }
        for (auto&& tDescription : registeredTags) {
            if (tagAction->Match(tDescription.second->GetName()) && (tagAction->GetTagAttributesFilter().IsEmpty() || tagAction->GetTagAttributesFilter().IsMatching(tDescription.second->GetGrouppingTags()))) {
                cache.ActionTags[tagAction->GetName()].emplace(tDescription.second);
            }
        }
    }

    if (const TTagEvolutionAction* tagAction = action.GetAs<TTagEvolutionAction>()) {
        for (auto&& tDescriptionFrom : registeredTags) {
            if (!tagAction->MatchFrom(*tDescriptionFrom.second)) {
                continue;
            }
            for (auto&& tDescriptionTo : registeredTags) {
                if (tagAction->MatchTo(*tDescriptionTo.second)) {
                    cache.EvolutionTags[tagAction->GetName()].emplace(std::make_pair(tDescriptionFrom.second->GetName(), tDescriptionTo.second->GetName()));
                }
            }
        }
    }
}

template <class T>
TActionsDB::TTagActionMatchCachePtr BuildTagActionMatchCache(const T& actions, const ITagsMeta::TTagDescriptionsByName& registeredTags) {
    auto result = MakeIntrusive<TActionsDB::TTagActionMatchCache>();
    for (auto&& action : actions) {
        BuildTagActionMatchCache(action, registeredTags, *result);
    }
    return result;
}

TActionsDB::TTagActionMatchCacheConstPtr TActionsDB::GetTagActionMatchCache(TInstant since) const {
    if (since && TagActionMatchCacheUpdateTimestamp < since) {
        auto evlog = NDrive::GetThreadEventLogger();
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "WaitTagActionMatchCacheUpdate")
                ("since", NJson::ToJson(since))
            );
        }
        size_t cycles = 0;
        while (TagActionMatchCacheUpdateTimestamp < since && TagActionMatchCacheRebuildThreadPool.Size()) {
            cycles += 1;
            ThreadYield();
        }
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "WaitTagActionMatchCacheUpdateDone")
                ("cycles", cycles)
                ("timestamp", NJson::ToJson(TagActionMatchCacheUpdateTimestamp))
            );
        }
    }
    return TagActionMatchCache.AtomicLoad();
}

template <class T>
void TActionsDB::RebuildTagActionMatchCache(const T& actions) const {
    auto guard = Guard(TagActionMatchCacheRebuildLock);
    auto timestamp = Now();
    auto registeredTags = TagsContext.GetTagsManager().GetRegisteredTags();
    auto tagActionMatchCache = BuildTagActionMatchCache(actions, registeredTags);
    {
        TagActionMatchCache.AtomicStore(tagActionMatchCache);
        TagActionMatchCacheUpdateTimestamp = timestamp;
    }
    INFO_LOG << "TagActionMatchCache rebuilt: " << registeredTags.size() << Endl;
}

bool TActionsDB::BuildJsonDiff(const NJson::TJsonValue& from, const NJson::TJsonValue& to, TJsonDiff& result) {
    const NJson::TJsonValue::TMapType* mapFrom;
    const NJson::TJsonValue::TMapType* mapTo;
    if (!from.GetMapPointer(&mapFrom) || !to.GetMapPointer(&mapTo)) {
        return false;
    }
    for (auto&& i : *mapFrom) {
        auto itTo = mapTo->find(i.first);
        if (itTo == mapTo->end()) {
            result.emplace(i.first, TJsonPatch(i.second, "__delete__"));
        } else if (itTo->second != i.second) {
            result.emplace(i.first, TJsonPatch(i.second, itTo->second));
        }
    }
    for (auto&& i : *mapTo) {
        if (!mapFrom->contains(i.first)) {
            result.emplace(i.first, TJsonPatch("__empty__", i.second));
        }
    }
    return true;
}

void TActionsDB::AcceptHistoryEventUnsafe(const TObjectEvent<TDBAction>& ev) const {
    if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
        Objects.erase(ev->GetName());
    } else {
        Objects[ev->GetName()] = ev.Clone();
    }
    bool scheduled = TagActionMatchCacheRebuildThreadPool.AddFunc([this, name = ev->GetName()] {
        if (!IsActive()) {
            INFO_LOG << "Skip TagActionMatchCache rebuilding: " << name << Endl;
            return;
        }

        INFO_LOG << "TagActionMatchCache rebuilding in AcceptHistoryEventUnsafe: " << name << Endl;
        auto actions = GetCachedObjectsVector();
        RebuildTagActionMatchCache(actions);
    });
    if (!scheduled) {
        ALERT_LOG << "cannot schedule TagActionMatchCacheRebuild" << Endl;
    }
}

bool TActionsDB::DoRebuildCacheUnsafe() const {
    NStorage::TObjectRecordsSet<TDBAction> records;
    {
        TTransactionPtr transaction = HistoryCacheDatabase->CreateTransaction(true);
        auto tagsTable = HistoryCacheDatabase->GetTable("drive_tag_actions");

        TQueryResultPtr queryResult = tagsTable->GetRows("", records, transaction);

        if (!queryResult || !queryResult->IsSucceed()) {
            return false;
        }
    }
    for (auto&& i : records) {
        if (!!i) {
            Objects.emplace(i->GetName(), i.Clone());
        }
    }
    {
        INFO_LOG << "TagActionMatchCache rebuilding in DoRebuildCacheUnsafe" << Endl;
        RebuildTagActionMatchCache(NContainer::Values(Objects));
    }
    return true;
}

bool TActionsDB::Process(IMessage* message) {
    auto notifyHistoryChanged = message ? message->As<TAfterCacheInvalidateNotification>() : nullptr;
    if (notifyHistoryChanged) {
        if (notifyHistoryChanged->GetOriginatorTableName() == "tags_description_standart_history") {
            bool scheduled = TagActionMatchCacheRebuildThreadPool.AddFunc([this] {
                INFO_LOG << "TagActionMatchCache rebuilding in Process" << Endl;
                auto actions = GetCachedObjectsVector();
                RebuildTagActionMatchCache(actions);
            });
            if (!scheduled) {
                ALERT_LOG << "cannot schedule TagActionMatchCacheRebuild" << Endl;
            }
        }
        return true;
    }
    return TBase::Process(message);
}

bool TActionsDB::ApplyDiffForChildren(const TJsonDiff& actionJsonDiff, const TString& actionName, const TString& userId, const bool force, NDrive::TEntitySession& session, const TSet<TString>& readyActions) const {
    TVector<TDBAction> actions = GetCachedObjectsVector();
    for (auto&& i : actions) {
        TSet<TString> readyActionsNext = readyActions;
        if (!readyActionsNext.emplace(i->GetName()).second) {
            continue;
        }
        if (i->GetActionParent() == actionName) {
            NJson::TJsonValue jsonReport = i->BuildJsonReport();
            if (!ApplyPatchJson(jsonReport["action_meta"], actionJsonDiff)) {
                session.SetErrorInfo("TActionsDB::ApplyDiffForChildren", "incorrect json patch", EDriveSessionResult::InconsistencySystem);
                return false;
            }
            if (jsonReport == i->BuildJsonReport()) {
                continue;
            }
            TUserAction::TPtr newAction = TUserAction::BuildFromJson(jsonReport);
            if (!newAction) {
                session.SetErrorInfo("TActionsDB::ApplyDiffForChildren", "incorrect patch for action " + i->GetName(), EDriveSessionResult::InconsistencySystem);
                return false;
            }
            if (!force) {
                if (!Upsert(newAction, userId, session, readyActionsNext)) {
                    return false;
                }
            } else {
                if (!ForceUpsert(newAction, userId, session, readyActionsNext)) {
                    return false;
                }
            }
        }
    }
    return true;
}

bool TActionsDB::ForceUpsert(TUserAction::TConstPtr action, const TString& userId, NDrive::TEntitySession& session, const TSet<TString>& readyActions) const {
    if (!action) {
        return true;
    }
    NStorage::TTableRecord rowCondition;
    rowCondition.Set("action_id", action->GetName());

    NStorage::TTableRecord rowUpdate = action->SerializeToTableRow();
    rowUpdate.ForceSet("action_revision", "nextval('" + GetTableName() + "_action_revision_seq')");

    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(GetTableName());
    bool isUpdate = false;
    NStorage::TObjectRecordsSet<TDBAction> records;
    if (!table->Upsert(rowUpdate, session.GetTransaction(), rowCondition, &isUpdate, &records)->IsSucceed() || records.size() != 1) {
        session.SetErrorInfo("upsert_action", session.GetMessages(), EDriveSessionResult::TransactionProblem);
        return false;
    }

    if (isUpdate) {
        TMaybe<TDBAction> dbAction = GetObject(action->GetName(), Now());
        if (!dbAction) {
            session.SetErrorInfo("ActionsDB::ForceUpsert", "no_previous_version", EDriveSessionResult::InconsistencySystem);
            return false;
        }
        const TDBAction& previousAction = *dbAction;
        if (!previousAction) {
            session.SetErrorInfo("ActionsDB::ForceUpsert", "invalid_previous_version", EDriveSessionResult::InconsistencySystem);
            return false;
        } else {
            TJsonDiff patch;
            if (!BuildJsonDiff(previousAction->BuildJsonReport()["action_meta"], records.front()->BuildJsonReport()["action_meta"], patch)) {
                session.SetErrorInfo("upsert_action", "incorrect patch construction", EDriveSessionResult::InconsistencySystem);
                return false;
            }
            if (patch.size()) {
                ApplyDiffForChildren(patch, action->GetName(), userId, true, session, readyActions);
            }
        }
    }

    return HistoryManager->AddHistory(records.GetObjects(), userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session);
}

bool TActionsDB::Upsert(TUserAction::TConstPtr action, const TString& userId, NDrive::TEntitySession& session, const TSet<TString>& readyActions, const TAvailableActions abilities) const {
    if (!action) {
        return true;
    }
    NStorage::TTableRecord rowCondition;
    rowCondition.Set("action_id", action->GetName());

    NStorage::TTableRecord rowUpdate = action->SerializeToTableRow();
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(GetTableName());

    NStorage::TObjectRecordsSet<TDBAction> records;
    bool isUpdate = false;
    switch (table->UpsertWithRevision(rowUpdate, session.GetTransaction(), rowCondition, action->OptionalRevision(), "action_revision", &records)) {
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::IncorrectRevision:
            session.SetErrorInfo("upsert_action", "incorrect_revision", EDriveSessionResult::InconsistencyUser);
            return false;
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::Failed:
            session.SetErrorInfo("upsert_action", "UpsertWithRevision failure", EDriveSessionResult::TransactionProblem);
            return false;
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::Updated:
            isUpdate = true;
        case NStorage::ITableAccessor::EUpdateWithRevisionResult::Inserted:
            break;
    }
    if (records.empty()) {
        session.SetErrorInfo("upsert_action", "parsed records set is empty", EDriveSessionResult::InconsistencySystem);
        return false;
    }
    if (isUpdate) {
        TMaybe<TDBAction> dbAction = GetObject(action->GetName(), Now());
        if (!dbAction) {
            session.SetErrorInfo("ActionsDB::Upsert", "no_previous_version", EDriveSessionResult::InconsistencySystem);
            return false;
        }
        const TDBAction& previousAction = *dbAction;
        if (!previousAction) {
            session.SetErrorInfo("ActionsDB::Upsert", "invalid_previous_version", EDriveSessionResult::InconsistencySystem);
            return false;
        } else {
            TJsonDiff patch;
            if (!BuildJsonDiff(previousAction->BuildJsonReport()["action_meta"], action->BuildJsonReport()["action_meta"], patch)) {
                session.SetErrorInfo("upsert_action", "incorrect patch construction", EDriveSessionResult::InconsistencySystem);
                return false;
            }
            if (patch.size()) {
                ApplyDiffForChildren(patch, action->GetName(), userId, false, session, readyActions);
            }
        }
    }
    TDBAction dbAction(records.GetObjects().front());
    if (isUpdate) {
        if ((abilities & (ui32)EAvailableActions::Modify) == 0) {
            session.SetErrorInfo("upsert_action", "incorrect_permissions_for_modify_action", EDriveSessionResult::NoUserPermissions);
            return false;
        }
    } else if ((abilities & (ui32)EAvailableActions::Add) == 0) {
        session.SetErrorInfo("upsert_action", "incorrect_permissions_for_add_action", EDriveSessionResult::NoUserPermissions);
        return false;
    }
    return HistoryManager->AddHistory(dbAction, userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session);
}

bool TActionsDB::RemoveAction(const TString& actionName, const TString& userId, NDrive::TEntitySession& session) const {
    return RemoveActions({actionName}, userId, session);
}

bool TActionsDB::RemoveActions(const TVector<TString>& actions, const TString& userId, NDrive::TEntitySession& session) const {
    if (actions.empty()) {
        return true;
    }
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(GetTableName());
    NStorage::TObjectRecordsSet<TDBAction> records;
    if (!table->RemoveRow("action_id IN (" + session.GetTransaction()->Quote(actions) + ")", session.GetTransaction(), &records)->IsSucceed()) {
        session.SetErrorInfo("ActionsDB::RemoveAction", "RemoveRow failure", EDriveSessionResult::TransactionProblem);
        return false;
    }
    return HistoryManager->AddHistory(records.GetObjects(), userId, EObjectHistoryAction::Remove, session);
}

bool TActionsDB::DeprecateActions(const TVector<TString>& actions, const TString& userId, NDrive::TEntitySession& session) const {
    if (actions.empty()) {
        return true;
    }
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(GetTableName());
    NStorage::TObjectRecordsSet<TDBAction> records;
    if (!table->UpdateRow("action_id IN (" + session.GetTransaction()->Quote(actions) + ")", "deprecated = true", session.GetTransaction(), &records)->IsSucceed()) {
        session.SetErrorInfo("ActionsDB::DeprecateAction", "UpdateRow failure", EDriveSessionResult::TransactionProblem);
        return false;
    }
    return HistoryManager->AddHistory(records.GetObjects(), userId, EObjectHistoryAction::Remove, session);
}

TDriveRoleHeader::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    Name = GetFieldDecodeIndex("role_id", decoderBase);
    Optional = GetFieldDecodeIndex("role_optional", decoderBase);
    IsIDM = GetFieldDecodeIndex("role_is_idm", decoderBase);
    IsPublic = GetFieldDecodeIndex("role_is_public", decoderBase);
    Group = GetFieldDecodeIndex("role_group", decoderBase);
    Description = GetFieldDecodeIndex("role_description", decoderBase);
    GrouppingTags = GetFieldDecodeIndex("role_groupping_tags", decoderBase);
}

bool TDriveRoleHeader::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, Name);
    READ_DECODER_VALUE(decoder, values, Optional);
    READ_DECODER_VALUE(decoder, values, IsIDM);
    READ_DECODER_VALUE(decoder, values, IsPublic);
    READ_DECODER_VALUE(decoder, values, Group);
    READ_DECODER_VALUE(decoder, values, Description);
    TString grouppingTags;
    READ_DECODER_VALUE_TEMP_OPT(decoder, values, grouppingTags, GrouppingTags);
    StringSplitter(grouppingTags).SplitBySet(", ").SkipEmpty().Collect(&GrouppingTags);
    return true;
}

bool TDriveRoleHeader::Parse(const NStorage::TTableRecord& row) {
    Name = row.Get("role_id");
    Description = row.Get("role_description");
    Optional = IsTrue(row.Get("role_optional"));
    IsIDM = IsTrue(row.Get("role_is_idm"));
    IsPublic = IsTrue(row.Get("role_is_public"));
    if (row.Has("role_group") && !row.TryGet("role_group", Group)) {
        return false;
    }
    StringSplitter(row.Get("role_groupping_tags")).SplitBySet(", ").SkipEmpty().Collect(&GrouppingTags);
    return true;
}

void TDriveRoleHeader::Serialize(NStorage::TTableRecord& row) const {
    row = SerializeToTableRecord();
}

NStorage::TTableRecord TDriveRoleHeader::SerializeToTableRecord() const {
    NStorage::TTableRecord row;
    row.Set("role_id", Name);
    row.Set("role_description", Description);
    row.Set("role_optional", Optional);
    row.Set("role_is_idm", IsIDM);
    row.Set("role_is_public", IsPublic);
    row.Set("role_group", Group);
    row.Set("role_groupping_tags", JoinSeq(", ", GrouppingTags));
    return row;
}

NJson::TJsonValue TDriveRoleHeader::BuildJsonReport() const {
    return SerializeToTableRecord().SerializeToJson();
}

bool TRolesDB::DoRebuildCacheUnsafe() const {
    NStorage::TObjectRecordsSet<TDriveRoleHeader> records;
    {
        TTransactionPtr transaction = HistoryCacheDatabase->CreateTransaction(true);
        auto tagsTable = HistoryCacheDatabase->GetTable("drive_roles");

        TQueryResultPtr queryResult = tagsTable->GetRows("", records, transaction);
        if (!queryResult || !queryResult->IsSucceed()) {
            return false;
        }
    }
    for (auto&& i : records) {
        Objects.emplace(i.GetRoleId(), i);
    }
    return true;
}

bool TRolesDB::Upsert(const TDriveRoleHeader& role, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::TTableRecord rowCondition;
    rowCondition.Set("role_id", role.GetName());

    NStorage::TTableRecord rowUpdate;
    role.Serialize(rowUpdate);
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable("drive_roles");
    bool isUpdate = false;
    NStorage::TObjectRecordsSet<TDriveRoleHeader> records;
    if (!table->Upsert(rowUpdate, session.GetTransaction(), rowCondition, &isUpdate, &records)->IsSucceed()) {
        session.SetErrorInfo("upsert_role", session.GetStringReport(), EDriveSessionResult::TransactionProblem);
        return false;
    }
    return HistoryManager->AddHistory(records.GetObjects(), userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session);
}

bool TRolesDB::RemoveRole(const TString& roleName, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::TTableRecord rowCondition;
    rowCondition.Set("role_id", roleName);
    NStorage::TObjectRecordsSet<TDriveRoleHeader> records;
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable("drive_roles");
    if (!table->RemoveRow(rowCondition, session.GetTransaction(), &records)->IsSucceed()) {
        session.SetErrorInfo("remove_role", session.GetStringReport(), EDriveSessionResult::TransactionProblem);
        return false;
    }

    for (auto&& i : records) {
        if (!HistoryManager->AddHistory(i, userId, EObjectHistoryAction::Remove, session)) {
            return false;
        }
    }
    return true;
}

TRoleInfoDB::TRoleInfoDB(const IHistoryContext& context, const THistoryConfig& historyConfig)
    : RoleActions(context, historyConfig)
    , RoleRoles(context, historyConfig)
{
    Y_ENSURE_BT(RoleActions.Start());
    Y_ENSURE_BT(RoleRoles.Start());
}

TRoleInfoDB::~TRoleInfoDB() {
    if (!RoleActions.Stop()) {
        ERROR_LOG << "cannot stop RoleActions manager" << Endl;
    }
    if (!RoleRoles.Stop()) {
        ERROR_LOG << "cannot stop RoleRoles manager" << Endl;
    }
}

bool TRoleInfoDB::UnlinkAllFromRole(const TString& roleId, const TString& userId, NDrive::TEntitySession& session) const {
    {
        TVector<TDriveRoleActions> roleInfo = RoleActions.GetInfo({roleId}, Now());
        for (auto&& i : roleInfo) {
            for (auto&& a : i.GetSlaves()) {
                if (!RoleActions.Unlink(a.GetSlaveObjectId(), a.GetRoleId(), userId, session)) {
                    return false;
                }
            }
        }
    }
    {
        TVector<TDriveRoleRoles> roleInfo = RoleRoles.GetInfo({roleId}, Now());
        for (auto&& i : roleInfo) {
            for (auto&& a : i.GetSlaves()) {
                if (!RoleRoles.Unlink(a.GetSlaveObjectId(), a.GetRoleId(), userId, session)) {
                    return false;
                }
            }
        }
    }
    return true;
}

TSet<TString> TRoleInfoDB::GetActions(const TSet<TString>& roles, TInstant actuality, TInstant ts, const TSet<TString>& disabledRoles) const {
    auto roleActionsSnapshot = RoleActions.GetObjectSnapshot();
    if (!roleActionsSnapshot || roleActionsSnapshot->GetTimestamp() < actuality) {
        auto guard = NDrive::BuildEventGuard("GetRoleActionsSlow");
        TMap<TString, TDriveRoleActions> roleActions;
        Y_ENSURE_BT(RoleActions.GetAllObjectsFromCache(roleActions, actuality));
        roleActionsSnapshot = MakeIntrusive<TRoleActionsDB::TObjectSnapshot>(std::move(roleActions), actuality);
    }
    auto roleRolesSnapshot = RoleRoles.GetObjectSnapshot();
    if (!roleRolesSnapshot || roleRolesSnapshot->GetTimestamp() < actuality) {
        auto guard = NDrive::BuildEventGuard("GetRoleRolesSlow");
        TMap<TString, TDriveRoleRoles> roleRoles;
        Y_ENSURE_BT(RoleRoles.GetAllObjectsFromCache(roleRoles, actuality));
        roleRolesSnapshot = MakeIntrusive<TRoleRolesDB::TObjectSnapshot>(std::move(roleRoles), actuality);
    }
    const auto& roleActions = Yensured(roleActionsSnapshot)->Get();
    const auto& roleRoles = Yensured(roleRolesSnapshot)->Get();
    TSet<TString> currentRoles = MakeSet(roles);
    TSet<TString> actionsReady;
    TSet<TString> rolesReady(disabledRoles);
    while (!currentRoles.empty()) {
        TSet<TString> nextRoles;
        for (auto&& i : currentRoles) {
            if (rolesReady.contains(i)) {
                continue;
            }
            {
                auto itRoleActions = roleActions.find(i);
                if (itRoleActions != roleActions.end()) {
                    for (auto&& action : itRoleActions->second.GetSlaves()) {
                        if (!actionsReady.contains(action.GetSlaveObjectId())) {
                            if (ts == TInstant::Max() || action.IsActual(ts)) {
                                actionsReady.emplace(action.GetSlaveObjectId());
                            }
                        }
                    }
                }
            }
            {
                auto itRoleRoles = roleRoles.find(i);
                if (itRoleRoles != roleRoles.end()) {
                    for (auto&& role : itRoleRoles->second.GetSlaves()) {
                        if (!rolesReady.contains(role.GetSlaveObjectId())) {
                            if (ts == TInstant::Max() || role.IsActual(ts)) {
                                nextRoles.emplace(role.GetSlaveObjectId());
                            }
                        }
                    }
                }
            }
            rolesReady.emplace(i);
        }
        std::swap(currentRoles, nextRoles);
    }
    return actionsReady;
}

bool TRoleSnapshot::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    NJson::TJsonValue dataJson;
    READ_DECODER_VALUE_JSON(decoder, values, dataJson, Data);
    return DeserializeFromJson(dataJson);
}

NJson::TJsonValue TRoleSnapshot::BuildJsonReport(bool isFullReport, bool serializeSlaveObjects) const {
    NJson::TJsonValue result = Role.SerializeToTableRecord().SerializeToJson();
    if (serializeSlaveObjects) {
        NJson::TJsonValue& actionsJson = result.InsertValue("actions", NJson::JSON_ARRAY);
        for (auto&& i : RoleActions.GetSlaves()) {
            auto it = Actions.find(i.GetSlaveObjectId());
            if (it != Actions.end()) {
                NJson::TJsonValue linkJson = i.BuildJsonReport();
                linkJson.InsertValue("action", it->second.BuildJsonReport());
                actionsJson.AppendValue(std::move(linkJson));
            } else if (!isFullReport) {
                actionsJson.AppendValue(i.BuildJsonReport());
            }
        }
        NJson::TJsonValue& rolesJson = result.InsertValue("slave_roles", NJson::JSON_ARRAY);
        for (auto&& i : RoleRoles.GetSlaves()) {
            rolesJson.AppendValue(i.BuildJsonReport());
        }
    } else {
        NJson::TJsonValue& actionIds = result.InsertValue("action_ids", NJson::JSON_ARRAY);
        for (auto&& i : RoleActions.GetSlaves()) {
            actionIds.AppendValue(i.GetSlaveObjectId());
        }
        NJson::TJsonValue& slaveRoleIds = result.InsertValue("slave_role_ids", NJson::JSON_ARRAY);
        for (auto&& i : RoleRoles.GetSlaves()) {
            slaveRoleIds.AppendValue(i.GetSlaveObjectId());
        }
    }
    return result;
}

NStorage::TTableRecord TRoleSnapshot::SerializeToTableRecord() const {
    NStorage::TTableRecord result;
    result.Set("data", SerializeToJson());
    return result;
}

NJson::TJsonValue TRoleSnapshot::SerializeToJson() const {
    NJson::TJsonValue result = Role.SerializeToTableRecord().SerializeToJson();
    NJson::TJsonValue& actionsJson = result.InsertValue("actions", NJson::JSON_ARRAY);
    for (auto&& i : RoleActions.GetSlaves()) {
        actionsJson.AppendValue(i.SerializeToTableRecord().SerializeToJson());
    }
    NJson::TJsonValue& rolesJson = result.InsertValue("slave_roles", NJson::JSON_ARRAY);
    for (auto&& i : RoleRoles.GetSlaves()) {
        rolesJson.AppendValue(i.SerializeToTableRecord().SerializeToJson());
    }
    return result;
}

bool TRoleSnapshot::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    NStorage::TTableRecord tr;
    if (!tr.DeserializeFromJson(jsonInfo) || !Role.Parse(tr)) {
        return false;
    }
    if (jsonInfo.Has("actions")) {
        const NJson::TJsonValue::TArray* actionsArray;
        if (!jsonInfo["actions"].GetArrayPointer(&actionsArray)) {
            return false;
        }
        for (auto&& i : *actionsArray) {
            TLinkedRoleActionHeader actionLink;
            if (!actionLink.DeserializeFromJson(i)) {
                return false;
            }
            RoleActions.MutableSlaves().emplace_back(std::move(actionLink));
        }
    }
    if (jsonInfo.Has("slave_roles")) {
        const NJson::TJsonValue::TArray* rolesArray;
        if (!jsonInfo["slave_roles"].GetArrayPointer(&rolesArray)) {
            return false;
        }
        for (auto&& i : *rolesArray) {
            TLinkedRoleRoleHeader roleLink;
            if (!roleLink.DeserializeFromJson(i)) {
                return false;
            }
            RoleRoles.MutableSlaves().emplace_back(std::move(roleLink));
        }
    }
    return true;
}

template class TRoleSlavesDB<TLinkedRoleActionHeader>;
template class TRoleSlavesDB<TLinkedRoleRoleHeader>;
