#pragma once

#include "action.h"

#include <drive/backend/database/entity/manager.h>
#include <drive/backend/database/history/cache.h>
#include <drive/backend/database/history/manager.h>
#include <drive/backend/database/history/propositions.h>

#include <rtline/library/storage/structured.h>
#include <rtline/library/time_restriction/time_restriction.h>

#include <library/cpp/threading/hot_swap/hot_swap.h>

class TActionPropositions: public TPropositionsManager<TDBAction> {
public:
    TActionPropositions(const ITagsHistoryContext& context, const TPropositionsManagerConfig& propositionsConfig)
        : TPropositionsManager<TDBAction>(context, "drive_action_propositions", propositionsConfig)
    {
    }
};

class TActionsHistoryManager: public TIndexedAbstractHistoryManager<TDBAction> {
private:
    using TBase = TIndexedAbstractHistoryManager<TDBAction>;
public:
    TActionsHistoryManager(const IHistoryContext& context, const THistoryConfig& config)
        : TBase(context, "drive_actions_standart_history", config)
    {
    }
};

class TActionsDB: public TDBCacheWithHistoryOwner<TActionsHistoryManager, TDBAction> {
public:
    using TActionTags = TSet<TTagDescription::TConstPtr>;
    using TEvolutionTags = TSet<std::pair<TString, TString>>;
    struct TTagActionMatchCache: public TAtomicRefCount<TTagActionMatchCache> {
        TMap<TString, TActionTags> ActionTags;
        TMap<TString, TEvolutionTags> EvolutionTags;
    };
    using TTagActionMatchCachePtr = TIntrusivePtr<TTagActionMatchCache>;
    using TTagActionMatchCacheConstPtr = TIntrusiveConstPtr<TTagActionMatchCache>;

private:
    class TJsonPatch {
        R_FIELD(NJson::TJsonValue, OldValue);
        R_FIELD(NJson::TJsonValue, NewValue);
    public:
        TJsonPatch(const NJson::TJsonValue& oldValue, const NJson::TJsonValue& newValue)
            : OldValue(oldValue)
            , NewValue(newValue)
        {
        }
    };

    using TBase = TDBCacheWithHistoryOwner<TActionsHistoryManager, TDBAction>;
    using TBase::HistoryCacheDatabase;
    using TJsonDiff = TMap<TString, TJsonPatch>;

private:
    TActionPropositions Propositions;
    const ITagsHistoryContext& TagsContext;

    TMutex TagActionMatchCacheRebuildLock;
    mutable THotSwap<TTagActionMatchCache> TagActionMatchCache;
    mutable TThreadPool TagActionMatchCacheRebuildThreadPool;
    mutable TInstant TagActionMatchCacheUpdateTimestamp;

private:
    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TDBAction>& ev) const override;
    virtual bool DoRebuildCacheUnsafe() const override;
    virtual TStringBuf GetEventObjectId(const TObjectEvent<TDBAction>& ev) const override {
        return ev->GetName();
    }

    virtual bool Process(IMessage* message) override;

    bool ApplyDiffForChildren(const TJsonDiff& actionJsonDiff, const TString& actionName, const TString& userId, const bool force, NDrive::TEntitySession& session, const TSet<TString>& readyActions) const;
    template <class T>
    void RebuildTagActionMatchCache(const T& actions) const;

private:
    static bool BuildJsonDiff(const NJson::TJsonValue& from, const NJson::TJsonValue& to, TJsonDiff& result);
    static bool ApplyPatchJson(NJson::TJsonValue& result, const TJsonDiff& patch);
    static TString GetTableName() {
        return "drive_tag_actions";
    }

public:
    TActionsDB(const ITagsHistoryContext& context, const THistoryConfig& config, const TPropositionsManagerConfig& propositionsConfig);

    const TPropositionsManager<TDBAction>& GetPropositions() const {
        return Propositions;
    }

    template <class T>
    TVector<T> GetActionsWithType() const {
        TVector<T> result;
        TVector<TDBAction> actions;
        Y_ENSURE_BT(GetAllObjectsFromCache(actions));
        for (auto&& action : actions) {
            if (auto actionT = action.GetAs<T>()) {
                result.emplace_back(*actionT);
            }
        }
        return result;
    }

    template <class T>
    TVector<TAtomicSharedPtr<const T>> GetActionsPtrWithType() const {
        TVector<TDBAction> actions;
        Y_ENSURE_BT(GetAllObjectsFromCache(actions));

        TVector<TAtomicSharedPtr<const T>> result;
        for (auto&& action : actions) {
            auto impl = std::dynamic_pointer_cast<const T>(action.Impl());
            if (impl) {
                result.push_back(std::move(impl));
            }
        }
        return result;
    }

    template <class T>
    TSet<TCiString> GetOfferBuilderTags() const {
        TSet<TCiString> result;
        auto actions = GetActionsPtrWithType<T>();
        for (auto&& i : actions) {
            result.insert(i->GetGrouppingTags().begin(), i->GetGrouppingTags().end());
        }
        return result;
    }

    template <class T>
    TVector<TString> GetActionNamesWithType(bool reportDeprecated) const {
        TVector<TString> result;
        TVector<TDBAction> actions;
        Y_ENSURE_BT(GetAllObjectsFromCache(actions));

        for (auto&& action : actions) {
            if (!reportDeprecated && action->GetDeprecated()) {
                continue;
            }
            if (auto actionT = action.GetAs<T>()) {
                result.emplace_back(actionT->GetName());
            }
        }
        return result;
    }

    TActionTags GetActionTags(const TString& name, TInstant since = TInstant::Zero()) const;
    TActionTags GetActionTags(const TString& name, const TTagActionMatchCache* tagActionMatchCache) const;
    TEvolutionTags GetEvolutionTags(const TString& name, TInstant since = TInstant::Zero()) const;
    TEvolutionTags GetEvolutionTags(const TString& name, const TTagActionMatchCache* tagActionMatchCache) const;
    TTagActionMatchCacheConstPtr GetTagActionMatchCache(TInstant since = TInstant::Zero()) const;

    enum class EAvailableActions: ui32 {
        Add = 1,
        Modify = 1 << 1
    };

    using TAvailableActions = ui32;
    static const ui32 AllActionsAvailable = Max<ui32>();

    bool Upsert(TUserAction::TConstPtr action, const TString& userId, NDrive::TEntitySession& session, const TSet<TString>& readyActions = {}, const TAvailableActions abilities = AllActionsAvailable) const;
    bool ForceUpsert(TUserAction::TConstPtr action, const TString& userId, NDrive::TEntitySession& session, const TSet<TString>& readyActions = {}) const;
    bool RemoveAction(const TString& actionName, const TString& userId, NDrive::TEntitySession& session) const;
    bool RemoveActions(const TVector<TString>& actions, const TString& userId, NDrive::TEntitySession& session) const;
    bool DeprecateActions(const TVector<TString>& actions, const TString& userId, NDrive::TEntitySession& session) const;
};

class TDriveRoleHeader {
    R_FIELD(TString, Name);
    R_FIELD(bool, Optional, false);
    R_FIELD(bool, IsIDM, false);
    R_FIELD(bool, IsPublic, false);
    R_FIELD(ui32, Group, 0);
    R_FIELD(TString, Description);
    R_FIELD(TSet<TCiString>, GrouppingTags);

public:
    const TString& GetRoleId() const {
        return Name;
    }

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, Name, -1);
        R_FIELD(i32, Optional, -1);
        R_FIELD(i32, IsIDM, -1);
        R_FIELD(i32, IsPublic, -1);
        R_FIELD(i32, Group, -1);
        R_FIELD(i32, Description, -1);
        R_FIELD(i32, GrouppingTags, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };
    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

    TDriveRoleHeader() {
    }
    TDriveRoleHeader(const TString& name)
        : Name(name)
    {
    }

    bool Parse(const NStorage::TTableRecord& row);
    void Serialize(NStorage::TTableRecord& row) const;

    NStorage::TTableRecord SerializeToTableRecord() const;
    NJson::TJsonValue BuildJsonReport() const;
};

class TRolesHistoryManager: public TIndexedAbstractHistoryManager<TDriveRoleHeader> {
private:
    using TBase = TIndexedAbstractHistoryManager<TDriveRoleHeader>;

public:
    TRolesHistoryManager(const IHistoryContext& context, const THistoryConfig& config)
        : TBase(context, "drive_roles_history", config)
    {
    }
};

class TRolesDB: public TDBCacheWithHistoryOwner<TRolesHistoryManager, TDriveRoleHeader> {
private:
    using TBase = TDBCacheWithHistoryOwner<TRolesHistoryManager, TDriveRoleHeader>;
    using TBase::HistoryManager;
    using TBase::HistoryCacheDatabase;

protected:
    virtual bool DoRebuildCacheUnsafe() const override;

    virtual TStringBuf GetEventObjectId(const TObjectEvent<TDriveRoleHeader>& ev) const override {
        return ev.GetRoleId();
    }

    virtual void AcceptHistoryEventUnsafe(const TObjectEvent<TDriveRoleHeader>& ev) const override {
        if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
            Objects.erase(ev.GetRoleId());
        } else {
            Objects[ev.GetRoleId()] = ev;
        }
    }

public:
    TRolesDB(const IHistoryContext& context, const THistoryConfig& hConfig)
        : TBase("drive_roles", context, hConfig)
    {
    }

    bool Upsert(const TDriveRoleHeader& role, const TString& userId, NDrive::TEntitySession& session) const;
    bool RemoveRole(const TString& roleName, const TString& userId, NDrive::TEntitySession& session) const;
};

class TBaseLinkHeader {
public:
    R_FIELD(TString, SlaveObjectId);
    R_FIELD(TString, RoleId);

private:
    TTimeRestrictionsPool<TTimeRestriction> TimeRestrictions;

public:
    TBaseLinkHeader() = default;
    TBaseLinkHeader(const TString& slaveObjectId, const TString& roleId)
        : SlaveObjectId(slaveObjectId)
        , RoleId(roleId)
    {
    }

    NJson::TJsonValue SerializeMetaToJson() const {
        NJson::TJsonValue jsonInfo(NJson::JSON_MAP);
        if (!TimeRestrictions.Empty()) {
            jsonInfo["time_restrictions"] = TimeRestrictions.SerializeToJson();
            jsonInfo["time_restriction"] = TimeRestrictions.GetRestrictions().front().SerializeToJson();
        }
        return jsonInfo;
    }

    template <class TDecoder>
    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
        READ_DECODER_VALUE(decoder, values, SlaveObjectId);
        READ_DECODER_VALUE(decoder, values, RoleId);
        if (!SlaveObjectId) {
            return false;
        }
        if (!RoleId) {
            return false;
        }
        TString tempMetaStr;
        READ_DECODER_VALUE_TEMP_OPT(decoder, values, tempMetaStr, Meta);
        if (!!tempMetaStr) {
            NJson::TJsonValue jsonInfo;
            if (!NJson::ReadJsonFastTree(tempMetaStr, &jsonInfo)) {
                return false;
            }
            if (!jsonInfo.IsMap()) {
                return false;
            }
            const auto& timeRestrictions = jsonInfo["time_restrictions"];
            const auto& timeRestriction = jsonInfo["time_restriction"];
            if (timeRestrictions.IsDefined()) {
                if (!TimeRestrictions.DeserializeFromJson(timeRestrictions)) {
                    return false;
                }
            } else if (timeRestriction.IsDefined()) {
                TTimeRestriction restriction;
                if (!restriction.DeserializeFromJson(timeRestriction)) {
                    return false;
                }
                SetTimeRestriction(restriction);
            }
        }
        return true;
    }

    TBaseLinkHeader& SetTimeRestriction(const TTimeRestriction& timeRestriction) {
        TTimeRestriction restriction = timeRestriction;
        TimeRestrictions.Clear();
        restriction.Compile();
        TimeRestrictions.Add(restriction);
        return *this;
    }

    bool IsActual(const TInstant ts) const {
        return TimeRestrictions.Empty() || TimeRestrictions.IsActualNow(ts);
    }
};

class TLinkedRoleActionHeader: public TBaseLinkHeader {
private:
    using TBase = TBaseLinkHeader;

public:
    static TString GetSlaveIdFieldName() {
        return "action_id";
    }

    static TString GetTableName() {
        return "drive_role_actions";
    }

    static TString GetHistoryTableName() {
        return "drive_role_actions_history";
    }

    static TString GetSlavesJsonField() {
        return "actions";
    }

    using TBase::TBase;
    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, SlaveObjectId, -1);
        R_FIELD(i32, RoleId, -1);
        R_FIELD(i32, Meta, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            SlaveObjectId = GetFieldDecodeIndex("action_id", decoderBase);
            RoleId = GetFieldDecodeIndex("role_id", decoderBase);
            Meta = GetFieldDecodeIndex("role_action_meta", decoderBase);
        }
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
        return TBase::DeserializeWithDecoder(decoder, values, hContext);
    }

    NJson::TJsonValue BuildJsonReport() const;

    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);
    bool DeserializeFromTableRow(const NStorage::TTableRecord& row);

    NStorage::TTableRecord SerializeToTableRecord() const;
};

class TLinkedRoleRoleHeader: public TBaseLinkHeader {
private:
    using TBase = TBaseLinkHeader;

public:
    static TString GetSlaveIdFieldName() {
        return "slave_role_id";
    }

    static TString GetTableName() {
        return "drive_role_roles";
    }

    static TString GetHistoryTableName() {
        return "drive_role_roles_history";
    }

    static TString GetSlavesJsonField() {
        return "roles";
    }

    using TBase::TBase;

    class TDecoder: public TBaseDecoder {
        R_FIELD(i32, SlaveObjectId, -1);
        R_FIELD(i32, RoleId, -1);
        R_FIELD(i32, Meta, -1);
    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            SlaveObjectId = GetFieldDecodeIndex("slave_role_id", decoderBase);
            RoleId = GetFieldDecodeIndex("role_id", decoderBase);
            Meta = GetFieldDecodeIndex("link_meta", decoderBase);
        }
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* hContext) {
        return TBase::DeserializeWithDecoder<TDecoder>(decoder, values, hContext);
    }

    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);
    bool DeserializeFromTableRow(const NStorage::TTableRecord& row);

    NJson::TJsonValue BuildJsonReport() const;
    NStorage::TTableRecord SerializeToTableRecord() const;
};

template <class TSlaveLink>
class TDriveRoleSlaves {
private:
    TVector<TSlaveLink> Slaves;

public:
    const TString& GetRoleId() const {
        return Slaves.size() ? Slaves.front().GetRoleId() : Default<TString>();
    }

    bool HasSlaves(const TSet<TString>& slaveIds) const {
        for (auto&& i : Slaves) {
            if (slaveIds.contains(i.GetSlaveObjectId())) {
                return true;
            }
        }
        return false;
    }

    const TVector<TSlaveLink>& GetSlaves() const {
        return Slaves;
    }

    TVector<TSlaveLink>& MutableSlaves() {
        return Slaves;
    }

    NJson::TJsonValue BuildJsonReport() const {
        NJson::TJsonValue result = NJson::JSON_MAP;
        NJson::TJsonValue roles(NJson::JSON_ARRAY);
        for (auto&& i : Slaves) {
            roles.AppendValue(i.BuildJsonReport());
        }
        result[TSlaveLink::GetSlavesJsonField()] = std::move(roles);
        return result;
    }

    void UpsertLink(const TSlaveLink& link) {
        for (auto&& i : Slaves) {
            if (i.GetRoleId() == link.GetRoleId() && i.GetSlaveObjectId() == link.GetSlaveObjectId()) {
                i = link;
                return;
            }
        }
        Slaves.emplace_back(link);
    }

    void RemoveLink(const TString& slaveLink) {
        const auto pred = [&slaveLink](const TSlaveLink& link) -> bool {
            return link.GetSlaveObjectId() == slaveLink;
        };
        Slaves.erase(std::remove_if(Slaves.begin(), Slaves.end(), pred), Slaves.end());
    }
};

using TDriveRoleRoles = TDriveRoleSlaves<TLinkedRoleRoleHeader>;
using TDriveRoleActions = TDriveRoleSlaves<TLinkedRoleActionHeader>;

template <class TSlaveLink>
class TDriveRoleSlavesHistoryManager: public TIndexedAbstractHistoryManager<TSlaveLink> {
private:
    using TBase = TIndexedAbstractHistoryManager<TSlaveLink>;

public:
    using THistoryEvent = TObjectEvent<TSlaveLink>;

    TDriveRoleSlavesHistoryManager(const IHistoryContext& context, const THistoryConfig& config)
        : TBase(context, TSlaveLink::GetHistoryTableName(), config) {
    }
};

using TDriveRoleRolesHistoryManager = TDriveRoleSlavesHistoryManager<TLinkedRoleRoleHeader>;
using TDriveRoleActionsHistoryManager = TDriveRoleSlavesHistoryManager<TLinkedRoleActionHeader>;

template <class TSlaveLink>
class TRoleSlavesDB: public TDBCacheWithHistoryOwner<TDriveRoleSlavesHistoryManager<TSlaveLink>, TDriveRoleSlaves<TSlaveLink>> {
private:
    using TBase = TDBCacheWithHistoryOwner<TDriveRoleSlavesHistoryManager<TSlaveLink>, TDriveRoleSlaves<TSlaveLink>>;
    using TBase::HistoryCacheDatabase;
    using TBase::Objects;
    using TBase::HistoryManager;

public:
    using TObjects = typename TBase::TObjects;

    class TObjectSnapshot: public TAtomicRefCount<TObjectSnapshot> {
    public:
        TObjectSnapshot(TObjects&& objects, TInstant timestamp)
            : Objects(std::move(objects))
            , Timestamp(timestamp)
        {
        }

        const TObjects& Get() const {
            return Objects;
        }

        TInstant GetTimestamp() const {
            return Timestamp;
        }

    private:
        const TObjects Objects;
        const TInstant Timestamp;
    };
    using TObjectSnapshotPtr = TIntrusivePtr<TObjectSnapshot>;
    using TObjectSnapshotConstPtr = TIntrusiveConstPtr<TObjectSnapshot>;
    using TRoleSlaves = TDriveRoleSlaves<TSlaveLink>;

public:
    TRoleSlavesDB(const IHistoryContext& context, const THistoryConfig& hConfig)
        : TBase(TSlaveLink::GetTableName(), context, hConfig)
    {
    }

    virtual TStringBuf GetEventObjectId(const TObjectEvent<TSlaveLink>& ev) const override {
        return ev.GetRoleId();
    }

    TObjectSnapshotConstPtr GetObjectSnapshot() const {
        return ObjectSnapshot.AtomicLoad();
    }

    TVector<TRoleSlaves> GetInfo(const TVector<TString>& roles, const TInstant reqActuality) const {
        TVector<TRoleSlaves> result;
        Y_ENSURE_BT(TBase::GetCustomObjectsFromCache(result, MakeSet(roles), reqActuality));
        return result;
    }

    TVector<TRoleSlaves> GetInfo(const TInstant reqActuality) const {
        TVector<TRoleSlaves> result;
        Y_ENSURE_BT(TBase::GetAllObjectsFromCache(result, reqActuality));
        return result;
    }

    bool ApplySnapshot(const TString& roleId, const TVector<TSlaveLink>& links, const TString& userId, NDrive::TEntitySession& session) const;

    bool Link(const TVector<TSlaveLink>& actions, const TString& userId, NDrive::TEntitySession& session) const;
    bool Link(const TSlaveLink& actionHeader, const TString& userId, NDrive::TEntitySession& session) const;

    bool Unlink(const TVector<TString>& slaves, const TString& role, const TString& userId, NDrive::TEntitySession& session) const;
    bool Unlink(const TString& slave, const TString& role, const TString& userId, NDrive::TEntitySession& session) const;

protected:
    void AcceptHistoryEventUnsafe(const TObjectEvent<TSlaveLink>& ev) const override;
    bool DoRebuildCacheUnsafe() const override;
    bool Refresh() override;

private:
    mutable THotSwap<TObjectSnapshot> ObjectSnapshot;
};

using TRoleActionsDB = TRoleSlavesDB<TLinkedRoleActionHeader>;
using TRoleRolesDB = TRoleSlavesDB<TLinkedRoleRoleHeader>;

class TRoleInfoDB {
private:
    mutable TRoleActionsDB RoleActions;
    mutable TRoleRolesDB RoleRoles;

public:
    TRoleInfoDB(const IHistoryContext& context, const THistoryConfig& historyConfig);
    ~TRoleInfoDB();

    bool UnlinkAllFromRole(const TString& roleId, const TString& userId, NDrive::TEntitySession& session) const;

    TRoleRolesDB& GetRoleRoles() const {
        return RoleRoles;
    }

    TRoleActionsDB& GetRoleActions() const {
        return RoleActions;
    }

    TSet<TString> GetActions(const TSet<TString>& roles, TInstant actuality = TInstant::Zero(), TInstant ts = TInstant::Max(), const TSet<TString>& disabledRoles = {}) const;
};

class TRoleSnapshot {
private:
    using TActions = TMap<TString, TDBAction>;
    using TRoles = TMap<TString, TDriveRoleHeader>;

private:
    R_FIELD(TDriveRoleHeader, Role);
    R_FIELD(TDriveRoleRoles, RoleRoles);
    R_FIELD(TDriveRoleActions, RoleActions);
    R_FIELD(TActions, Actions);
    R_FIELD(TRoles, Roles);

public:
    class TDecoder: public TBaseDecoder {
    private:
        using TBase = TBaseDecoder;

    private:
        R_FIELD(i32, Data, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase) {
            Data = TBase::GetFieldDecodeIndex("data", decoderBase);
        }
    };

public:
    NJson::TJsonValue BuildJsonReport(bool isFullReport = false, bool serializeSlaveObjects = true) const;

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);
    bool DeserializeFromJson(const NJson::TJsonValue& jsonInfo);
    NJson::TJsonValue SerializeToJson() const;
    NStorage::TTableRecord SerializeToTableRecord() const;
};
