#pragma once

#include "roles.h"

template <class TSlaveLink>
bool TRoleSlavesDB<TSlaveLink>::ApplySnapshot(const TString& roleId, const TVector<TSlaveLink>& links, const TString& userId, NDrive::TEntitySession& session) const {
    TSet<TString> slaveIds;
    for (auto&& i : links) {
        if (i.GetRoleId() != roleId) {
            session.SetErrorInfo("TRoleRolesDB::ApplySnapshot", "incorrect_links_set", EDriveSessionResult::IncorrectRequest);
            return false;
        }
        slaveIds.emplace(i.GetSlaveObjectId());
    }
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(TSlaveLink::GetTableName());
    {
        NStorage::TObjectRecordsSet<TSlaveLink> removedLinks;
        if (!table->RemoveRow("role_id = '" + roleId + "' AND " + TSlaveLink::GetSlaveIdFieldName() + " NOT IN ('" + JoinSeq(",", slaveIds) + "')", session.GetTransaction(), &removedLinks)->IsSucceed()) {
            session.SetErrorInfo("remove_links_role_role", session.GetStringReport(), EDriveSessionResult::TransactionProblem);
            return false;
        }
        if (!HistoryManager->AddHistory(removedLinks.GetObjects(), userId, EObjectHistoryAction::Remove, session)) {
            return false;
        }
    }
    for (auto&& i : links) {
        auto row = i.SerializeToTableRecord();
        NStorage::TObjectRecordsSet<TSlaveLink> recordsUpsert;
        bool isUpdate = false;
        NStorage::TTableRecord rowUnique;
        rowUnique.Set("role_id", roleId).Set(TSlaveLink::GetSlaveIdFieldName(), i.GetSlaveObjectId());
        if (!table->Upsert(row, session.GetTransaction(), rowUnique, &isUpdate, &recordsUpsert)->IsSucceed()) {
            session.SetErrorInfo("link_role_role", session.GetStringReport(), EDriveSessionResult::TransactionProblem);
            return false;
        }
        if (!HistoryManager->AddHistory(recordsUpsert.GetObjects(), userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
            return false;
        }
    }
    return true;
}

template <class TSlaveLink>
bool TRoleSlavesDB<TSlaveLink>::Link(const TVector<TSlaveLink>& actions, const TString& userId, NDrive::TEntitySession& session) const {
    for (auto&& i : actions) {
        if (!Link(i, userId, session)) {
            return false;
        }
    }
    return true;
}

template <class TSlaveLink>
bool TRoleSlavesDB<TSlaveLink>::Link(const TSlaveLink& actionHeader, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(TSlaveLink::GetTableName());
    NStorage::TTableRecord row = actionHeader.SerializeToTableRecord();
    bool isUpdate = false;
    if (!table->Upsert(row, session.GetTransaction(), row, &isUpdate)->IsSucceed()) {
        session.SetErrorInfo("link_role_action", session.GetStringReport(), EDriveSessionResult::TransactionProblem);
        return false;
    }
    return HistoryManager->AddHistory(actionHeader, userId, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session);
}

template <class TSlaveLink>
bool TRoleSlavesDB<TSlaveLink>::Unlink(const TVector<TString>& slaves, const TString& role, const TString& userId, NDrive::TEntitySession& session) const {
    for (auto&& i : slaves) {
        if (!Unlink(i, role, userId, session)) {
            return false;
        }
    }
    return true;
}

template <class TSlaveLink>
bool TRoleSlavesDB<TSlaveLink>::Unlink(const TString& slave, const TString& role, const TString& userId, NDrive::TEntitySession& session) const {
    NStorage::ITableAccessor::TPtr table = HistoryCacheDatabase->GetTable(TSlaveLink::GetTableName());
    NStorage::TTableRecord row;
    row.Set("role_id", role).Set(TSlaveLink::GetSlaveIdFieldName(), slave);
    NStorage::TObjectRecordsSet<TSlaveLink> records;
    if (!table->RemoveRow(row, session.GetTransaction(), &records)->IsSucceed()) {
        session.SetErrorInfo("unlink_role_action", session.GetStringReport(), EDriveSessionResult::TransactionProblem);
        return false;
    }
    return HistoryManager->AddHistory(records.GetObjects(), userId, EObjectHistoryAction::Remove, session);
}

template <class TSlaveLink>
void TRoleSlavesDB<TSlaveLink>::AcceptHistoryEventUnsafe(const TObjectEvent<TSlaveLink>& ev) const {
    auto it = Objects.find(ev.GetRoleId());
    if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
        if (it == Objects.end()) {
            return;
        }
        it->second.RemoveLink(ev.GetSlaveObjectId());
        if (!it->second.GetSlaves().size()) {
            Objects.erase(it);
        }
    } else {
        Objects[ev.GetRoleId()].UpsertLink(ev);
    }
}

template <class TSlaveLink>
bool TRoleSlavesDB<TSlaveLink>::DoRebuildCacheUnsafe() const {
    NStorage::TObjectRecordsSet<TSlaveLink> records;
    TInstant timestamp;
    {
        TTransactionPtr transaction = HistoryCacheDatabase->CreateTransaction(true);
        auto tagsTable = HistoryCacheDatabase->GetTable(TSlaveLink::GetTableName());

        TQueryResultPtr queryResult = tagsTable->GetRows("", records, transaction);

        if (!queryResult->IsSucceed()) {
            return false;
        }
        timestamp = Now();
    }
    for (auto&& i : records) {
        Objects[i.GetRoleId()].MutableSlaves().emplace_back(i);
    }
    {
        auto objects = Objects;
        auto snapshot = MakeIntrusive<TObjectSnapshot>(std::move(objects), timestamp);
        ObjectSnapshot.AtomicStore(snapshot);
    }
    return true;
}

template <class TSlaveLink>
bool TRoleSlavesDB<TSlaveLink>::Refresh() {
    if (!TBase::Refresh()) {
        return false;
    }
    TObjectSnapshotPtr snapshot;
    {
        auto guard = TBase::MakeObjectReadGuard();
        auto objects = Objects;
        auto timestamp = TBase::GetLastUpdateTimestamp();
        snapshot = MakeIntrusive<TObjectSnapshot>(std::move(objects), timestamp);
    }
    ObjectSnapshot.AtomicStore(snapshot);
    return true;
}
