#include "user.h"

#include <drive/backend/logging/evlog.h>

#include <util/string/type.h>

namespace {
    TUnistatSignal<double> CacheHit { { "user_roles-cache-hit" }, false };
    TUnistatSignal<double> CacheMiss { { "user_roles-cache-miss" }, false };
    TUnistatSignal<double> CacheExpired { { "user_roles-cache-expired" }, false };
    TUnistatSignal<double> CacheInvalidated { { "user_roles-cache-invalidated" }, false };
}

TOptionalObjectEvents<TUserRole> TUserRolesHistoryManager::GetEventsByUser(const TString& userId, NDrive::TEntitySession& session, TInstant since) const {
    return GetEvents({}, since, session, TQueryOptions()
        .AddGenericCondition("user_id", userId)
    );
}

TUserRolesDB::TUserRolesDB(const IHistoryContext& context, TDuration defaultLifetime)
    : IAutoActualization("user_roles-cache", TDuration::Seconds(1))
    , TBaseEntityManager<TUserRole>(context.GetDatabase())
    , HistoryManager(context)
    , DefaultLifetime(defaultLifetime)
    , ObjectCache(128 * 1024)
{
}

TDriveUserRoles TUserRolesDB::GetCachedUserRolesImpl(const TUserRolesDB& self, const TString& userId, TInstant statementDeadline) {
    auto eventLogger = NDrive::GetThreadEventLogger();
    auto lifetimeKey = "user_roles.cache_lifetime";
    auto lifetime = NDrive::HasServer()
        ? NDrive::GetServer().GetSettings().GetValue<TDuration>(lifetimeKey)
        : Nothing();
    auto now = Now();
    auto threshold = now - lifetime.GetOrElse(self.DefaultLifetime);
    auto optionalObject = self.ObjectCache.find(userId);
    if (optionalObject && optionalObject->GetTimestamp() > threshold) {
        if (eventLogger) {
            eventLogger->AddEvent(NJson::TMapBuilder
                ("event", "UserRolesCacheHit")
                ("user_id", userId)
                ("timestamp", NJson::ToJson(optionalObject->GetTimestamp()))
            );
        }
        CacheHit.Signal(1);
        return std::move(*optionalObject);
    }
    if (optionalObject) {
        CacheExpired.Signal(1);
    }

    if (eventLogger) {
        eventLogger->AddEvent(NJson::TMapBuilder
            ("event", "UserRolesCacheMiss")
            ("user_id", userId)
        );
    }

    auto lockTimeout = TDuration::Zero();
    auto statementTimeout = statementDeadline - now;
    auto session = self.BuildSession(true, false, lockTimeout, statementTimeout);
    auto restoredObject = self.RestoreUserRoles(userId, session);
    if (!restoredObject) {
        session.Check();
    }

    Y_ASSERT(userId == restoredObject->GetUserId());
    self.ObjectCache.update(userId, *restoredObject);
    CacheMiss.Signal(1);
    return std::move(*restoredObject);
}

TUserRolesDB::TExpectedUserRoles TUserRolesDB::GetCachedUserRoles(const TString& userId, TInstant statementDeadline) const {
    return WrapUnexpected<TCodedException>(GetCachedUserRolesImpl, *this, userId, statementDeadline);
}

TUserRolesDB::TOptionalUserRoles TUserRolesDB::RestoreUserRoles(const TString& userId, NDrive::TEntitySession& session) const {
    auto queryOptions = TQueryOptions()
        .AddGenericCondition("user_id", userId);
    auto timestamp = Now();
    auto optionalRecords = Fetch(session, queryOptions);
    if (!optionalRecords) {
        return {};
    }
    TDriveUserRoles result;
    result.SetTimestamp(timestamp);
    result.SetUserId(userId);
    for (auto&& userRole : *optionalRecords) {
        Y_ASSERT(userRole.GetUserId() == userId);
        bool added = result.AddRole(userRole);
        Y_ASSERT(added);
    }
    return result;
}

bool TUserRolesDB::Refresh() {
    auto session = BuildTx<NSQL::ReadOnly>();
    if (!LastEventId) {
        LastEventId = HistoryManager.GetMaxEventIdOrThrow(session);
    }

    auto since = LastEventId ? *LastEventId + 1 : 0;
    auto optionalEvents = HistoryManager.GetEvents(since, session, 1000);
    if (!optionalEvents) {
        ERROR_LOG << GetName() << ": cannot GetEventsSince " << since << ": " << session.GetStringReport() << Endl;
        return false;
    }
    for (auto&& ev : *optionalEvents) {
        LastEventId = std::max(LastEventId.GetOrElse(0), ev.GetHistoryEventId());
        bool erased = ObjectCache.erase(ev.GetUserId());
        if (erased) {
            CacheInvalidated.Signal(1);
            INFO_LOG << GetName() << ": invalidate " << ev.GetUserId() << Endl;
        }
    }
    return true;
}

bool TUserRolesDB::GetUsersWithRoles(const TSet<TString>& roles, TVector<TString>& result, bool checkActive, ui64 limit) const {
    auto session = BuildTx<NSQL::ReadOnly>();
    auto queryOptions = NSQL::TQueryOptions().SetGenericCondition("role_id", roles);
    if (checkActive) {
        queryOptions.SetGenericCondition("active", true);
    }
    if (limit) {
        queryOptions.SetLimit(limit);
        queryOptions.SetOrderBy({ "user_id "});
    }
    auto records = Fetch(session, queryOptions);
    if (!records) {
        return false;
    }
    TMap<TString, ui32> filter;
    for (auto&& userRole : *records) {
        const auto& userId = userRole.GetUserId();
        if (++filter[userId] >= roles.size()) {
            result.push_back(userId);
        }
    }
    return true;
}

bool TUserRolesDB::GetUsersWithAtLeastOneRoles(const TSet<TString>& roles, TSet<TString>& result, bool checkActive) const {
    auto session = BuildTx<NSQL::ReadOnly>();
    auto queryOptions = NSQL::TQueryOptions().SetGenericCondition("role_id", roles);
    if (checkActive) {
        queryOptions.SetGenericCondition("active", true);
    }
    auto records = Fetch(session, queryOptions);
    if (!records) {
        return false;
    }
    for (auto&& userRole : *records) {
        const auto& userId = userRole.GetUserId();
        result.insert(userId);
    }
    return true;
}

bool TUserRolesDB::Link(const TUserRole& userRole, const TString& userActor, NDrive::TEntitySession& session) const {
    NStorage::TTableRecord rowUpdate = userRole.SerializeToTableRecord();

    NStorage::TTableRecord rowCondition;
    rowCondition.Set("user_id", userRole.GetUserId()).Set("role_id", userRole.GetRoleId());
    NStorage::ITableAccessor::TPtr table = GetDatabase().GetTable("user_roles");
    bool isUpdate = false;
    auto result = table->Upsert(rowUpdate, session.GetTransaction(), rowCondition, &isUpdate);
    if (!result || !result->IsSucceed()) {
        session.SetErrorInfo("link_user_role", "Upsert failed", EDriveSessionResult::TransactionProblem);
        return false;
    }
    if (!HistoryManager.AddHistory(userRole, userActor, isUpdate ? EObjectHistoryAction::UpdateData : EObjectHistoryAction::Add, session)) {
        return false;
    }
    return true;
}

bool TUserRolesDB::Unlink(const TString& userId, const TString& roleName, const TString& userActor, NDrive::TEntitySession& session) const {
    NStorage::TTableRecord rowCondition;
    rowCondition.Set("user_id", userId).Set("role_id", roleName);

    NStorage::ITableAccessor::TPtr table = GetDatabase().GetTable("user_roles");
    TRecordsSet records;
    auto result = table->RemoveRow(rowCondition.BuildCondition(*session.GetTransaction()), session.GetTransaction(), &records);
    if (!result || !result->IsSucceed()) {
        session.SetErrorInfo("unlink_user_role", "RemoveRow failed", EDriveSessionResult::TransactionProblem);
        return false;
    }
    for (auto&& i : records) {
        TUserRole userRole;
        if (!userRole.Parse(i)) {
            session.SetErrorInfo("cannot parse", "removed user_role", EDriveSessionResult::InternalError);
            return false;
        }
        if (!HistoryManager.AddHistory(userRole, userActor, EObjectHistoryAction::Remove, session)) {
            return false;
        }
    }
    return true;
}

TUserRole::TDecoder::TDecoder(const TMap<TString, ui32>& decoderBase) {
    UserId = GetFieldDecodeIndex("user_id", decoderBase);
    RoleId = GetFieldDecodeIndex("role_id", decoderBase);
    Deadline = GetFieldDecodeIndex("deadline", decoderBase);
    Active = GetFieldDecodeIndex("active", decoderBase);
    Forced = GetFieldDecodeIndex("forced", decoderBase);
}

bool TUserRole::DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/) {
    READ_DECODER_VALUE(decoder, values, UserId);
    READ_DECODER_VALUE(decoder, values, RoleId);
    READ_DECODER_VALUE(decoder, values, Active);
    READ_DECODER_VALUE_INSTANT(decoder, values, Deadline);
    if (decoder.GetForced() >= 0) {
        CHECK_WITH_LOG(static_cast<size_t>(decoder.GetForced()) < values.size());
        const auto& value = values[decoder.GetForced()];
        if (value && !TryFromString(value, Forced)) {
            ERROR_LOG << "cannot parse Forced field: " << value << Endl;
            return false;
        }
    }
    return true;
}

NJson::TJsonValue TUserRole::BuildJsonReport() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "user_id", UserId);
    NJson::InsertField(result, "role_id", RoleId);
    NJson::InsertField(result, "active", ToString(Active));
    if (Deadline != TInstant::Max()) {
        NJson::InsertField(result, "deadline", ToString(Deadline.Seconds()));
    }
    if (Forced) {
        NJson::InsertField(result, "forced", Forced);
    }
    return result;
}

bool TUserRole::IsActive(TInstant timestamp) const {
    return Active && Deadline > timestamp;
}

bool TUserRole::DeserializeFromTableRecord(const NStorage::TTableRecord& row, const ITagsHistoryContext* /*context*/) {
    return Parse(row);
}

bool TUserRole::Parse(const NStorage::TTableRecord& row) {
    UserId = row.Get("user_id");
    if (GetUuid(UserId).IsEmpty()) {
        return false;
    }
    RoleId = row.Get("role_id");
    if (!RoleId) {
        return false;
    }
    if (!row.Has("active")) {
        Active = true;
    } else {
        Active = IsTrue(row.Get("active"));
    }
    if (row.Has("deadline")) {
        ui64 dl;
        if (!row.TryGet("deadline", dl)) {
            return false;
        }
        Deadline = TInstant::Seconds(dl);
    } else {
        Deadline = TInstant::Max();
    }
    if (row.Has("forced")) {
        Forced = IsTrue(row.Get("forced"));
    }
    return true;
}

NStorage::TTableRecord TUserRole::SerializeToTableRecord() const {
    NStorage::TTableRecord row;
    row.Set("user_id", UserId);
    row.Set("role_id", RoleId);
    row.Set("deadline", Deadline.Seconds());
    row.Set("active", Active);
    if (Forced) {
        row.Set("forced", Forced);
    }
    return row;
}

void TUserRole::DoBuildReportItem(NJson::TJsonValue& value) const {
    value["object_id"] = UserId;
    value["role_id"] = RoleId;
    value["deadline"] = Deadline.Seconds();
    value["active"] = Active;
    value["forced"] = Forced;
}

TUserRoleInfo::TUserRoleInfo(const TUserRole& userRole, const TDriveRoleHeader* roleInfo)
    : TUserRole(userRole)
{
    if (roleInfo) {
        RoleInfo = MakeAtomicShared<TDriveRoleHeader>(*roleInfo);
    }
}

NJson::TJsonValue TUserRoleInfo::BuildJsonReport() const {
    NJson::TJsonValue result = TUserRole::BuildJsonReport();
    if (!!RoleInfo) {
        result.InsertValue("role_info", RoleInfo->BuildJsonReport());
    }
    return result;
}

bool TDriveUserRoles::AddRole(const TUserRole& role) {
    for (auto&& i : Roles) {
        if (role.GetRoleId() == i.GetRoleId()) {
            return false;
        }
    }
    Roles.push_back(role);
    return true;
}
