#pragma once

#include "roles.h"

#include <drive/library/cpp/threading/concurrent_cache.h>

#include <rtline/util/auto_actualization.h>

class TUserRole {
    R_FIELD(TString, UserId);
    R_FIELD(TString, RoleId);
    R_FIELD(TInstant, Deadline, TInstant::Max());
    R_FIELD(bool, Active, true);
    R_FIELD(bool, Forced, false);

public:
    TUserRole() = default;
    TUserRole(const TString& userId, const TString& roleId)
        : UserId(userId)
        , RoleId(roleId)
    {
    }

public:
    class TDecoder: public TBaseDecoder {
    private:
        R_FIELD(i32, UserId, -1);
        R_FIELD(i32, RoleId, -1);
        R_FIELD(i32, Deadline, -1);
        R_FIELD(i32, Active, -1);
        R_FIELD(i32, Forced, -1);

    public:
        TDecoder() = default;
        TDecoder(const TMap<TString, ui32>& decoderBase);
    };

    bool DeserializeWithDecoder(const TDecoder& decoder, const TConstArrayRef<TStringBuf>& values, const IHistoryContext* /*hContext*/);

public:
    NJson::TJsonValue BuildJsonReport() const;
    bool IsActive(TInstant timestamp) const;

    bool DeserializeFromTableRecord(const NStorage::TTableRecord& row, const ITagsHistoryContext* /*context*/);
    NStorage::TTableRecord SerializeToTableRecord() const;

    bool Parse(const NStorage::TTableRecord& row);

protected:
    void DoBuildReportItem(NJson::TJsonValue& value) const;
};

class TUserRoleInfo: public TUserRole {
private:
    TAtomicSharedPtr<TDriveRoleHeader> RoleInfo;

public:
    TUserRoleInfo(const TUserRole& userRole, const TDriveRoleHeader* roleInfo);

    NJson::TJsonValue BuildJsonReport() const;
};

class TDriveUserRoles {
public:
    R_FIELD(TString, UserId);
    R_FIELD(TInstant, Timestamp);

private:
    TVector<TUserRole> Roles;

public:
    const TVector<TUserRole>& GetRoles() const {
        return Roles;
    }

    bool AddRole(const TUserRole& role);
};

class TUserRolesHistoryManager: public TDatabaseHistoryManager<TUserRole> {
private:
    using TBase = TDatabaseHistoryManager<TUserRole>;

public:
    TUserRolesHistoryManager(const IHistoryContext& context)
        : TBase(context, "user_roles_history")
    {
    }

    TOptionalObjectEvents<TUserRole> GetEventsByUser(const TString& userId, NDrive::TEntitySession& session, TInstant since = TInstant::Zero()) const;
};

class TUserRolesDB
    : public IAutoActualization
    , private TBaseEntityManager<TUserRole>
{
public:
    using TExpectedUserRoles = TExpected<TDriveUserRoles, TCodedException>;
    using TOptionalUserRoles = TMaybe<TDriveUserRoles>;

public:
    TUserRolesDB(const IHistoryContext& context, TDuration defaultLifetime);

    const TUserRolesHistoryManager& GetHistoryManager() const {
        return HistoryManager;
    }

    TExpectedUserRoles GetCachedUserRoles(const TString& userId, TInstant statementDeadline = TInstant::Zero()) const;
    TOptionalUserRoles RestoreUserRoles(const TString& userId, NDrive::TEntitySession& session) const;

    bool GetUsersWithRoles(const TSet<TString>& roles, TVector<TString>& result, bool checkActive = false, ui64 limit = 0) const;
    bool GetUsersWithAtLeastOneRoles(const TSet<TString>& roles, TSet<TString>& result, bool checkActive) const;

    bool Link(const TUserRole& userRole, const TString& userActor, NDrive::TEntitySession& session) const;
    bool Unlink(const TString& userId, const TString& roleName, const TString& userActor, NDrive::TEntitySession& session) const;

protected:
    TString GetTableName() const override {
        return "user_roles";
    }
    bool GetStartFailIsProblem() const override {
        return false;
    }
    bool Refresh() override;

private:
    static TDriveUserRoles GetCachedUserRolesImpl(const TUserRolesDB& self, const TString& userId, TInstant statementDeadline);

private:
    TUserRolesHistoryManager HistoryManager;
    TDuration DefaultLifetime;

    TOptionalObjectEventId<TUserRole> LastEventId;

    mutable NUtil::TConcurrentCache<TString, TDriveUserRoles> ObjectCache;
};
