#include "process.h"

#include <drive/backend/data/additional_service.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/offers/offers/additional_service.h>

namespace NDrive {

    const TString TAdditionalServiceStartProcess::TypeName = "additional_service_start";

    TAdditionalServiceStartProcess::TFactory::TRegistrator<TAdditionalServiceStartProcess> TAdditionalServiceStartProcess::Registrator(TAdditionalServiceStartProcess::TypeName);

    TString TAdditionalServiceStartProcess::GetType() const {
        return TypeName;
    }

    TExpectedState TAdditionalServiceStartProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
        Y_UNUSED(state);
        const auto& server = context.GetServerAs<IServer>();
        const auto& driveAPI = *Yensured(server.GetDriveAPI());
        const auto& userTagManager = driveAPI.GetTagsManager().GetUserTags();
        TDBTags offerHolders;
        {
            auto tx = userTagManager.BuildTx<NSQL::ReadOnly>();
            auto optionalOfferHolders = userTagManager.RestoreTags(TVector<TString>{}, {TAdditionalServiceOfferHolderTag::Planned}, tx);
            if (!optionalOfferHolders) {
                return MakeUnexpected<TString>("cannot RestoreTags: " + tx.GetStringReport());
            }
            offerHolders = *optionalOfferHolders;
        }
        auto robotPermissions = server.GetDriveAPI()->GetUserPermissions(GetRobotUserId(), {});
        if (!robotPermissions) {
            return MakeUnexpected<TString>("cannot GetUserPermissions");
        }
        for (auto&& offerHolder : offerHolders) try {
            auto offerHolderTag = offerHolder.GetTagAs<TAdditionalServiceOfferHolderTag>();
            if (!offerHolderTag) {
                ALERT_LOG << GetRobotId() << ": cannot cast tag " << offerHolder.GetTagId() << " to TAdditionalServiceOfferHolderTag" << Endl;
                continue;
            }
            if (offerHolderTag->GetName() != TAdditionalServiceOfferHolderTag::Planned) {
                continue;
            }
            auto offer = offerHolderTag->GetOffer();
            if (!offer) {
                ALERT_LOG << GetRobotId() << ": cannot get offer from tag " << offerHolder.GetTagId() << Endl;
                continue;
            }
            auto serviceOffer = std::dynamic_pointer_cast<TAdditionalServiceOffer>(offer);
            if (!serviceOffer) {
                ALERT_LOG << GetRobotId() << ": cannot cast offer from tag " << offerHolder.GetTagId() << "to TAdditionalServiceOffer" << Endl;
                continue;
            }
            if (serviceOffer->GetScheduledAt() > Now()) {
                DEBUG_LOG << GetRobotId() << ": servising from tag " << offerHolder.GetTagId() << " not scheduled yet" << Endl;
                continue;
            }
            auto tx = userTagManager.BuildTx<NSQL::Writable | NSQL::RepeatableRead>();
            if (!TAdditionalServiceOfferHolderTag::Start(offerHolder, tx, *serviceOffer, *robotPermissions, server)) {
                NDrive::TEventLog::Log(GetRobotId(), NJson::TMapBuilder
                    ("message", "cannot start servicing")
                    ("tag_id", offerHolder.GetTagId())
                    ("error", tx.GetStringReport())
                );
                continue;
            }
            if (!tx.Commit()) {
                NDrive::TEventLog::Log(GetRobotId(), NJson::TMapBuilder
                    ("message", "cannot commit for tag")
                    ("tag_id", offerHolder.GetTagId())
                    ("error", tx.GetStringReport())
                );
                continue;
            }
        } catch (...) {
            NDrive::TEventLog::Log(GetRobotId(), NJson::TMapBuilder
                ("message", "cannot process tag")
                ("tag_id", offerHolder.GetTagId())
                ("exception", CurrentExceptionInfo(true).GetStringRobust())
            );
        }
        return MakeAtomicShared<IRTBackgroundProcessState>();
    }

}
