#include "config.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/user_devices/manager.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTAdjustTagsWatcher> TRTAdjustTagsWatcher::Registrator(TRTAdjustTagsWatcher::GetTypeName());

TExpectedState TRTAdjustTagsWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const auto& tagsManager = server.GetDriveAPI()->GetTagsManager();

    if (!server.GetUserDevicesManager()) {
        return MakeUnexpected<TString>("devices manager undefined");
    }

    ITagsMeta::TTagDescriptionsByName descriptionByName;
    TVector<TString> descriptionNames;
    {
        auto adjustTags = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TAdjustUserTag::TypeName);
        for (const auto& description : adjustTags) {
            descriptionByName[description->GetName()] = description;
            descriptionNames.push_back(description->GetName());
        }
    }

    TDBTags adjustTags;
    const auto& userTagManager = tagsManager.GetUserTags();
    {
        auto session = userTagManager.BuildSession(true);
        auto optionalTags = userTagManager.RestoreTags(TVector<TString>{}, descriptionNames, session);
        if (!optionalTags) {
            return MakeUnexpected("cannot RestoreTags: " + session.GetStringReport());
        }
        adjustTags = std::move(*optionalTags);
    }

    {
        auto session = tagsManager.GetUserTags().BuildSession(false);
        TVector<TDBTag> toRemove;
        TVector<TDBTag> update;
        for (auto&& commonTag : adjustTags) {
            TAdjustUserTag* tagData = dynamic_cast<TAdjustUserTag*>(commonTag.GetData().Get());
            auto itDescription = descriptionByName.find(commonTag->GetName());
            if (!tagData || itDescription == descriptionByName.end()) {
                continue;
            }

            const TAdjustUserTag::TDescription* adjustDesc = dynamic_cast<const TAdjustUserTag::TDescription*>(itDescription->second.Get());
            if (!adjustDesc) {
                continue;
            }

            if (auto res = server.GetUserDevicesManager()->RegisterEvent(IUserDevicesManager::EEventGlobalTypes::Undefined, adjustDesc->GetToken(), commonTag.GetObjectId(), tagData->GetTimestamp()); !res) {
                TUnistatSignalsCache::SignalAdd(GetRTProcessName(), ::ToString(res.GetError()), 1);
                if (res.GetError() == IUserDevicesManager::EEventResult::NoToken) {
                    commonTag->SetComment(::ToString(res.GetError()));
                    if (!tagsManager.GetUserTags().UpdateTagData(commonTag, GetRobotUserId(), session)) {
                        return MakeUnexpected<TString>("cannot UpdateTagData: " + session.GetStringReport());
                    }
                    toRemove.emplace_back(commonTag);
                }
                continue;
            }
            toRemove.emplace_back(commonTag);
        }
        if (!tagsManager.GetUserTags().RemoveTagsSimple(toRemove, GetRobotUserId(), session, false) || !session.Commit()) {
            return MakeUnexpected<TString>("cannot RemoveTags or commit: " + session.GetStringReport());
        }
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}
