#include "processor.h"

TAttachmentsExpirationWatcher::TFactory::TRegistrator<TAttachmentsExpirationWatcher> TAttachmentsExpirationWatcher::Registrator(TAttachmentsExpirationWatcher::GetTypeName());

TExpectedState TAttachmentsExpirationWatcher::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    try {
        TSimpleAttachmentWatcher watcher(server, WatcherConfig);
        if (context.GetFilteredCarIds() && !watcher.ActualizeTags(context.GetFilteredCarIds())) {
            return MakeUnexpected<TString>({});
        }
    } catch (const std::exception& e) {
        ERROR_LOG << "Watcher failed. Exception: " + FormatExc(e) << Endl;
        return MakeUnexpected<TString>({});
    }

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TAttachmentsExpirationWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSVariants>("attachment_type", "Тип устройства").InitVariants<EDocumentAttachmentType>();
    scheme.Add<TFSString>("field_name", "Смотреть на поле (как в Гараже)").SetRequired(true);

    scheme.Add<TFSArray>("intervals", "Интервалы слежки относительно момента 'сейчас'. Пары чисел (секунды), разделенные одиночным пробелом").SetElement<TFSString>();

    scheme.Add<TFSDuration>("tag_name", "То навесить тег").SetRequired(true);
    scheme.Add<TFSDuration>("tag_comment", "С комментарием").SetRequired(true);

    scheme.Add<TFSVariants>("notifier_name", "И отправить нотификацию в чат").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSBoolean>("dry_run", "Только нотификация");
    scheme.Add<TFSVariants>("missing_field_policy", "Политика отсутствующего поля").InitVariants<TAttachmentWatcherConfig::EMissingFieldPolicy>();

    return scheme;
}

NJson::TJsonValue TAttachmentsExpirationWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "attachment_type", ToString(WatcherConfig.GetAttachmentType()));
    TJsonProcessor::Write(result, "field_name", WatcherConfig.GetInstantFieldName());

    {
        NJson::TJsonValue intervals = NJson::JSON_ARRAY;
        for (auto&& interval : WatcherConfig.GetTriggerIntervals()) {
            intervals.AppendValue(interval.ToString());
        }
        TJsonProcessor::Write(result, "intervals", std::move(intervals));
    }

    TJsonProcessor::Write(result, "tag_name", WatcherConfig.GetTagName());
    TJsonProcessor::Write(result, "tag_comment", WatcherConfig.GetTagComment());

    TJsonProcessor::Write(result, "notifier_name", WatcherConfig.GetNotifierName());
    TJsonProcessor::Write(result, "dry_run", WatcherConfig.IsDryRun());
    TJsonProcessor::Write(result, "missing_field_policy", ToString(WatcherConfig.GetMissingFieldPolicy()));

    return result;
}

bool TAttachmentsExpirationWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    TString attachmentType;
    JREAD_STRING(jsonInfo, "attachment_type", attachmentType);
    if (!TryFromString(attachmentType, WatcherConfig.MutableAttachmentType())) {
        return false;
    }
    JREAD_STRING(jsonInfo, "field_name", WatcherConfig.MutableInstantFieldName());

    if (!jsonInfo.Has("intervals") || !jsonInfo["intervals"].IsArray()) {
        return false;
    }

    WatcherConfig.MutableTriggerIntervals().clear();
    for (auto&& intervalStr : jsonInfo["intervals"].GetArray()) {
        if (!intervalStr.IsString()) {
            return false;
        }
        TAttachmentWatcherConfig::TInterval interval;
        if (!interval.FromString(intervalStr.GetString())) {
            return false;
        }
        WatcherConfig.MutableTriggerIntervals().emplace_back(std::move(interval));
    }

    JREAD_STRING(jsonInfo, "tag_name", WatcherConfig.MutableTagName());
    JREAD_STRING(jsonInfo, "tag_comment", WatcherConfig.MutableTagComment());

    JREAD_STRING_OPT(jsonInfo, "notifier_name", WatcherConfig.MutableNotifierName());

    {
        TString missingFieldPolicyStr;
        JREAD_STRING_OPT(jsonInfo, "missing_field_policy", missingFieldPolicyStr);
        if (missingFieldPolicyStr && !TryFromString(missingFieldPolicyStr, WatcherConfig.MutableMissingFieldPolicy())) {
            return false;
        }
    }

    {
        bool isDryRun = false;
        JREAD_BOOL_OPT(jsonInfo, "dry_run", isDryRun);
        WatcherConfig.SetDryRun(isDryRun);
    }

    return true;
}
