#include "watcher.h"

#include <drive/backend/tags/tags_manager.h>

bool TSimpleAttachmentWatcher::ActualizeTags(const TSet<TString>& requestCarIds) const {
    const auto& assignmentManager = Server.GetDriveAPI()->GetCarAttachmentAssignments();
    auto carsFR = Server.GetDriveAPI()->GetCarsData()->GetCachedObjectsMap();
    const auto& tagsManager = Server.GetDriveAPI()->GetTagsManager().GetDeviceTags();
    auto allAssignments = assignmentManager.GetAssignmentsOfType(requestCarIds, Config.GetAttachmentType(), Now());

    TMap<TString, TDBTag> taggedCars;
    {
        TVector<TDBTag> dbTags;
        auto session = Server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!Server.GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags({}, {Config.GetTagName()}, dbTags, session)) {
            MaybeNotify("❌ Не могу восстановить теги под именем " + Config.GetTagName());
            return false;
        }
        for (auto&& tag : dbTags) {
            taggedCars.emplace(tag.GetObjectId(), tag);
        }
    }

    TSet<TString> carIds;
    for (auto&& assignmentIt : allAssignments) {
        carIds.insert(assignmentIt.first);
    }

    TVector<TString> idsForAdd;
    TVector<TString> idsForRemove;
    for (auto&& carId : carIds) {
        auto range = allAssignments.equal_range(carId);
        bool allAttachmentsMatch = true;
        size_t numIgnoreVotes = 0;
        size_t totalAttachments = 0;
        for (auto&& it = range.first; it != range.second; ++it) {
            totalAttachments += 1;
            if (!AttachmentMatchesCondition(it->second, numIgnoreVotes)) {
                allAttachmentsMatch = false;
                break;
            }
        }
        if (numIgnoreVotes == totalAttachments) {
            continue;
        }

        if (allAttachmentsMatch && !taggedCars.contains(carId)) {
            idsForAdd.push_back(carId);
        } else if (!allAttachmentsMatch && taggedCars.contains(carId)) {
            idsForRemove.push_back(carId);
        }
    }

    if (Config.GetNotifierName() && Server.GetNotifier(Config.GetNotifierName())) {
        ReportCars("✅ Добавляем тег " + Config.GetTagName() + " на следующие машины", idsForAdd, carsFR);
        ReportCars("❌ Снимаем тег " + Config.GetTagName() + " со следующих машин", idsForRemove, carsFR);
    }

    if (!Config.IsDryRun()) {
        // Add tags
        {
            auto session = Server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
            size_t tagsAddedWithinSession = 0;
            for (auto&& carId : idsForAdd) {
                auto tag = Server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(Config.GetTagName(), Config.GetTagComment());
                if (!tag || !tagsManager.AddTag(tag, "robot-expiration-watcher", carId, &Server, session)) {
                    MaybeNotify("Не могу добавить тег " + Config.GetTagName() + " на машину " + carId);
                    return false;
                }
                ++tagsAddedWithinSession;
                if (tagsAddedWithinSession == 200) {
                    if (!session.Commit()) {
                        MaybeNotify("Не могу закоммитить сессию с добавлением " + ToString(tagsAddedWithinSession) + " тегов " + Config.GetTagName());
                        return false;
                    }
                    tagsAddedWithinSession = 0;
                    session = Server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
                    Sleep(TDuration::MilliSeconds(300));
                }
            }
            if (tagsAddedWithinSession) {
                if (!session.Commit()) {
                    MaybeNotify("Не могу закоммитить сессию с добавлением " + ToString(tagsAddedWithinSession) + " тегов " + Config.GetTagName());
                    return false;
                }
            }
        }

        // Remove tags
        {
            TVector<TDBTag> tagsForRemove;
            for (auto&& carId : idsForRemove) {
                auto tc = taggedCars.find(carId);
                if (tc == taggedCars.end()) {
                    continue;
                }
                tagsForRemove.emplace_back(tc->second);
                if (tagsForRemove.size() == 200) {
                    auto session = Server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
                    if (!tagsManager.RemoveTagsSimple(tagsForRemove, "robot-expiration-watcher", session, true) || !session.Commit()) {
                        MaybeNotify("Не могу закоммитить сессию с удалением " + ToString(tagsForRemove.size()) + " тегов " + Config.GetTagName());
                        return false;
                    }
                    tagsForRemove.clear();
                    Sleep(TDuration::MilliSeconds(300));
                }
            }
            if (!tagsForRemove.empty()) {
                auto session = Server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
                if (!tagsManager.RemoveTagsSimple(tagsForRemove, "robot-expiration-watcher", session, true) || !session.Commit()) {
                    MaybeNotify("Не могу закоммитить сессию с удалением " + ToString(tagsForRemove.size()) + " тегов " + Config.GetTagName());
                    return false;
                }
            }
        }
    }

    return true;
}

void TSimpleAttachmentWatcher::ReportCars(const TString& commonHeader, const TVector<TString>& carIds, const TMap<TString, TDriveCarInfo>& cachedData) const {
    TVector<TString> reportLines;
    for (auto&& carId : carIds) {
        auto it = cachedData.find(carId);
        if (it == cachedData.end()) {
            reportLines.push_back(carId);
        } else {
            reportLines.push_back(it->second.GetHRReport());
        }
    }
    NDrive::INotifier::MultiLinesNotify(Server.GetNotifier(Config.GetNotifierName()), commonHeader, reportLines);
}

void TSimpleAttachmentWatcher::MaybeNotify(const TString& message) const {
    if (Config.GetNotifierName() && Server.GetNotifier(Config.GetNotifierName())) {
        NDrive::INotifier::Notify(Server.GetNotifier(Config.GetNotifierName()), message);
    }
}

bool TSimpleAttachmentWatcher::AttachmentMatchesCondition(const TCarGenericAttachment& attachment, size_t& numIgnoreVotes) const {
    auto blob = attachment->SerializeToJsonBlob();
    TInstant observedInstant;

    if (!blob.Has(Config.GetInstantFieldName())) {
        if (Config.GetMissingFieldPolicy() == TAttachmentWatcherConfig::EMissingFieldPolicy::PlusTenYears) {
            observedInstant = Now() + TDuration::Days(3653);
        } else if (Config.GetMissingFieldPolicy() == TAttachmentWatcherConfig::EMissingFieldPolicy::MinusTenYears) {
            observedInstant = Now() - TDuration::Days(3653);
        } else if (Config.GetMissingFieldPolicy() == TAttachmentWatcherConfig::EMissingFieldPolicy::Ignore) {
            numIgnoreVotes += 1;
            return true;
        }
    } else {
        NJson::TJsonValue fieldValue = blob[Config.GetInstantFieldName()];
        if (fieldValue.IsInteger()) {
            observedInstant = TInstant::Seconds(fieldValue.GetInteger());
        } else if (fieldValue.IsString()) {
            TString fieldStr = fieldValue.GetString();
            if (!TInstant::TryParseIso8601(fieldStr, observedInstant)) {
                ui64 timestamp;
                if (TryFromString(fieldStr, timestamp)) {
                    observedInstant = TInstant::Seconds(timestamp);
                } else {
                    return false;
                }
            }
        } else {
            return false;
        }
    }

    i64 effectiveNow = Now().Seconds();
    i64 observedInstantTS = observedInstant.Seconds();

    for (auto&& interval : Config.GetTriggerIntervals()) {
        i64 left = effectiveNow + interval.Start;
        i64 right = effectiveNow + interval.End;
        if (left <= observedInstantTS && observedInstantTS <= right) {
            return true;
        }
    }

    return false;
}
