#pragma once

#include <drive/backend/cars/hardware.h>
#include <drive/backend/database/drive_api.h>

#include <rtline/util/types/accessor.h>

#include <algorithm>

class TAttachmentWatcherConfig {
public:
    struct TInterval {
        i64 Start;
        i64 End;

        TInterval() = default;
        TInterval(const i64 start, const i64 end)
            : Start(start)
            , End(end)
        {
        }

        bool FromString(const TString& s) {
            auto tokens = SplitString(s, " ");
            if (tokens.size() != 2) {
                return false;
            }
            return TryFromString(tokens[0], Start) && TryFromString(tokens[1], End) && Start <= End;
        }

        TString ToString() const {
            return ::ToString(Start) + " " + ::ToString(End);
        }
    };

    void AddTriggerInterval(const i64 from, const i64 until) {
        TriggerIntervals.emplace_back(TInterval(from, until));
    }

    enum EMissingFieldPolicy {
        Ignore = 0 /* "ignore" */,
        PlusTenYears = 1 /* "plus_ten_years" */,
        MinusTenYears = 2 /* "minus_ten_years" */,
    };

    R_FIELD(TString, TagName);
    R_FIELD(TString, TagComment);
    R_FIELD(TString, NotifierName);
    R_FIELD(EDocumentAttachmentType, AttachmentType, EDocumentAttachmentType::CarRegistryDocument);
    R_FIELD(TString, InstantFieldName);
    R_FIELD(TVector<TInterval>, TriggerIntervals);
    R_FIELD(EMissingFieldPolicy, MissingFieldPolicy, EMissingFieldPolicy::Ignore);
    R_FIELD(bool, DryRun, false);
};

class TSimpleAttachmentWatcher {
public:
    TSimpleAttachmentWatcher(const NDrive::IServer& server, const TAttachmentWatcherConfig& config)
        : Server(server)
        , Config(config)
    {
    }

    bool ActualizeTags(const TSet<TString>& requestCarIds) const;

private:
    void MaybeNotify(const TString& message) const;
    void ReportCars(const TString& commonHeader, const TVector<TString>& carIds, const TMap<TString, TDriveCarInfo>& cachedData) const;
    bool AttachmentMatchesCondition(const TCarGenericAttachment& attachment, size_t& numIgnoreVotes) const;

    const NDrive::IServer& Server;
    TAttachmentWatcherConfig Config;
};
