#include "process.h"

#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/database/drive/url.h>

#include <drive/library/cpp/covid_pass/client.h>
#include <drive/library/cpp/threading/future_cast.h>

#include <rtline/library/json/adapters.h>

TExpectedState TPassCheckProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();

    const auto& deviceTagManager = server.GetDriveDatabase().GetTagsManager().GetDeviceTags();
    const auto& userTagManager = server.GetDriveDatabase().GetTagsManager().GetUserTags();

    const auto notifier = NotifierName ? server.GetNotifier(NotifierName) : nullptr;

    TSet<EPassApiType> types;
    try {
        TString apiSettings = server.GetSettings().GetValueDef<TString>("covid_pass.api_types", "moscow");
        StringSplitter(apiSettings).SplitBySet(", \n\r").SkipEmpty().ParseInto(&types);
    } catch(...) {
        ERROR_LOG << "Incorrect API settings for pass check" << Endl;
        types = {EPassApiType::Moscow};
    }

    auto sessionBuilder = deviceTagManager.GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    if (!sessionBuilder) {
        return MakeUnexpected<TString>("cannot GetSessionsBuilder");
    }

    TDBTags tags;
    {
        auto session = userTagManager.BuildSession(true);
        auto optionalTags = userTagManager.RestoreTags(TVector<TString>{}, { PassTagName }, session);
        if (!optionalTags) {
            return MakeUnexpected("cannot RestoreTags: " + session.GetStringReport());
        }
        tags = std::move(*optionalTags);
    }

    for (auto&& tag : tags) try {
        if (!tag) {
            continue;
        }

        auto userSessions = sessionBuilder->GetUserSessions(tag.GetObjectId());
        if (userSessions.empty()) {
            continue;
        }
        auto lastSession = userSessions.back();
        Y_ENSURE(lastSession);
        if (lastSession->GetClosed()) {
            DEBUG_LOG << GetRobotId() << ": " << tag.GetObjectId() << " last session is closed" << Endl;
            continue;
        }

        const auto& objectId = lastSession->GetObjectId();
        auto session = userTagManager.BuildSession();

        TMaybe<bool> validationPassed;
        {

            if (tag->GetName() != PassTagName) {
                continue;
            }
            INFO_LOG << GetRobotId() << ": " << tag.GetObjectId() << " " << tag.GetTagId() << Endl;

            auto passTag = tag.MutableTagAs<TUserIsolationPassTag>();
            if (!passTag->IsValid()) {
                continue;
            }
            if (passTag->GetCarId() != objectId) {
                continue;
            }

            auto validationTimestamp = passTag->GetValidationTime();
            auto now = Now();
            if (now < validationTimestamp + PassValidationPeriod) {
                INFO_LOG << GetRobotId() << ": " << tag.GetObjectId() << " skipping " << tag.GetTagId() << Endl;
                continue;
            }
            auto until = now;

            TMap<EPassApiType, NThreading::TFuture<TCovidPassClient::TPassData>> passes;
            for (auto&& type : types) {
                auto client = server.GetCovidPassClient(type);
                if (!client) {
                    ERROR_LOG << GetRobotId() << ": " << tag.GetObjectId() << " " << tag.GetTagId() << " cannot get CovidPassClient " << type << Endl;
                    continue;
                }
                passes.emplace(type, client->GetPassData(passTag->GetToken(), until));
            }
            bool valid = false;
            bool hasError = false;
            for (auto&& [type, asyncPassData] : passes) {
                asyncPassData.Wait();
                NDrive::TEventLog::Log("CovidPassClientResponse", NJson::TMapBuilder
                    ("user_id", tag.GetObjectId())
                    ("tag_id", tag.GetTagId())
                    ("pass", passTag->GetToken())
                    ("type", ToString(type))
                    ("response", NJson::ToJson(asyncPassData))
                );
                if (!asyncPassData.HasValue()) {
                    ERROR_LOG
                        << GetRobotId() << ": " << tag.GetObjectId() << " " << tag.GetTagId()
                        << " CovidPassClient " << type << " exception: " << NThreading::GetExceptionMessage(asyncPassData) << Endl;
                    hasError = true;
                    continue;
                }
                const auto& passData = asyncPassData.GetValueSync();
                if (passData.GetIsValid()) {
                    valid = true;
                    break;
                }
            }
            if (hasError && !valid) {
                continue;
            }
            passTag->SetValid(valid);
            passTag->SetValidationTime(now);
            if (validationPassed) {
                validationPassed = *validationPassed || valid;
            } else {
                validationPassed = valid;
            }

            Y_ENSURE(userTagManager.UpdateTagData(tag, GetRobotUserId(), session), "cannot UpdateTagData " << tag.GetTagId() << ": " << session.GetStringReport());
            INFO_LOG << GetRobotId() << ": " << tag.GetObjectId() << " UpdateTagData " << tag.GetTagId() << Endl;
        }
        if (validationPassed && !*validationPassed) {
            ERROR_LOG << GetRobotId() << ": " << tag.GetObjectId() << " is illegal" << Endl;
            if (notifier) {
                auto message = TStringBuilder()
                    << "Пользователь " << TCarsharingUrl().ClientInfo(tag.GetObjectId()) << " катается без пропуска\n"
                    << "Поездка: " << TCarsharingUrl().SessionPage(lastSession->GetSessionId());
                auto notificationResult = notifier->Notify(message);
                if (!notificationResult || notificationResult->HasErrors()) {
                    ERROR_LOG << GetRobotId() << ": " << tag.GetObjectId() << " cannot notify: " << (notificationResult ? notificationResult->GetErrorMessage() : "null") << Endl;
                }
            }
            if (PushTagName) {
                auto mark = server.GetDriveDatabase().GetTagsManager().GetTagsMeta().CreateTag(PushTagName);
                Y_ENSURE(mark, "cannot create tag " << PushTagName);
                Y_ENSURE(userTagManager.AddTag(mark, GetRobotUserId(), tag.GetObjectId(), &server, session), "cannot add tag " << PushTagName << ": " << session.GetStringReport());
            }
            if (PassCheckLandingId) {
                //Y_ENSURE(server.GetDriveAPI()->SendLandingViaChat(&server, tag.GetObjectId(), PassCheckLandingId, GetRobotUserId()));
            }
            if (PassCheckChatName) {
                auto chatRobot = server.GetChatRobot(PassCheckChatName);
                Y_ENSURE(chatRobot, "cannot acquire chat robot " << PassCheckChatName);
                auto topic = lastSession->GetSessionId() + "-reask-" + ToString(Seconds());
                chatRobot->EnsureChat(tag.GetObjectId(), topic, session, true);
            }
        } else {
            INFO_LOG << GetRobotId() << ": " << tag.GetObjectId() << " is ok" << Endl;
        }
        Y_ENSURE(session.Commit(), "cannot Commit: " << session.GetStringReport());
    } catch (const std::exception& e) {
        ERROR_LOG << GetRobotId() << ": " << tag.GetObjectId() << " triggered an exception: " << FormatExc(e) << Endl;
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TPassCheckProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSVariants>("notifier").SetVariants(server.GetNotifierNames());
    result.Add<TFSString>("pass_check_chat_name").SetDefault(PassCheckChatName);
    result.Add<TFSString>("pass_check_landing_id").SetDefault(PassCheckLandingId);
    result.Add<TFSString>("pass_tag").SetDefault(PassTagName);
    result.Add<TFSString>("push_tag").SetDefault(PushTagName);
    result.Add<TFSDuration>("pass_validation_period").SetDefault(PassValidationPeriod);
    return result;
}

bool TPassCheckProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    return TBase::DoDeserializeFromJson(value) &&
        NJson::ParseField(value["notifier"], NotifierName) &&
        NJson::ParseField(value["pass_check_chat_name"], PassCheckChatName) &&
        NJson::ParseField(value["pass_check_landing_id"], PassCheckLandingId) &&
        NJson::ParseField(value["pass_tag"], PassTagName) &&
        NJson::ParseField(value["push_tag"], PushTagName) &&
        NJson::ParseField(value["pass_validation_period"], PassValidationPeriod);
}

NJson::TJsonValue TPassCheckProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["notifier"] = NJson::ToJson(NJson::Nullable(NotifierName));
    result["pass_check_chat_name"] = NJson::ToJson(NJson::Nullable(PassCheckChatName));
    result["pass_check_landing_id"] = NJson::ToJson(NJson::Nullable(PassCheckLandingId));
    result["pass_tag"] = NJson::ToJson(NJson::Nullable(PassTagName));
    result["push_tag"] = NJson::ToJson(NJson::Nullable(PushTagName));
    result["pass_validation_period"] = NJson::ToJson(PassValidationPeriod);
    return result;
}

TExpectedState TPassCheckStatusProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const auto& server = context.GetServerAs<NDrive::IServer>();

    const auto& deviceTagManager = server.GetDriveDatabase().GetTagsManager().GetDeviceTags();
    const auto& userTagManager = server.GetDriveDatabase().GetTagsManager().GetUserTags();

    auto sessionBuilder = deviceTagManager.GetHistoryManager().GetSessionsBuilder("billing", TInstant::Zero());
    if (!sessionBuilder) {
        ERROR_LOG << GetRobotId() << ": cannot billing session builder" << Endl;
        return MakeUnexpected<TString>({});
    }

    TDBTags tags;
    {
        auto session = userTagManager.BuildSession(true);
        auto optionalTags = userTagManager.RestoreTags(TVector<TString>{}, { PassTagName }, session);
        if (!optionalTags) {
            return MakeUnexpected("cannot RestoreTags: " + session.GetStringReport());
        }
        tags = std::move(*optionalTags);
    }

    auto addTag = [&, this](const TString& userId) {
        auto session = userTagManager.BuildSession();
        TVector<TDBTag> tags;
        Y_ENSURE(userTagManager.RestoreTags({ userId }, { InvalidPassTagName }, tags, session), "cannot restore tag " << InvalidPassTagName << ": " << session.GetStringReport());
        if (tags.empty()) {
            auto mark = server.GetDriveDatabase().GetTagsManager().GetTagsMeta().CreateTag(InvalidPassTagName);
            Y_ENSURE(mark, "cannot create tag " << InvalidPassTagName);
            Y_ENSURE(userTagManager.AddTag(mark, GetRobotUserId(), userId, &server, session), "cannot add tag " << InvalidPassTagName << ": " << session.GetStringReport());
        }
        Y_ENSURE(session.Commit(), "cannot Commit: " << session.GetStringReport());
        INFO_LOG << GetRobotId() << ": added tag " << InvalidPassTagName << " for " << userId << Endl;
    };

    auto removeTag = [&, this](const TString& userId) {
        auto session = userTagManager.BuildSession();
        TVector<TDBTag> tags;
        Y_ENSURE(userTagManager.RestoreTags({ userId }, { InvalidPassTagName }, tags, session), "cannot restore tag " << InvalidPassTagName << ": " << session.GetStringReport());
        for (auto&& tag : tags) {
            Y_ENSURE(userTagManager.RemoveTag(tag, GetRobotUserId(), &server, session), "cannot remove tag " << tag.GetTagId() << ": " << session.GetStringReport());
        }
        Y_ENSURE(session.Commit(), "cannot Commit: " << session.GetStringReport());
        INFO_LOG << GetRobotId() << ": removed " << tags.size() << " tag(s) " << InvalidPassTagName << " for " << userId << Endl;
    };

    for (auto&& tag : tags) try {
        auto userSessions = sessionBuilder->GetUserSessions(tag.GetObjectId());
        if (userSessions.empty()) {
            continue;
        }
        auto lastSession = userSessions.back();
        Y_ENSURE(lastSession);
        if (lastSession->GetClosed()) {
            DEBUG_LOG << GetRobotId() << ": " << tag.GetObjectId() << " last session is closed" << Endl;
            removeTag(tag.GetObjectId());
            continue;
        }

        const auto& objectId = lastSession->GetObjectId();

        bool valid = false;
        {
            if (!tag) {
                continue;
            }
            if (tag->GetName() != PassTagName) {
                continue;
            }

            auto passTag = tag.MutableTagAs<TUserIsolationPassTag>();
            if (passTag->GetCarId() != objectId) {
                continue;
            }
            valid |= passTag->IsValid();
            if (valid) {
                INFO_LOG << GetRobotId() << ": " << tag.GetObjectId() << " valid by " << tag.GetTagId() << Endl;
            }
        }
        if (valid) {
            removeTag(tag.GetObjectId());
        } else {
            addTag(tag.GetObjectId());
        }
    } catch (const std::exception& e) {
        ERROR_LOG << GetRobotId() << ": " << tag.GetObjectId() << " triggered an exception: " << FormatExc(e) << Endl;
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TPassCheckStatusProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSString>("invalid_pass_tag").SetDefault(InvalidPassTagName);
    result.Add<TFSString>("pass_tag").SetDefault(PassTagName);
    return result;
}

bool TPassCheckStatusProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    return TBase::DoDeserializeFromJson(value) &&
        NJson::ParseField(value["invalid_pass_tag"], InvalidPassTagName) &&
        NJson::ParseField(value["pass_tag"], PassTagName);
}

NJson::TJsonValue TPassCheckStatusProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["invalid_pass_tag"] = NJson::ToJson(NJson::Nullable(InvalidPassTagName));
    result["pass_tag"] = NJson::ToJson(NJson::Nullable(PassTagName));
    return result;
}

TPassCheckProcess::TFactory::TRegistrator<TPassCheckProcess> TPassCheckProcess::Registrator(TPassCheckProcess::GetTypeName());
TPassCheckStatusProcess::TFactory::TRegistrator<TPassCheckStatusProcess> TPassCheckStatusProcess(TPassCheckStatusProcess::GetTypeName());
