#include "accounts_notify.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/library/cpp/balance/client.h>

#include <drive/library/cpp/balance/client.h>

TMaybe<TVector<TString>> GetUsers(const NDrive::IServer& server, const TSet<TString>& emails, const TString& operatorUserId, NDrive::TEntitySession& session) {
    TVector<TString> result;
    for (const auto& email : emails) {
        TString userId;
        NDrive::TExternalUser externalUser;
        externalUser.SetEmail(email);
        auto user = server.GetDriveAPI()->GetUsersData()->FindOrRegisterExternal(operatorUserId, session, externalUser);
        if (!user) {
            ERROR_LOG << "Cannot select users : " << session.GetStringReport() << Endl;
            return {};
        }
        result.emplace_back(user->GetUserId());
    }
    return result;
}


NDrive::TScheme TRTLimitedAccountsNotifier::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("account_types", "Типы кошелька").SetRequired(true);
    scheme.Add<TFSString>("mail_tag", "Тип тега для рассылки").SetRequired(true);
    scheme.Add<TFSString>("template_balance_param", "Параметр для отображения баланса в шаблоне").SetDefault("");
    scheme.Add<TFSNumeric>("max_account_count", "Максимальное количество кошельков в обработке за раз").SetDefault(1000);
    scheme.Add<TFSDuration>("notify_period", "Минимальный период отправки баланса кошелька").SetDefault(TDuration::Days(1));
    return scheme;
}

NJson::TJsonValue TRTLimitedAccountsNotifier::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    JWRITE(result, "account_types", JoinSeq(",", AccountTypes));
    JWRITE(result, "mail_tag", MailTag);
    JWRITE(result, "template_balance_param", TemplateBalanceParam);
    JWRITE(result, "max_account_count", MaxAccountsCount);
    JWRITE_DURATION(result, "notify_period", NotifyPeriod);
    return result;
}

bool TRTLimitedAccountsNotifier::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    TString accountTypes;
    JREAD_STRING(jsonInfo, "account_types", accountTypes);
    StringSplitter(accountTypes).SplitBySet(", ").SkipEmpty().Collect(&AccountTypes);
    JREAD_STRING(jsonInfo, "mail_tag", MailTag);
    JREAD_STRING_OPT(jsonInfo, "template_balance_param", TemplateBalanceParam);
    JREAD_UINT_OPT(jsonInfo, "max_account_count", MaxAccountsCount);
    JREAD_DURATION_OPT(jsonInfo, "notify_period", NotifyPeriod);
    return true;
}

TExpectedState TRTLimitedAccountsNotifier::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();

    if (!server.GetBalanceClient()) {
        ERROR_LOG << "Undefined balance client" << Endl;
        return nullptr;
    }

    TVector<NDrive::NBilling::IBillingAccount::TPtr> accounts;
    for (const auto& accountType : AccountTypes) {
        auto accountsSelect = server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetAccountsByName(accountType);
        if (!accountsSelect) {
            return MakeUnexpected<TString>(GetRobotId() + ": Cannot take account descriptions. '" + accountType + "'");
        }
        accounts.insert(accounts.end(), accountsSelect->begin(), accountsSelect->end());
    }
    TVector<NDrive::NBilling::IBillingAccount::TPtr> relevantAccounts;
    TMap<ui32, TSet<TString>> accountEmails;
    {
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        for (const auto& account : accounts) {
            if (relevantAccounts.size() >= MaxAccountsCount) {
                break;
            }
            const NDrive::NBilling::TLimitedAccount* impl = account->GetAsSafe<NDrive::NBilling::TLimitedAccount>();
            if (!impl || !impl->IsActive() || impl->IsPersonal()) {
                continue;
            }
            const NDrive::NBilling::TLimitedAccountRecord* record = impl->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
            if (!record
                || !record->HasNotifySettings()
                || (!record->HasBalanceInfo() && record->GetNotifySettingsRef().GetEmails().empty()))
            {
                continue;
            }
            auto relevance = CheckRelevance(server.GetDriveAPI()->GetBillingManager().GetAccountsManager(), *record, session);
            if (!relevance) {
                return MakeUnexpected<TString>(GetRobotId() + ": Fail on CheckRelevance. " + session.GetStringReport());
            }
            if (!*relevance) {
                continue;
            }
            relevantAccounts.push_back(account);
            if (record->GetNotifySettingsRef().GetEmails()) {
                accountEmails[account->GetId()].insert(record->GetNotifySettingsRef().GetEmails().begin(), record->GetNotifySettingsRef().GetEmails().end());
            }
        }
    }

    for (const auto& account : relevantAccounts) {
        const NDrive::NBilling::TLimitedAccountRecord* record = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
        if (record && record->HasBalanceInfo()) {
            auto person = server.GetBalanceClient()->GetPerson(record->GetBalanceInfoRef().GetPersonId());
            if (!person) {
                NDrive::TEventLog::Log(GetRobotId(), NJson::TMapBuilder
                    ("account_id", account->GetId())
                    ("person_id", record->GetBalanceInfoRef().GetPersonId())
                    ("code", person.GetError().GetCode())
                    ("error", person.GetError().GetFullError())
                );
                continue;
            }
            if (person->HasEmail()) {
                accountEmails[account->GetId()].insert(person->GetEmailRef());
            }
        }
    }

    {
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
        for (const auto& account : relevantAccounts) {
            auto emailsIt = accountEmails.find(account->GetId());
            if (emailsIt == accountEmails.end() || emailsIt->second.empty()) {
                NDrive::TEventLog::Log(GetRobotId(), NJson::TMapBuilder
                    ("account_id", account->GetId())
                    ("error", "no emails")
                );
                continue;
            }
            const NDrive::NBilling::TLimitedAccount* impl = account->GetAsSafe<NDrive::NBilling::TLimitedAccount>();
            const NDrive::NBilling::TLimitedAccountRecord* record = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
            if (!impl || !record || !record->HasNotifySettings()) {
                continue;
            }
            auto tag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(MailTag);
            auto mailTag = std::dynamic_pointer_cast<TUserMailTag>(tag);
            if (!mailTag) {
                ERROR_LOG << GetRobotId() << ": Incorrect notification tag name. '" << MailTag << "'" << Endl;
                return nullptr;
            }
            if (!TemplateBalanceParam.empty()) {
                mailTag->MutableTemplateArgs().emplace(TemplateBalanceParam, server.GetLocalization()->FormatPrice(ELocalization::Rus, impl->GetBalance()));
            }

            auto userIds = GetUsers(server, emailsIt->second, GetRobotUserId(), session);
            if (!userIds) {
                return MakeUnexpected<TString>(GetRobotId() + ": Cannot fetch users. " + session.GetStringReport());
            }

            for (const auto& userId : *userIds) {
                if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, GetRobotUserId(), userId, &server, session)) {
                    return MakeUnexpected<TString>(GetRobotId() + ": Cannot add tag. " + session.GetStringReport());
                }
            }
            if (!account->PatchAccountData(NJson::TMapBuilder(GetLastNotificationKey(record->GetNotifySettingsRef()), StartInstant.Seconds()), GetRobotUserId(), session)) {
                return MakeUnexpected<TString>(GetRobotId() + ": Cannot patch account data. " + session.GetStringReport());
            }
        }
        if (!session.Commit()) {
            return MakeUnexpected<TString>(GetRobotId() + ": Commit fails. " + session.GetStringReport());
        }
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTDailyLimitedAccountsBalanceNotifier> TRTDailyLimitedAccountsBalanceNotifier::Registrator(TRTDailyLimitedAccountsBalanceNotifier::GetTypeName());

TMaybe<bool> TRTDailyLimitedAccountsBalanceNotifier::CheckRelevance(const NDrive::NBilling::TAccountsManager& /*manager*/, const NDrive::NBilling::TLimitedAccountRecord& record, NDrive::TEntitySession& /*session*/) const {
    return record.HasNotifySettings()
        && record.GetNotifySettingsRef().GetDailyBalanceNotify().GetActive()
        && record.GetNotifySettingsRef().GetDailyBalanceNotify().GetLastNotify() < StartInstant - GetNotifyPeriod();
}

TString TRTDailyLimitedAccountsBalanceNotifier::GetLastNotificationKey(const NDrive::NBilling::TNotifySettings& settings) const {
    return settings.GetDailyBalanceNotify().GetLastNotifyKey();
}

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTLimitedAccountsLowBalanceNotifier> TRTLimitedAccountsLowBalanceNotifier::Registrator(TRTLimitedAccountsLowBalanceNotifier::GetTypeName());

i64 GetBalance(const NDrive::NBilling::TLimitedAccountRecord& record) {
    return record.GetSoftLimit() - record.GetExpenditure();
}

TMaybe<bool> TRTLimitedAccountsLowBalanceNotifier::CheckRelevance(const NDrive::NBilling::TAccountsManager& manager, const NDrive::NBilling::TLimitedAccountRecord& record, NDrive::TEntitySession& session) const {
    if (!record.HasNotifySettings()) {
        return false;
    }
    const auto& notifySetting = record.GetNotifySettingsRef().GetLowBalanceNotify();
    if (!notifySetting.GetActive() || notifySetting.GetLastNotify() > StartInstant - GetNotifyPeriod() || notifySetting.GetLimit() < GetBalance(record)) {
        return false;
    }
    if (notifySetting.GetLastNotify() == TInstant::Zero()) {
        return true;
    }
    auto events = manager.GetAccountsHistory().GetEventsByAccountId(record.GetAccountId(), notifySetting.GetLastNotify(), HistoryRefresher, session);
    if (!events) {
        return {};
    }
    const auto lastSoftLimit = record.GetSoftLimit();
    for (const auto& ev : *events) {
        auto eventRecord = std::dynamic_pointer_cast<NDrive::NBilling::TLimitedAccountRecord>(ev.GetRecord());
        if (!eventRecord) {
            continue;
        }
        const auto eventSoftLimit = eventRecord->GetSoftLimit();
        if (lastSoftLimit > eventSoftLimit) {
            return true;
        }
    }
    return false;
}

TString TRTLimitedAccountsLowBalanceNotifier::GetLastNotificationKey(const NDrive::NBilling::TNotifySettings& settings) const {
    return settings.GetLowBalanceNotify().GetLastNotifyKey();
}

NDrive::TScheme TRTLimitedAccountsLowBalanceNotifier::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("history_refresher", "Идентификатор робота, обновляющего балансы кошельков");
    return scheme;
}

NJson::TJsonValue TRTLimitedAccountsLowBalanceNotifier::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    if (HasHistoryRefresher()) {
        JWRITE(result, "history_refresher", GetHistoryRefresherUnsafe());
    }
    return result;
}

bool TRTLimitedAccountsLowBalanceNotifier::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    if (jsonInfo.Has("history_refresher")) {
        HistoryRefresher = jsonInfo["history_refresher"].GetStringRobust();
    }
    return true;
}

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTLimitedAccountsBalanceChangeNotifier> TRTLimitedAccountsBalanceChangeNotifier::Registrator(TRTLimitedAccountsBalanceChangeNotifier::GetTypeName());

TMaybe<bool> TRTLimitedAccountsBalanceChangeNotifier::CheckRelevance(const NDrive::NBilling::TAccountsManager& manager, const NDrive::NBilling::TLimitedAccountRecord& record, NDrive::TEntitySession& session) const {
    if (!record.HasNotifySettings()) {
        return false;
    }
    const auto& notifySetting = record.GetNotifySettingsRef().GetBalanceChangeNotify();
    if (!notifySetting.GetActive()) {
        return false;
    }
    const auto since = Max(StartInstant - CheckPeriod, notifySetting.GetLastNotify());
    auto events = manager.GetAccountsHistory().GetEventsByAccountId(record.GetAccountId(), since, {}, session);
    if (!events) {
        return {};
    }
    auto oldEvents = manager.GetAccountsHistory().GetEventsByAccountId(record.GetAccountId(), { TInstant::Zero(), since }, {}, session, 1);
    if (!oldEvents) {
        return {};
    }
    if (!oldEvents->empty()) {
        events->push_back(oldEvents->front());
    }
    if (events->empty()) {
        return false;
    }
    auto maxBalanceEvent = MaxElementBy(*events, [](const auto& entry) {
        auto eventRecord = std::dynamic_pointer_cast<NDrive::NBilling::TLimitedAccountRecord>(entry.GetRecord());
        if (!eventRecord) {
            return Min<i64>();
        }
        return GetBalance(*eventRecord);
    });
    auto eventRecord = std::dynamic_pointer_cast<NDrive::NBilling::TLimitedAccountRecord>(maxBalanceEvent->GetRecord());
    if (!eventRecord) {
        return false;
    }
    const auto oldBalance = GetBalance(*eventRecord);
    const auto diff = oldBalance - GetBalance(record);
    if (diff <= 0) {
        return false;
    }
    const i64 limit = notifySetting.GetLimit() > 0 ? notifySetting.GetLimit() : PercentLimit;
    return diff > oldBalance * limit / 100;
}

TString TRTLimitedAccountsBalanceChangeNotifier::GetLastNotificationKey(const NDrive::NBilling::TNotifySettings& settings) const {
    return settings.GetBalanceChangeNotify().GetLastNotifyKey();
}

NDrive::TScheme TRTLimitedAccountsBalanceChangeNotifier::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("percent_limit", "Лимит изменения в процентах").SetDefault(20).SetRequired(true);
    scheme.Add<TFSDuration>("check_period", "Проверяемый период изменений").SetDefault(TDuration::Days(1)).SetRequired(true);
    scheme.Remove("notify_period");
    return scheme;
}

NJson::TJsonValue TRTLimitedAccountsBalanceChangeNotifier::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    JWRITE(result, "percent_limit", PercentLimit);
    JWRITE_DURATION(result, "check_period", CheckPeriod);
    return result;
}

bool TRTLimitedAccountsBalanceChangeNotifier::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_UINT(jsonInfo, "percent_limit", PercentLimit);
    JREAD_DURATION(jsonInfo, "check_period", CheckPeriod);
    return true;
}
