#include "accounts_refresh.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/billing/accounts/bonus.h>
#include <drive/backend/billing/accounts/limited.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/account_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/library/cpp/balance/client.h>

#include <util/string/join.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTBillingAccountsWatcher> TRTBillingAccountsWatcher::Registrator(TRTBillingAccountsWatcher::GetTypeName());
IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTBillingAccountsCleaner> TRTBillingAccountsCleaner::Registrator(TRTBillingAccountsCleaner::GetTypeName());
IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTBillingB2BBalanceRefreshWatcher> TRTBillingB2BBalanceRefreshWatcher::Registrator(TRTBillingB2BBalanceRefreshWatcher::GetTypeName());
IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTBillingUpdateUndefinedClients> TRTBillingUpdateUndefinedClients::Registrator(TRTBillingUpdateUndefinedClients::GetTypeName());

TExpectedState TRTBillingAccountsWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const NDrive::NBilling::TAccountsManager& accountsManager = server.GetDriveAPI()->GetBillingManager().GetAccountsManager();

    TVector<TString> refreshNames;
    if (!AccountFilter) {
        for (auto&& account : accountsManager.GetRegisteredAccounts(StartInstant)) {
            if (account.GetRefreshPolicy() != NDrive::NBilling::TRefreshSchedule::EPolicy::None) {
                refreshNames.emplace_back(account.GetName());
            }
        }
    } else {
        auto accounts = accountsManager.GetAccountsChildren(AccountFilter, StartInstant);
        if (!accounts) {
            return MakeUnexpected<TString>("Cannot fetch accounts");
        }
        refreshNames = MakeVector(NContainer::Keys(*accounts));
    }

    const TString robotUserId = GetRobotUserId();
    for (auto&& aName : refreshNames) {
        auto session = server.GetDriveAPI()->GetBillingManager().BuildSession(false);
        auto accounts = accountsManager.GetAccountsByName(aName);
        if (!accounts) {
            return MakeUnexpected<TString>("Cannot fetch accounts");
        }
        ui32 processed = 0;
        for (auto&& account : *accounts) {
            switch (account->Refresh(robotUserId, StartInstant, session)) {
            case ERefreshStatus::Error:
                return MakeUnexpected<TString>(session.GetStringReport());
            case ERefreshStatus::Skip:
                continue;
            case ERefreshStatus::Ok:
                ++processed;
                if (processed == ObjectsAtOnce) {
                    if (!session.Commit()) {
                        return MakeUnexpected<TString>(session.GetStringReport());
                    }
                    TUnistatSignalsCache::SignalAdd("accounts_refresh-" + GetRTProcessName(), "refresh", processed);
                    processed = 0;
                    session = server.GetDriveAPI()->GetBillingManager().BuildSession(false);
                }
                break;
            default:
                break;
            }
        }
        if (processed) {
            if (!session.Commit()) {
                return MakeUnexpected<TString>(session.GetStringReport());
            }
            TUnistatSignalsCache::SignalAdd("accounts_refresh-" + GetRTProcessName(), "refresh", processed);
        }
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TRTBillingAccountsWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("at_once_count", "Максимальное число объектов в пакете");
    scheme.Add<TFSArray>("parent_ids", "Id родительских кошельков").SetElement<TFSString>();
    const NDrive::IServer* serverImpl = VerifyDynamicCast<const NDrive::IServer*>(&server);
    scheme.Add<TFSVariants>("accounts", "Кошельки").SetVariants(serverImpl->GetDriveAPI()->GetBillingManager().GetKnownAccounts()).SetMultiSelect(true);
    return scheme;
}

bool TRTBillingAccountsWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_UINT_OPT(jsonInfo, "at_once_count", ObjectsAtOnce);
    TJsonProcessor::ReadContainer(jsonInfo, "parent_ids", AccountFilter.ParentIds);
    TJsonProcessor::ReadContainer(jsonInfo, "accounts", AccountFilter.Accounts);
    AccountFilter.WithParent = true;
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTBillingAccountsWatcher::DoSerializeToJson() const {
    auto jsonInfo = TBase::DoSerializeToJson();
    JWRITE(jsonInfo, "at_once_count", ObjectsAtOnce);
    TJsonProcessor::WriteContainerArray(jsonInfo, "parent_ids", AccountFilter.ParentIds);
    TJsonProcessor::WriteContainerArray(jsonInfo, "accounts", AccountFilter.Accounts);
    return jsonInfo;
}

TExpectedState TRTBillingAccountsCleaner::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const NDrive::NBilling::TAccountsManager& accountsManager = server.GetDriveAPI()->GetBillingManager().GetAccountsManager();
    const NDrive::NBilling::TAccountsLinksDB& linksManager = accountsManager.GetLinksManager();

    auto registeredAccounts = accountsManager.GetRegisteredAccounts(StartInstant);
    TVector<TString> candidates;
    TSet<ui32> typeIds;
    for (auto&& description : registeredAccounts) {
        if (Accounts.contains(description.GetName()) && description.GetIsPersonal()) {
            candidates.emplace_back(description.GetName());
            typeIds.emplace(description.GetId());
        }
    }

    TVector<ui64> toRemove;
    toRemove.reserve(GetObjectsAtOnce());

    for (auto&& accName : candidates) {
        if (toRemove.size() == GetObjectsAtOnce()) {
            break;
        }
        auto accounts = accountsManager.GetAccountsByName(accName);
        if (!accounts) {
            return MakeUnexpected<TString>("Cannot fetch accounts");
        }

        TSet<ui64> accountIds;
        Transform(accounts->begin(), accounts->end(), std::inserter(accountIds, accountIds.begin()), [](const NDrive::NBilling::IBillingAccount::TPtr& acc){ return acc->GetId(); });

        auto session = server.GetDriveAPI()->GetBillingManager().BuildSession(true);
        auto notUsedAccounts = linksManager.FilterAccounts(accountIds, false, session);
        if (!notUsedAccounts) {
            return MakeUnexpected(session.GetStringReport());
        }
        TSet<ui64> accountsWithoutLink = std::move(*notUsedAccounts);

        auto linksHistory = linksManager.GetHistoryManager().GetEventsByAccounts(accountsWithoutLink, StartInstant - ActivityDuration, session);
        if (!linksHistory) {
            return MakeUnexpected(session.GetStringReport());
        }

        for (auto&& link : *linksHistory) {
            accountsWithoutLink.erase(link.GetAccountId());
        }

        for (auto&& account : accountsWithoutLink) {
            toRemove.push_back(account);
            if (toRemove.size() == GetObjectsAtOnce()) {
                break;
            }
        }
    }

    auto session = server.GetDriveAPI()->GetBillingManager().BuildSession(false);
    for (auto&& accountId : toRemove) {
        if (!accountsManager.RemoveAccount(accountId, GetRobotUserId(), session)) {
            ERROR_LOG << "Errors removing account " << accountId << Endl;
            return MakeUnexpected(session.GetStringReport());
        }
    }

    if (!session.Commit()) {
        return MakeUnexpected(session.GetStringReport());
    }
    TUnistatSignalsCache::SignalAdd("accounts_remove-" + GetRTProcessName(), "aName", toRemove.size());
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TRTBillingAccountsCleaner::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("at_once_count", "Максимальное число объектов в пакете");
    scheme.Add<TFSNumeric>("activity_duration", "Период отсутствия активности");

    const NDrive::IServer* serverImpl = VerifyDynamicCast<const NDrive::IServer*>(&server);
    scheme.Add<TFSVariants>("accounts", "Кошельки").SetVariants(serverImpl->GetDriveAPI()->GetBillingManager().GetKnownAccounts()).SetMultiSelect(true);
    return scheme;
}

bool TRTBillingAccountsCleaner::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_DURATION(jsonInfo, "activity_duration", ActivityDuration);
    JREAD_INT(jsonInfo, "at_once_count", ObjectsAtOnce);
    TJsonProcessor::ReadContainer(jsonInfo, "accounts", Accounts);
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTBillingAccountsCleaner::DoSerializeToJson() const {
    auto jsonInfo = TBase::DoSerializeToJson();
    JWRITE_DURATION(jsonInfo, "activity_duration", ActivityDuration);
    JWRITE(jsonInfo, "at_once_count", ObjectsAtOnce);
    TJsonProcessor::WriteContainerArray(jsonInfo, "accounts", Accounts);
    return jsonInfo;
}

TExpectedState TRTBillingB2BBalanceRefreshWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "update", 0);
    TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "sum", 0);

    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    if (!server.GetBalanceClient()) {
        ERROR_LOG << "Incorrect balance client" << Endl;
        return nullptr;
    }

    const NDrive::NBilling::TAccountsManager& accountsManager = server.GetDriveAPI()->GetBillingManager().GetAccountsManager();
    const auto& tagsManager = server.GetDriveDatabase().GetTagsManager();
    const auto& accountsTags = tagsManager.GetAccountTags();
    const auto& tagsMeta = tagsManager.GetTagsMeta();

    auto registeredAccounts = accountsManager.GetRegisteredAccounts(StartInstant);
    TVector<TString> candidates;
    TSet<ui32> typeIds;
    for (auto&& description : registeredAccounts) {
        if (!description.GetIsPersonal() && description.GetType() == NDrive::NBilling::EAccount::Wallet) {
            candidates.emplace_back(description.GetName());
        }
    }

    TMap<ui64, NDrive::NBilling::IBillingAccount::TPtr> contracts;
    for (auto&& accName : candidates) {
        auto accounts = accountsManager.GetAccountsByName(accName);
        if (!accounts) {
            return MakeUnexpected<TString>("Cannot fetch accounts");
        }
        for (auto&& account : *accounts) {
            auto accountRecord = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
            if (accountRecord && accountRecord->HasBalanceInfo()) {
                if (!contracts.emplace(accountRecord->GetBalanceInfoRef().GetContractId(), account).second) {
                    ERROR_LOG << "dublicate contract id " << accountRecord->GetBalanceInfoRef().GetContractId() << Endl;
                    contracts.erase(accountRecord->GetBalanceInfoRef().GetContractId());
                    TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "duplicate", 1);
                }
            }
        }
    }

    TMap<TString, TDBTag> expiredTags;
    {
        if (RemoveDeprecatedTags) {
            TSet<TString> tagsNames;
            for (const auto& desc : tagsMeta.GetTagsByType(TExpiredDebtAccountTag::TypeName)) {
                tagsNames.emplace(desc->GetName());
            }

            tagsNames.erase(ExpiredDebtAmountTagName);
            if (tagsNames) {
                TVector<TDBTag> tags;
                auto tx = server.GetDriveAPI()->GetBillingManager().BuildSession(false);
                if (!accountsTags.RestoreTags({}, ::MakeVector(tagsNames), tags, tx)) {
                    return MakeUnexpected<TString>("Cannot restore tags with names: " + JoinSeq(", ", tagsNames) + " " + tx.GetStringReport());
                }

                if (tags) {
                    WARNING_LOG << "Reconfiguration of expired debt tag detected, existed tags will be removed: " << JoinSeq(", ", tagsNames) << Endl;
                    if (!accountsTags.RemoveTags(tags, GetRobotUserId(), &server, tx)) {
                        return MakeUnexpected<TString>("Cannot remove tags with names: " + JoinSeq(", ", tagsNames) + " " + tx.GetStringReport());
                    }

                    if (!tx.Commit()) {
                        return MakeUnexpected<TString>("Cannot commit remove of expired tags with names: " + JoinSeq(", ", tagsNames) + " " + tx.GetStringReport());
                    }
                }
            }
        }

        if (ExpiredDebtAmountTagName) {
            auto session = server.GetDriveAPI()->GetBillingManager().BuildSession(true);
            TVector<TDBTag> tags;
            if (!accountsTags.RestoreTags({}, {ExpiredDebtAmountTagName}, tags, session)) {
                return MakeUnexpected<TString>("Cannot restore tags " + ExpiredDebtAmountTagName);
            }

            for (auto&& tag : tags) {
                if (auto impl = std::dynamic_pointer_cast<TExpiredDebtAccountTag>(tag.GetData())) {
                    expiredTags.emplace(tag.GetObjectId(), tag);
                }
            }
        }
    }

    auto nds = server.GetSettings().GetValueDef<ui32>("billing.nds", 0);

    if (contracts.size()) {
        TSet<ui64> localContracts;
        ui32 count = 0;
        for (const auto& contract : contracts) {
            localContracts.emplace(contract.first);
            count++;
            if (count % BalanceBatch == 0 || count == contracts.size()) {
                TExpected<TMap<ui64, TBalanceClient::TBalance>, TBalanceClient::TError> balances = MakeUnexpected<TBalanceClient::TError>({});
                while (!balances && localContracts.size()) {
                    balances = server.GetBalanceClient()->GetBalance(Service, MakeVector(localContracts));
                    if (!balances) {
                        if (balances.GetError().GetCode() != "CONTRACT_IS_NOT_SIGNED" || !balances.GetError().GetContractId()) {
                            return MakeUnexpected(balances.GetError().GetFullError());
                        } else {
                            localContracts.erase(balances.GetError().GetContractId());
                        }
                    }
                }
                if (!balances) {
                    continue;
                }

                for (const auto& [contractId, balanceStruct] : *balances) {
                    auto account = contracts.find(contractId);
                    if (account == contracts.end()) {
                        ERROR_LOG << "Undefined contract " << contractId << Endl;
                        continue;
                    }

                    const auto rawBalance = balanceStruct.ReceiptSum;
                    auto overdraftBalance = rawBalance;
                    auto accountRecord = account->second->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>();
                    if (!accountRecord || !accountRecord->HasBalanceInfo()) {
                        ERROR_LOG << "Undefined balance info " << contractId << Endl;
                        continue;
                    }

                    const auto& balanceInfo = accountRecord->GetBalanceInfoRef();
                    overdraftBalance += balanceInfo.GetOverdraft();

                    auto balance = nds > 0 ? rawBalance * 100  / (100 + nds) : rawBalance;
                    overdraftBalance = nds > 0 ? overdraftBalance * 100  / (100 + nds) : overdraftBalance;

                    bool updateAccount = account->second->GetSoftLimit() != overdraftBalance || rawBalance != balanceInfo.GetReceiptSum() || balanceStruct.ActSum != balanceInfo.GetActSum();
                    bool updateExpireInfo = balanceStruct.HasExpiredDT() || expiredTags.contains(::ToString(accountRecord->GetAccountId()));
                    if (updateAccount || updateExpireInfo) {
                        auto session = server.GetDriveAPI()->GetBillingManager().BuildSession(false);
                        auto taggedAccount = accountsTags.RestoreObject(::ToString(account->second->GetId()), session);
                        if (!taggedAccount) {
                            ERROR_LOG << "cannot get account" << session.GetStringReport() << Endl;
                            continue;
                        }

                        if (updateAccount) {
                            if (account->second->GetSoftLimit() != overdraftBalance) {
                                session.SetComment(GetRTProcessName());
                            }

                            if (balance > account->second->GetSoftLimit()) {
                                auto dbTags = taggedAccount->GetTagsByClass<TAccountTemporaryActionTag>();
                                bool fail = false;
                                for (auto&& tag : dbTags) {
                                    auto tagImpl = tag.MutableTagAs<TAccountTemporaryActionTag>();
                                    if (tagImpl && !tagImpl->GetActive() && tagImpl->HasRequiredSum() && (balance - account->second->GetSoftLimit()) * (100 + nds) >= tagImpl->GetRequiredSumRef() * 100) {
                                        tagImpl->SetActive(true);
                                        if (!server.GetDriveAPI()->GetTagsManager().GetAccountTags().UpdateTagData(tag, GetRobotUserId(), session)) {
                                            ERROR_LOG << "cannot update account tags " << session.GetStringReport() << Endl;
                                            fail = true;
                                            break;
                                        }
                                    }
                                }
                                if (fail) {
                                    continue;
                                }
                            }
                            auto newBalanceInfo = balanceInfo;
                            newBalanceInfo.SetActSum(balanceStruct.ActSum);
                            newBalanceInfo.SetReceiptSum(balanceStruct.ReceiptSum);
                            NJson::TJsonValue accountData;
                            accountData["balance_info"] = newBalanceInfo.ToJson();
                            accountData["soft_limit"] = overdraftBalance;
                            accountData["hard_limit"] = overdraftBalance;
                            if (!account->second->PatchAccountData(accountData, GetRobotUserId(), session)) {
                                ERROR_LOG << "cannot update account " << session.GetStringReport() << Endl;
                                continue;
                            }
                        }

                        TExpiredDebtAccountTag::TPtr expiredDebtTagImpl;
                        TDBTag expiredDebtTag;
                        if (auto it = expiredTags.find(taggedAccount->GetId()); it != expiredTags.end()) {
                            expiredDebtTag = it->second;
                            expiredDebtTagImpl = std::dynamic_pointer_cast<TExpiredDebtAccountTag>(expiredDebtTag.GetData());
                        }

                        if (expiredDebtTagImpl && !balanceStruct.HasExpiredDT()) {
                            if (!accountsTags.RemoveTags({expiredDebtTag}, GetRobotUserId(), &server, session)) {
                                ERROR_LOG << "cannot remove expired debt tag" << session.GetStringReport() << Endl;
                                continue;
                            }

                            expiredDebtTagImpl.Reset();
                        } else if (!expiredDebtTagImpl && balanceStruct.HasExpiredDT() && balanceInfo.GetOverdraft() != 0) {
                            expiredDebtTagImpl = tagsMeta.CreateTagAs<TExpiredDebtAccountTag>(ExpiredDebtAmountTagName);
                            if (expiredDebtTagImpl) {
                                expiredDebtTagImpl->OptionalFirstExpiredDT() = balanceStruct.OptionalFirstDebtFromDT();
                                expiredDebtTagImpl->OptionalDebtAmount() = balanceStruct.OptionalExpiredDebtAmount();
                                if (!expiredDebtTagImpl->CalculateValues(&server, session)) {
                                    ERROR_LOG << "cannot calculate tag values account " << session.GetStringReport() << Endl;
                                    continue;
                                }

                                if (!accountsTags.AddTag(expiredDebtTagImpl, GetRobotUserId(), ToString(account->second->GetId()), &server, session)) {
                                    ERROR_LOG << "cannot add expired debt tag account id: " << account->second->GetId() << " " << session.GetStringReport() << Endl;
                                    continue;
                                }
                            }
                        }  else if (expiredDebtTagImpl) {
                            auto currentTag = expiredDebtTagImpl->Clone(tagsManager);
                            TExpiredDebtAccountTag::TPtr currentTagImpl = std::dynamic_pointer_cast<TExpiredDebtAccountTag>(currentTag);
                            if (!currentTagImpl) {
                                ERROR_LOG << "error while cloning debt account tag: " << session.GetStringReport() << Endl;
                                continue;
                            }

                            expiredDebtTagImpl->OptionalFirstExpiredDT() = balanceStruct.OptionalFirstDebtFromDT();
                            expiredDebtTagImpl->OptionalDebtAmount() = balanceStruct.OptionalExpiredDebtAmount();
                            if (!expiredDebtTagImpl->CalculateValues(&server, session)) {
                                ERROR_LOG << "cannot calculate tag values account " << session.GetStringReport() << Endl;
                                continue;
                            }

                            if (!expiredDebtTagImpl->IsEqual(*currentTagImpl.Get())) {
                                if (!accountsTags.UpdateTagData(expiredDebtTag, GetRobotUserId(), session)) {
                                    ERROR_LOG << "cannot update expired debt tag, account id: " << account->second->GetId() << " " << session.GetStringReport() << Endl;
                                    continue;
                                }
                            }
                        }

                        if (expiredDebtTagImpl && VerboseLog) {
                            INFO_LOG << GrepLogSubstr << " Account_id: " << account->second->GetId() << " " <<  expiredDebtTagImpl->SerializeToLog();
                        }

                        if (expiredDebtTagImpl && expiredDebtTagImpl->MustBeDeactivated(&server, session) && DeactivateAccounts) {
                            if (!accountsManager.ActivateAccounts({ (ui32)account->second->GetId() }, false, GetRobotUserId(), session)) {
                                ERROR_LOG << "cannot deactivate account " << session.GetStringReport() << Endl;
                                continue;
                            }
                        } else if (session->IsWritable()){
                            if (!account->second->GetRecord()->GetActive() && balance && !accountsManager.ActivateAccounts({ (ui32)account->second->GetId() }, true, GetRobotUserId(), session)) {
                                ERROR_LOG << "cannot activate account " << session.GetStringReport() << Endl;
                                continue;
                            }
                        }

                        if (!session.Commit()) {
                            ERROR_LOG << "cannot update account " << session.GetStringReport() << Endl;
                            continue;
                        }

                        TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "update", 1);
                        TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "sum", balance - account->second->GetHardLimit());
                    }
                }

                localContracts.clear();
            }
        }
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TRTBillingB2BBalanceRefreshWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("service", "Сервис").SetDefault(702).SetRequired(true);
    scheme.Add<TFSNumeric>("balance_batch", "максимальное кол-во контрактов для запроса в баланс").SetDefault(30).SetRequired(true);

    const NDrive::IServer* serverImpl = VerifyDynamicCast<const NDrive::IServer*>(&server);
    if (serverImpl) {
        auto&& tagsMeta = serverImpl->GetDriveDatabase().GetTagsManager().GetTagsMeta();
        auto tags = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Account, {TExpiredDebtAccountTag::TypeName});
        scheme.Add<TFSVariants>("expired_debt_amount_tag_name", "Тег для процесса блокировки организаций в овердрафте").SetVariants( NContainer::Keys(tags));
    }

    scheme.Add<TFSBoolean>("deactivate_accounts", "Деактивировать аккаунты с долгом в балансе").SetDefault(false);
    scheme.Add<TFSBoolean>("verbose_log", "Писать в лог информацию о состоянии аккаунтов").SetDefault(false);
    scheme.Add<TFSString>("log_grep_substr", "Подстрока для поиска по логу").SetDefault(GrepLogSubstr);
    scheme.Add<TFSBoolean>("remove_deprecated_tags", "Удалять все типы тегов задолженности кроме текущего").SetDefault(RemoveDeprecatedTags);

    return scheme;
}

bool TRTBillingB2BBalanceRefreshWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return NJson::ParseField(jsonInfo, "service", Service, true)
        && NJson::ParseField(jsonInfo, "balance_batch", BalanceBatch, false)
        && NJson::ParseField(jsonInfo, "expired_debt_amount_tag_name", ExpiredDebtAmountTagName, false)
        && NJson::ParseField(jsonInfo, "deactivate_accounts", DeactivateAccounts, false)
        && NJson::ParseField(jsonInfo, "verbose_log", VerboseLog, false)
        && NJson::ParseField(jsonInfo, "log_grep_substr", GrepLogSubstr, false)
        && NJson::ParseField(jsonInfo, "remove_deprecated_tags", RemoveDeprecatedTags, false)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTBillingB2BBalanceRefreshWatcher::DoSerializeToJson() const {
    auto jsonInfo = TBase::DoSerializeToJson();
    NJson::InsertField(jsonInfo, "service", Service);
    NJson::InsertField(jsonInfo, "balance_batch", BalanceBatch);
    NJson::InsertField(jsonInfo, "expired_debt_amount_tag_name", ExpiredDebtAmountTagName);
    NJson::InsertField(jsonInfo, "deactivate_accounts", DeactivateAccounts);
    NJson::InsertField(jsonInfo, "verbose_log", VerboseLog);
    NJson::InsertField(jsonInfo, "log_grep_substr", GrepLogSubstr);
    NJson::InsertField(jsonInfo, "remove_deprecated_tags", RemoveDeprecatedTags);
    return jsonInfo;
}

TExpectedState TRTBillingUpdateUndefinedClients::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    if (!server.GetBalanceClient()) {
        ERROR_LOG << "Incorrect balance client" << Endl;
        return nullptr;
    }

    TVector<TDBTag> tags;
    {
        auto session = server.GetDriveAPI()->GetTagsManager().GetUserTags().BuildSession(true);
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, { TagName }, tags, session)) {
            return MakeUnexpected<TString>(session.GetStringReport());
        }
    }
    TSet<TString> userIds;
    for (auto&& tag : tags) {
        userIds.insert(tag.GetObjectId());
    }

    TBalanceClientConfig testConfig = TBalanceClientConfig::ParseFromString(
        TStringBuilder() << "Host: " << "http://greed-ts.paysys.yandex.ru:30702" << Endl
        << "Uri: xmlrpc" << Endl
        << "<RequestConfig>" << Endl
        << "MaxAttempts: 1" << Endl
        << "TimeoutSendingms: 1000" << Endl
        << "</RequestConfig>" << Endl
        << "MethodPrefix: " << "TestBalance" << Endl
    );
    TBalanceClient testBalanceClient(testConfig);

    const NDrive::NBilling::TAccountsManager& accountsManager = server.GetDriveAPI()->GetBillingManager().GetAccountsManager();
    const auto userData = server.GetDriveAPI()->GetUsersData();
    const auto userFetchResult = userData->FetchInfo(userIds);

    auto nds = server.GetSettings().GetValueDef<ui32>("billing.nds", 0);

    for (const auto& tag : tags) {
        auto walletTag = tag.GetTagAs<TWalletAccessTag>();
        if (walletTag && walletTag->GetParentId()) {
            auto account = accountsManager.GetAccountById(walletTag->GetParentId());
            const NDrive::NBilling::TLimitedAccountRecord* accountRecord;
            if (account && (accountRecord = account->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>()) && accountRecord->HasBalanceInfo()) {
                TBalanceClient::TClient client = DefaultClient;
                TBalanceClient::TPerson person = DefaultPerson;
                TBalanceClient::TContract contract;
                auto userDataPtr = userFetchResult.GetResultPtr(tag.GetObjectId());
                if (!userDataPtr) {
                    ERROR_LOG << "incorrect user " << tag.GetObjectId() << Endl;
                    TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "incorrect_user", 1);
                    continue;
                }
                auto restorePassport = server.GetBalanceClient()->RestorePassport(userDataPtr->GetUid());
                if (!restorePassport) {
                    return MakeUnexpected(restorePassport.GetError().GetFullError());
                }
                auto clientId = server.GetBalanceClient()->GetOrCreateContractForUid(userDataPtr->GetUid(), client, person, contract);
                if (!clientId) {
                    return MakeUnexpected(clientId.GetError().GetFullError());
                }
                if (clientId->first != accountRecord->GetBalanceInfoRef().GetClientId() || contract.GetId() != accountRecord->GetBalanceInfoRef().GetContractId()) {
                    auto personId = person.GetId() ? person.GetId() : contract.GetPersonId();
                    auto balances = server.GetBalanceClient()->GetBalance(Service, { contract.GetId() });
                    if (!balances) {
                        return MakeUnexpected(balances.GetError().GetFullError());
                    }
                    auto contractBalance = balances->find(contract.GetId());
                    if (contractBalance == balances->end()) {
                        return MakeUnexpected<TString>("Cannot restore balance");
                    }

                    auto currentBalance = accountRecord->GetSoftLimit() * (100 + nds) / 100 - accountRecord->GetBalanceInfoRef().GetOverdraft();
                    if (currentBalance > contractBalance->second.ReceiptSum) {
                        auto result = testBalanceClient.TopupBalance(currentBalance - contractBalance->second.ReceiptSum, personId);
                        if (!result) {
                            return MakeUnexpected(result.GetError().GetFullError());
                        }
                    }

                    NDrive::NBilling::TBalanceClientInfo balanceInfo;
                    balanceInfo.SetClientId(clientId->first);
                    balanceInfo.SetPersonId(personId);
                    balanceInfo.SetContractId(contract.GetId());
                    balanceInfo.SetExternalContractId(contract.GetExternalId());

                    NJson::TJsonValue accountData;
                    accountData["balance_info"] = balanceInfo.ToJson();
                    auto session = server.GetDriveAPI()->GetBillingManager().BuildSession(false);
                    if (!account->PatchAccountData(accountData, GetRobotUserId(), session) || !session.Commit()) {
                        return MakeUnexpected<TString>(session.GetStringReport());
                    }
                }
            } else {
                ERROR_LOG << "incorrect account " << walletTag->GetParentId() << Endl;
                TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "incorrect_account", 1);
            }
        } else {
            ERROR_LOG << "Incorrect tag for " << tag.GetObjectId() << Endl;
            TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "incorrect_tag", 1);
        }
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TRTBillingUpdateUndefinedClients::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("service", "Сервис").SetDefault(702).SetRequired(true);
    scheme.Add<TFSString>("tag_name", "Имя тега руководителя").SetRequired(true);
    auto& accountStructure = scheme.Add<TFSStructure>("default_account", "Данные дефолтной организации").SetStructure<NDrive::TScheme>();
    accountStructure.Add<TFSString>("company", "Название организации").SetRequired(true);

    {
        auto descriptionStr = server.GetSettings().GetValue<TString>("scheme.balance_client.person").GetOrElse("");
        NJson::TJsonValue descriptionJson;
        if (!ReadJsonTree(descriptionStr, &descriptionJson)) {
            WARNING_LOG << "Error while parsing gvar value scheme.balance_client.person";
        }

        TBalanceClient::TPerson::AddToScheme(accountStructure, descriptionJson);
    }
    {
        auto descriptionStr = server.GetSettings().GetValue<TString>("scheme.balance_client.client").GetOrElse("");
        NJson::TJsonValue descriptionJson;
        if (!ReadJsonTree(descriptionStr, &descriptionJson)) {
            WARNING_LOG << "Error while parsing gvar value scheme.balance_client.client";
        }

        TBalanceClient::TClient::AddToScheme(accountStructure, descriptionJson);
    }

    return scheme;
}

bool TRTBillingUpdateUndefinedClients::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return NJson::ParseField(jsonInfo, "service", Service, true)
        && NJson::ParseField(jsonInfo, "tag_name", TagName, true)
        && DefaultClient.FromJson(jsonInfo["default_account"])
        && DefaultPerson.FromJson(jsonInfo["default_account"])
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTBillingUpdateUndefinedClients::DoSerializeToJson() const {
    auto jsonInfo = TBase::DoSerializeToJson();
    NJson::InsertField(jsonInfo, "service", Service);
    NJson::InsertField(jsonInfo, "tag_name", TagName);
    DefaultPerson.AddToJson(jsonInfo["default_account"]);
    return jsonInfo;
}

