#include "alerts.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/database/drive_api.h>

#include <rtline/util/algorithm/container.h>

#include <util/string/join.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTBillingAlerts> TRTBillingAlerts::Registrator(TRTBillingAlerts::GetTypeName());
TRTBillingAlertsState::TFactory::TRegistrator<TRTBillingAlertsState> TRTBillingAlertsState::Registrator(TRTBillingAlerts::GetTypeName());

TString TRTBillingAlertsState::GetType() const {
    return TRTBillingAlerts::GetTypeName();
}

namespace {
    TNamedSignalSimple BillingFinalSum("billing-final-sum");
    TNamedSignalSimple BillingPaymentsFullSum("billing-payments-full-sum");

    TNamedSignalSimple CompiledBillsCount("compiled-bills-count");
    TNamedSignalSimple CompiledRidesCount("compiled-rides-count");
    TNamedSignalSimple CompiledBillsSum("compiled-bills-sum");
    TNamedSignalSimple CompiledRidesSum("compiled-rides-sum");
}

TExpectedState TRTBillingAlerts::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    const TBillingManager& billingManager = server->GetDriveAPI()->GetBillingManager();

    const TRTBillingAlertsState* state = dynamic_cast<const TRTBillingAlertsState*>(stateExt.Get());
    THolder<TRTBillingAlertsState> result(new TRTBillingAlertsState);
    const ui64 historyIdCursor = billingManager.GetHistoryManager().GetLockedMaxEventId();
    const ui64 lastEventId = state ? state->GetLastEventId() : historyIdCursor;

    if (lastEventId >= historyIdCursor) {
        result->SetLastEventId(lastEventId);
        return result.Release();
    }

    auto session = billingManager.BuildSession(true);
    auto events = billingManager.GetHistoryManager().GetEvents({lastEventId + 1, historyIdCursor + 1}, {}, session);
    if (!events) {
        return MakeUnexpected<TString>(session.GetStringReport());
    }
    TVector<TString> ids;
    for (auto&& ev : *events) {
        if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
            ids.emplace_back(ev.GetId());
        }
    }

    if (ids.empty()) {
        result->SetLastEventId(historyIdCursor);
        return result.Release();
    }

    TMap<TString, TCachedPayments> snapshots;
    if (!billingManager.GetPaymentsManager().GetPayments(snapshots, ids, session, true)) {
        return MakeUnexpected<TString>(session.GetStringReport());
    }

    for (auto&& ev : *events) {
        if (ev.GetHistoryAction() == EObjectHistoryAction::Remove) {
            BillingFinalSum.Signal(ev.GetBillCorrected());
            auto it = snapshots.find(ev.GetId());
            if (it != snapshots.end()) {
                BillingPaymentsFullSum.Signal(it->second.GetHeldSum());
                if (ev.GetBillCorrected() != it->second.GetHeldSum()) {
                    TUnistatSignalsCache::SignalAdd("alert-billing-payments", "difference", (ev.GetBillCorrected() > it->second.GetHeldSum()) ? ev.GetBillCorrected() - it->second.GetHeldSum() : (-1 * (it->second.GetHeldSum() - ev.GetBillCorrected())));
                    NDrive::TEventLog::Log("BillingAlert", "Difference in payments for" + ev.GetId() + "/" + ::ToString(ev.GetBillCorrected()) + "/" + ::ToString(it->second.GetHeldSum()));
                }
            } else if (ev.GetBillCorrected() > 100) {
                TUnistatSignalsCache::SignalAdd("alert-billing-payments", "not_found", 1);
                NDrive::TEventLog::Log("BillingAlert", "Cannot find payments for " + ev.GetId());
            }
        }
    }
    result->SetLastEventId(historyIdCursor);
    return result.Release();
}

NDrive::TScheme TRTBillingAlerts::DoGetScheme(const IServerBase& server) const {
    return TBase::DoGetScheme(server);
}

bool TRTBillingAlerts::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTBillingAlerts::DoSerializeToJson() const {
    return TBase::DoSerializeToJson();
}

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTBillCompareAlerts> TRTBillCompareAlerts::Registrator(TRTBillCompareAlerts::GetTypeName());
TRTBillCompareAlertsState::TFactory::TRegistrator<TRTBillCompareAlertsState> TRTBillCompareAlertsState::Registrator(TRTBillCompareAlerts::GetTypeName());

TString TRTBillCompareAlertsState::GetType() const {
    return TRTBillCompareAlerts::GetTypeName();
}


TExpectedState TRTBillCompareAlerts::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    const TMinimalRidingHistoryManager& compiledRides = server->GetDriveAPI()->GetMinimalCompiledRides();

    if (!server->GetDriveAPI()->HasBillingManager()) {
        return nullptr;
    }

    const TRTBillCompareAlertsState * state = dynamic_cast<const TRTBillCompareAlertsState *>(stateExt.Get());
    THolder<TRTBillCompareAlertsState > result(new TRTBillCompareAlertsState);
    const ui64 historyIdCursor = compiledRides.GetLockedMaxEventId();
    const ui64 lastEventId = state ? state->GetLastEventId() : historyIdCursor;

    if (lastEventId >= historyIdCursor) {
        result->SetLastEventId(lastEventId);
        return result.Release();
    }

    auto session = compiledRides.BuildSession(true);
    auto optionalEvents = compiledRides.GetEvents(lastEventId + 1, session);
    if (!optionalEvents) {
        return MakeUnexpected("cannot GetEvents: " + session.GetStringReport());
    }

    TMap<TString, ui32> bills;
    for (auto&& ev : *optionalEvents) {
        if (ev.GetHistoryEventId() > historyIdCursor) {
            break;
        }
        ui32 totalSum = ev.GetSumPrice();
        bills.emplace(ev.GetSessionId(), totalSum);
    }

    if (bills.empty()) {
        result->SetLastEventId(historyIdCursor);
        return result.Release();
    }

    auto compiledBills = server->GetDriveAPI()->GetBillingManager().GetCompiledBills().GetFullBillsFromDB(MakeVector(NContainer::Keys(bills)), session);
    if (!compiledBills) {
        return MakeUnexpected(session.GetStringReport());
    }

    ui32 fullCompiledBills = 0;
    for (auto&& [sessionId, compiledSum] : bills) {
        auto itBill = compiledBills->find(sessionId);
        if (itBill == compiledBills->end()) {
            ERROR_LOG << GetRobotId() << "No compiled bills checking for " << sessionId << Endl;
            continue;
        }
        const auto& cBill = itBill->second;
        if (!cBill.GetFinal()) {
            ERROR_LOG << GetRobotId() << " Inconsistance in compiled bills checking for non final " << sessionId << Endl;
            continue;
        }
        fullCompiledBills++;
        CompiledBillsSum.Signal(cBill.GetBill());
        CompiledRidesSum.Signal(compiledSum);
        TUnistatSignalsCache::SignalAdd("compiled-bills-rides", "sum-diff", (i64)compiledSum - cBill.GetBill());
        if (compiledSum != cBill.GetBill()) {
            ERROR_LOG << "Inconsistance in compiled bills checking for " << cBill.GetSessionId() << ":" << compiledSum << "/" << cBill.GetBill() << Endl;
        }
    }

    CompiledRidesCount.Signal(bills.size());
    CompiledBillsCount.Signal(fullCompiledBills);
    TUnistatSignalsCache::SignalAdd("compiled-bills-rides", "count-diff", (i64)bills.size() - fullCompiledBills);

    result->SetLastEventId(historyIdCursor);
    return result.Release();
}

NDrive::TScheme TRTBillCompareAlerts::DoGetScheme(const IServerBase& server) const {
    return TBase::DoGetScheme(server);
}

bool TRTBillCompareAlerts::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTBillCompareAlerts::DoSerializeToJson() const {
    return TBase::DoSerializeToJson();
}
