#include "bonus_tags.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/chat/engine.h>
#include <drive/backend/chat_robots/state_storage.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

#include <util/string/join.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTBillingTagsWatcher> TRTBillingTagsWatcher::Registrator(TRTBillingTagsWatcher::GetTypeName());

TExpectedState TRTBillingTagsWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const auto& tagsManager = server.GetDriveAPI()->GetTagsManager();
    const auto& billingManager = server.GetDriveAPI()->GetBillingManager();
    if (!server.GetChatEngine()) {
        ERROR_LOG << "Chat storage undefined" << Endl;
        return nullptr;
    }

    TVector<TString> billingTagNames;
    for (auto&& tag : server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags()) {
        if (tag.second->GetType() == TOperationTag::TypeName || tag.second->GetType() == TFixedSumTag::TypeName) {
            billingTagNames.emplace_back(tag.second->GetName());
        }
    }

    TVector<TDBTag> billingTags;
    {
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, billingTagNames, billingTags, session)) {
            ERROR_LOG << "Cannot restore tags " << session.GetStringReport() << Endl;
            return MakeUnexpected<TString>("Cannot restore tags " + session.GetStringReport());
        }
    }

    ui32 allTags = 0;
    TMap<TString, TMap<TString, NDrive::NBilling::IBillingAccount::TPtr>> newlyCreated;
    for (auto&& commonTag : billingTags) {
        const TString userId = commonTag.GetObjectId();
        bool userError = false;

        TVector<TDBTag> toRemove;
        TVector<std::pair<TDBTag, ITag::TPtr>> toUpdate;

        auto copyTagData = commonTag.Clone(server.GetDriveAPI()->GetTagsHistoryContext());
        TBillingTag* tagData = copyTagData.MutableTagAs<TBillingTag>();
        if (!tagData || tagData->GetActivateSince() > StartInstant) {
            continue;
        }

        const TBillingTagDescription* billingDesc = dynamic_cast<const TBillingTagDescription*>(tagsManager.GetTagsMeta().GetDescriptionByName(commonTag->GetName()).Get());
        if (!billingDesc) {
            continue;
        }

        if (billingDesc->GetAvailableAccounts().size() != 1) {
            ERROR_LOG << "Not implemented for multiple accounts " << billingDesc->GetName() << Endl;
            continue;
        }

        auto session = tagsManager.GetUserTags().BuildSession(false);

        const TString tagAccountName = *billingDesc->GetAvailableAccounts().begin();
        auto accountIter = newlyCreated[userId].find(tagAccountName);
        if (accountIter == newlyCreated[userId].end()) {
            auto accountDescription = billingManager.GetAccountsManager().GetDescriptionByName(tagAccountName);
            if (accountDescription.Defined()) {
                accountIter = newlyCreated[userId].emplace(tagAccountName, billingManager.GetAccountsManager().GetOrCreateAccount(userId, accountDescription.GetRef(), GetRobotUserId(), session)).first;
            } else {
                ERROR_LOG << "Can't find account for " << commonTag.GetTagId() << Endl;
                TUnistatSignalsCache::SignalAdd("active-billing-tags", "no-account", toRemove.size() + toUpdate.size());
                continue;
            }
        }

        if (!accountIter->second) {
            ERROR_LOG << "Can't find account for " << commonTag.GetTagId() << Endl;
            TUnistatSignalsCache::SignalAdd("active-billing-tags", "no-account", toRemove.size() + toUpdate.size());
            continue;
        }

        NDrive::TEntitySession chatSession;
        if (tagData->GetTopicLink(server)) {
            chatSession = server.GetChatEngine()->BuildSession();
        }

        const TString lastResolution = tagData->GetResolution();
        auto resultOperation = tagData->ProcessOperation(commonTag.GetTagId(), accountIter->second, GetRobotUserId(), server, session, (chatSession ? &chatSession : nullptr));
        switch (resultOperation) {
        case TBillingTag::EOperationAction::Error:
            ERROR_LOG << "Error processing tag " <<  commonTag.GetObjectId() << "/" << commonTag->GetName() << "/" << commonTag.GetTagId() << Endl;
            userError = true;
            break;
        case TBillingTag::EOperationAction::Remove:
        case TBillingTag::EOperationAction::RemoveWithMessage:
            toRemove.emplace_back(commonTag);
            if (tagData->GetResolution() != lastResolution) {
                toUpdate.emplace_back(commonTag, copyTagData.GetData());
            }
            break;
        case TBillingTag::EOperationAction::Update:
            toUpdate.emplace_back(commonTag, copyTagData.GetData());
            break;
        default:
            break;
        }

        if (userError) {
            TUnistatSignalsCache::SignalAdd("active-billing-tags", "user-error", 1);
            continue;
        }

        for (const auto& [dbTag, tag] : toUpdate) {
            if (!tagsManager.GetUserTags().UpdateTagData(dbTag, tag, GetRobotUserId(), &server, session)) {
                ERROR_LOG << "Commit fails TRTBillingTagsWatcher " << session.GetStringReport() << (chatSession ? " " + chatSession.GetStringReport() : "") << Endl;
                TUnistatSignalsCache::SignalAdd("active-billing-tags", "fail", 1);
                break;
            }
        }

        if (!tagsManager.GetUserTags().RemoveTagsSimple(toRemove, GetRobotUserId(), session, false)
            || !session.Commit()
            || (chatSession && !chatSession.Commit())) {
            ERROR_LOG << "Commit fails TRTBillingTagsWatcher " << session.GetStringReport() << (chatSession ? " " + chatSession.GetStringReport() : "") << Endl;
            TUnistatSignalsCache::SignalAdd("active-billing-tags", "fail", toRemove.size() + toUpdate.size());
            continue;
        }
        allTags = allTags + toRemove.size() + toUpdate.size();
    }
    TUnistatSignalsCache::SignalAdd("active-billing-tags", "count", allTags);
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TRTBillingTagsWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    return scheme;
}

bool TRTBillingTagsWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTBillingTagsWatcher::DoSerializeToJson() const {
    return TBase::DoSerializeToJson();
}
