#include "cashback.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/data/billing_tags.h>
#include <drive/backend/data/user_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/action.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/library/cpp/blackbox/client.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTCashbackWatcher> TRTCashbackWatcher::Registrator(TRTCashbackWatcher::GetTypeName());
TRTHistoryWatcherState::TFactory::TRegistrator<TRTCashbackWatcherState> TRTCashbackWatcherState::Registrator(TRTCashbackWatcher::GetTypeName());

ECashbackUserPolicy TCashbackDefaultUserConfig::Type = ECashbackUserPolicy::Default;
TCashbackDefaultUserConfig::TFactory::TRegistrator<TCashbackDefaultUserConfig> TCashbackDefaultUserConfig::Registrator(TCashbackDefaultUserConfig::Type);

ECashbackUserPolicy TCashbackConnectedUserConfig::Type = ECashbackUserPolicy::Connection;
TCashbackConnectedUserConfig::TFactory::TRegistrator<TCashbackConnectedUserConfig> TCashbackConnectedUserConfig::Registrator(TCashbackConnectedUserConfig::Type);

TCashbackReferralUserConfig::TFactory::TRegistrator<TCashbackReferralUserConfig> TCashbackReferralUserConfig::Registrator(ECashbackUserPolicy::Referral);

THolder<ICashbackUserPolicy> TCashbackConnectedUserConfig::BuildPolicy() const {
    return THolder(new TCashbackConnectedUser(*this));
}

TMaybe<TUserPermissionsConstPtr> ICashbackUserPolicy::GetCashbackUserPermissions(const TString& id, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    TString targetId;
    auto found = GetCashbackUser(id, targetId, server, session);
    if (!found) {
        return {};
    }
    auto permissions = Yensured(server.GetDriveAPI())->GetUserPermissions(targetId, TUserPermissionsFeatures());
    if (!permissions) {
        session.SetErrorInfo("GetCashbackUser", "cannot constract permissions");
        return {};
    }
    return permissions;
}

NDrive::TScheme TCashbackConnectedUserConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme;
    auto frServer = server.GetAs<NDrive::IServer>();
    if (frServer) {
        auto descriptions = frServer->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TConnectionUserTag::TypeName);
        TSet<TString> tagNames;
        for (auto& tag : descriptions) {
            tagNames.emplace(tag->GetName());
        }
        scheme.Add<TFSVariants>("tag_name", "Имя тега").SetVariants(tagNames).SetRequired(true);
    }
    return scheme;
}

bool TCashbackConnectedUser::GetConnectionTag(const TString& id, TMaybe<TDBTag>& tag, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    const auto& userTagManager = server.GetDriveAPI()->GetTagsManager().GetUserTags();
    auto optionalTags = userTagManager.RestoreTags(TVector<TString>{ id }, { Config.GetTagName() }, session);
    if (!optionalTags) {
        return false;
    }
    if (optionalTags->empty()) {
        return true;
    }
    tag = std::move(optionalTags->front());
    return true;
}

bool TCashbackConnectedUser::GetCashbackUser(const TString& id, TString& targetId, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    TMaybe<TDBTag> connectionTag;
    if (!GetConnectionTag(id, connectionTag, server, session)) {
        return false;
    }
    if (!connectionTag.Defined()) {
        return true;
    }
    auto tagImpl = connectionTag->GetTagAs<TConnectionUserTag>();
    if (!tagImpl) {
        return false;
    }
    targetId = tagImpl->GetConnectedUserId();
    return true;
}

bool TCashbackConnectedUser::CorrectCashbackAmount(const TString& /*id*/, const ui32 originalAmount, ui32& resultAmount, const NDrive::IServer& /*server*/, NDrive::TEntitySession& /*session*/, const TString& /*robotId*/) const {
    resultAmount = originalAmount;
    return true;
}

NDrive::TScheme TCashbackReferralUserConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TCashbackConnectedUserConfig::GetScheme(server);
    scheme.Add<TFSVariants>("referral_type", "Тип рефералки").InitVariants<EReferralType>().SetDefault(ToString(EReferralType::Bonuses));
    return scheme;
}

THolder<ICashbackUserPolicy> TCashbackReferralUserConfig::BuildPolicy() const {
    return THolder(new TCashbackReferralUser(*this));
}

bool TCashbackReferralUser::GetCashbackUser(const TString& id, TString& targetId, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    return TCashbackConnectedUser::GetCashbackUser(id, targetId, server, session);
}

TMaybe<TUserPermissionsConstPtr> TCashbackReferralUser::GetCashbackUserPermissions(const TString& id, const NDrive::IServer& server, NDrive::TEntitySession& session) const {
    TString targetId;
    auto found = GetCashbackUser(id, targetId, server, session);
    if (!found) {
        return {};
    }
    if (!targetId) {
        return nullptr;
    }

    auto usersData = server.GetDriveAPI()->GetUsersData()->FetchInfo(targetId, session);
    if (!usersData) {
        return {};
    }
    auto userPtr = usersData.GetResultPtr(targetId);
    if (!userPtr) {
        session.SetErrorInfo("GetCashbackUser", "incorrect uset");
        return {};
    }
    auto client = userPtr->GetBlackboxClient(server);
    if (!client) {
        session.SetErrorInfo("GetCashbackUser", "cannot construst client");
        return {};
    }
    auto blackboxInfoFuture = client->UidInfoRequest(userPtr->GetUid(), "8.8.8.8").Apply([
        client
    ] (const auto& waiter) -> NThreading::TFuture<NDrive::TBlackboxInfo> {
        auto responcePtr = waiter.GetValue();
        if (client) {
            return  NThreading::MakeFuture(client->Parse(*responcePtr));
        }
        return NThreading::TExceptionFuture() << "Incorrect client";
    });

    blackboxInfoFuture.Wait();
    if (!blackboxInfoFuture.HasValue()) {
        session.SetErrorInfo("GetCashbackUser", NThreading::GetExceptionMessage(blackboxInfoFuture));
        return {};
    }

    auto bbInfo = blackboxInfoFuture.ExtractValue();

    TUserPermissionsFeatures upFeatures;
    upFeatures.SetIsPlusUser(bbInfo.IsPlusUser);
    upFeatures.SetIsYandexUser(bbInfo.IsYandexoid);

    TUserPermissions::TPtr userPermissions = server.GetDriveAPI()->GetUserPermissions(targetId, upFeatures);
    if (!userPermissions) {
        session.SetErrorInfo("GetCashbackUser", "cannot constract permissions");
        return {};
    }
    TString code;
    if (!server.GetDriveAPI()->CheckReferralProgramParticipationImpl(code, *userPermissions, session)) {
        return nullptr;
    }
    auto referralType = userPermissions->GetSetting<EReferralType>("referral_type").GetOrElse(EReferralType::Bonuses);
    if (referralType != ReferralType) {
        return nullptr;
    }
    return userPermissions;
}

bool TCashbackReferralUser::CorrectCashbackAmount(const TString& id, const ui32 originalAmount, ui32& resultAmount, const NDrive::IServer& server, NDrive::TEntitySession& session, const TString& robotId) const {
    auto amount = originalAmount;
    if (!TCashbackConnectedUser::CorrectCashbackAmount(id, originalAmount, amount, server, session, robotId)) {
        return false;
    }
    TMaybe<TDBTag> connectionTag;
    if (!GetConnectionTag(id, connectionTag, server, session) || !connectionTag.Defined()) {
        return false;
    }

    auto tagImpl = connectionTag->MutableTagAs<TReferralConnectionTag>();
    if (!tagImpl) {
        return false;
    }

    resultAmount = Min<ui32>(amount, tagImpl->GetAvailableBalance());
    if (resultAmount) {
        tagImpl->SetAvailableBalance(tagImpl->GetAvailableBalance() - resultAmount);
        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(*connectionTag, robotId, session)) {
            return false;
        }
    }
    return true;
}

TString TRTCashbackWatcherState::GetType() const {
    return TRTCashbackWatcher::GetTypeName();
}

TExpectedState TRTCashbackWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    auto result = MakeHolder<TRTCashbackWatcherState>();

    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    if (!server.GetDriveAPI()->HasBillingManager()) {
        return MakeUnexpected<TString>({});
    }
    const auto& billingManager = server.GetDriveAPI()->GetBillingManager();

    const TRTCashbackWatcherState* state = dynamic_cast<const TRTCashbackWatcherState*>(stateExt.Get());
    const ui64 lastEventId = state ? state->GetLastEventId() : StartEventId;
    ui64 historyIdCursor = billingManager.GetHistoryManager().GetLockedMaxEventId();

    if (EventsPerCycle > 0) {
        historyIdCursor = Min<ui32>(historyIdCursor, lastEventId + EventsPerCycle);
    }

    if (historyIdCursor <= lastEventId) {
        return stateExt;
    }

    auto session = billingManager.BuildSession();
    auto billingTasks = billingManager.GetHistoryManager().GetEvents({lastEventId + 1, historyIdCursor + 1}, {}, session);
    if (!billingTasks) {
        return MakeUnexpected<TString>(session.GetStringReport());
    }

    THolder<ICashbackUserPolicy> targetUserBuilder;
    if (CashbackUserPolicyConfig) {
        targetUserBuilder = CashbackUserPolicyConfig->BuildPolicy();
        if (!targetUserBuilder) {
            return MakeUnexpected<TString>("Incorrect configuration");
        }
    }

    auto tagDescription = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(CashbackTag);
    auto billingDescription = dynamic_cast<const TBillingTagDescription*>(tagDescription.Get());
    if (!billingDescription || billingDescription->GetAvailableAccounts().size() != 1) {
        return MakeUnexpected<TString>("incorrect CashbackTag configuration");
    }
    const TString cashbackAccount = *(billingDescription->GetAvailableAccounts().begin());

    for (auto&& task : *billingTasks) {
        if (task.GetHistoryAction() != EObjectHistoryAction::Remove) {
            continue;
        }
        if (task.GetBillingType() != EBillingType::CarUsage) {
            continue;
        }

        TUserPermissionsConstPtr targetUserPtr;
        if (targetUserBuilder) {
            auto permissions = targetUserBuilder->GetCashbackUserPermissions(task.GetUserId(), server, session);
            if (!permissions) {
                return MakeUnexpected<TString>(session.GetStringReport());
            }
            targetUserPtr = *permissions;
        } else {
            targetUserPtr = Yensured(server.GetDriveAPI())->GetUserPermissions(task.GetUserId(), TUserPermissionsFeatures());
        }
        if (!targetUserPtr) {
            continue;
        }
        auto targetUserId = targetUserPtr->GetUserId();

        if (!AutoCreateAccount) {
            auto userAccounts = billingManager.GetAccountsManager().GetUserAccounts(targetUserId, session);
            if (!userAccounts) {
                return MakeUnexpected<TString>(session.GetStringReport());
            }
            bool hasAccount = false;
            for (auto&& acc : *userAccounts) {
                if (acc->GetUniqueName() == cashbackAccount && acc->IsActive()) {
                    hasAccount = true;
                    break;
                }
            }
            if (!hasAccount) {
                continue;
            }
        }

        TCachedPayments payments;
        billingManager.GetPaymentsManager().GetPayments(payments, task.GetId(), session);

        if (payments.GetFirstPaymentTs() < MinimalStartInstant) {
            continue;
        }

        auto tag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(CashbackTag);
        auto billingTag = dynamic_cast<TBillingTag*>(tag.Get());
        if (!billingTag) {
            ERROR_LOG << "Incorrect tag type " << CashbackTag << Endl;
            return nullptr;
        }
        billingTag->SetSessionId(task.GetId());
        billingTag->SetPlusUser(targetUserPtr->GetUserFeatures().GetIsPlusUser());

        if (auto operationTag = dynamic_cast<TOperationTag*>(tag.Get())) {
            ui32 cashbackBase = 0;
            for (auto&& payment : payments.GetPayments()) {
                if (payment.GetAccountId() != 0) {
                    auto account = billingManager.GetAccountsManager().GetAccountById(payment.GetAccountId());
                    if (!account || !AvalableForCashback.contains(account->GetUniqueName())) {
                        continue;
                    }
                } else if (!AvalableForCashback.contains(::ToString(payment.GetPaymentType()))) {
                    continue;
                }
                if (payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Authorized) {
                    cashbackBase += payment.GetSum();
                } else if (payment.GetStatus() == NDrive::NTrustClient::EPaymentStatus::Cleared) {
                    cashbackBase += payment.GetCleared();
                }
            }

            ui32 cashbackSum = (cashbackBase / 100.0) * Discount;
            cashbackSum = cashbackSum  - (cashbackSum % 100);

            if (targetUserBuilder && !targetUserBuilder->CorrectCashbackAmount(task.GetUserId(), cashbackSum, cashbackSum, server, session, GetRobotUserId())) {
                return MakeUnexpected<TString>(session.GetStringReport());
            }

            if (cashbackSum == 0) {
                continue;
            }

            operationTag->SetAmount(cashbackSum);
        }

        if (!server.GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tag, GetRobotUserId(), targetUserId, &server, session)) {
            return MakeUnexpected<TString>(session.GetStringReport());
        }
    }

    if (!session.Commit()) {
        return MakeUnexpected<TString>(session.GetStringReport());
    }

    result->SetLastEventId(historyIdCursor);
    return result.Release();
}

NDrive::TScheme TRTCashbackWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    const NDrive::IServer* serverImpl = VerifyDynamicCast<const NDrive::IServer*>(&server);
    scheme.Add<TFSVariants>("available_accounts", "Aккаунты для кешбека").SetMultiSelect(true).SetReference("accounts");

    auto operationTags = serverImpl->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::User);
    TSet<TString> billingTagNames;
    for (auto& [tagName, tag] : operationTags) {
        if (tag->GetType() == TOperationTag::TypeName || tag->GetType() == TFixedSumTag::TypeName) {
            billingTagNames.emplace(tagName);
        }
    }
    scheme.Add<TFSVariants>("cashback_tag", "Тег для кешбека").SetVariants(billingTagNames).SetMultiSelect(false).SetRequired(true);
    scheme.Add<TFSBoolean>("auto_create").SetDefault(true);
    scheme.Add<TFSNumeric>("amount", "Процент кешбека").SetDefault(0);
    scheme.Add<TFSNumeric>("events_per_cycle", "Максимальное число событий за цикл").SetDefault(10000);
    scheme.Add<TFSVariants>("cashback_user_policy", "Политика выбора пользователя").InitVariants<ECashbackUserPolicy>().SetDefault(::ToString(ECashbackUserPolicy::Default)).SetRequired(true);
    scheme.Add<TFSJson>("cashback_user_policy_config", "Политика выбора пользователя(конфиг)");
    scheme.Add<TFSNumeric>("start_event").SetDefault(0);
    scheme.Add<TFSNumeric>("minimal_start_instant").SetVisual(TFSNumeric::EVisualType::DateTime).SetDefault(0);
    return scheme;
}

bool TRTCashbackWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING_OPT(jsonInfo, "cashback_tag", CashbackTag);
    JREAD_UINT(jsonInfo, "amount", Discount);
    JREAD_UINT(jsonInfo, "events_per_cycle", EventsPerCycle);
    ECashbackUserPolicy userPolicy = ECashbackUserPolicy::Default;
    JREAD_FROM_STRING_OPT(jsonInfo, "cashback_user_policy", userPolicy);
    CashbackUserPolicyConfig.Reset(ICashbackUserPolicyConfig::TFactory::Construct(userPolicy));
    if (!CashbackUserPolicyConfig || !CashbackUserPolicyConfig->DeserializeFromJson(jsonInfo["cashback_user_policy_config"])) {
        return false;
    }

    if (!jsonInfo.Has("available_accounts") || !jsonInfo["available_accounts"].IsArray()) {
        return false;
    }
    for (auto&& account : jsonInfo["available_accounts"].GetArraySafe()) {
        AvalableForCashback.insert(account.GetString());
    }

    JREAD_UINT_OPT(jsonInfo, "start_event", StartEventId);
    JREAD_BOOL_OPT(jsonInfo, "auto_create", AutoCreateAccount);
    JREAD_INSTANT_OPT(jsonInfo, "minimal_start_instant", MinimalStartInstant);
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTCashbackWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    JWRITE(result, "amount", Discount);
    JWRITE(result, "events_per_cycle", EventsPerCycle);
    JWRITE(result, "cashback_tag", CashbackTag);
    JWRITE_DEF(result, "auto_create", AutoCreateAccount, true);
    if (CashbackUserPolicyConfig) {
        JWRITE(result, "cashback_user_policy", ::ToString(CashbackUserPolicyConfig->GetType()));
        JWRITE(result, "cashback_user_policy_config", CashbackUserPolicyConfig->SerializeToJson());
    }

    NJson::TJsonValue accountsJson(NJson::JSON_ARRAY);
    for (auto&& i : AvalableForCashback) {
        accountsJson.AppendValue(i);
    }
    result["available_accounts"] = accountsJson;
    JWRITE(result, "start_event", StartEventId);
    JWRITE_INSTANT(result, "minimal_start_instant", MinimalStartInstant);
    return result;
}

