#pragma once

#include <drive/backend/rt_background/common/state.h>
#include <drive/backend/rt_background/manager/settings.h>

#include <drive/backend/game/logic.h>

#include <rtline/util/types/accessor.h>


class TRTCashbackWatcherState : public TRTHistoryWatcherState {
    static TFactory::TRegistrator<TRTCashbackWatcherState> Registrator;
public:
    virtual TString GetType() const override;
};

enum class ECashbackUserPolicy {
    Default,
    Connection,
    Referral,
};

class ICashbackUserPolicy {
public:
    virtual ~ICashbackUserPolicy() = default;
    virtual bool CorrectCashbackAmount(const TString& id, const ui32 originalAmount, ui32& resultAmount, const NDrive::IServer& server, NDrive::TEntitySession& session, const TString& robotId) const = 0;
    virtual TMaybe<TIntrusiveConstPtr<TUserPermissions>> GetCashbackUserPermissions(const TString& id, const NDrive::IServer& server, NDrive::TEntitySession& session) const;

private:
    virtual bool GetCashbackUser(const TString& id, TString& targetId, const NDrive::IServer& server, NDrive::TEntitySession& session) const = 0;

};

class ICashbackUserPolicyConfig {
public:
    virtual ~ICashbackUserPolicyConfig() = default;
    virtual bool DeserializeFromJson(const NJson::TJsonValue& json) = 0;
    virtual NJson::TJsonValue SerializeToJson() const = 0;
    virtual THolder<ICashbackUserPolicy> BuildPolicy() const = 0;
    virtual NDrive::TScheme GetScheme(const IServerBase& server) const = 0;
    virtual ECashbackUserPolicy GetType() const = 0;

    using TFactory = NObjectFactory::TParametrizedObjectFactory<ICashbackUserPolicyConfig, ECashbackUserPolicy>;
};

class TCashbackDefaultUser : public ICashbackUserPolicy {
public:
    virtual bool GetCashbackUser(const TString& id, TString& targetId, const NDrive::IServer& /*server*/, NDrive::TEntitySession& /*session*/) const override {
        targetId = id;
        return true;
    }
    virtual bool CorrectCashbackAmount(const TString& /*id*/, const ui32 originalAmount, ui32& resultAmount, const NDrive::IServer& /*server*/, NDrive::TEntitySession& /*session*/, const TString& /*robotId*/) const override {
        resultAmount = originalAmount;
        return true;
    }
};

class TCashbackDefaultUserConfig : public ICashbackUserPolicyConfig {
public:
    virtual bool DeserializeFromJson(const NJson::TJsonValue& /*json*/) override {
        return true;
    }
    virtual NJson::TJsonValue SerializeToJson() const override {
        return {};
    }
    virtual THolder<ICashbackUserPolicy> BuildPolicy() const override {
        return THolder(new TCashbackDefaultUser());
    }
    virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override {
        return {};
    }
    virtual ECashbackUserPolicy GetType() const override {
        return Type;
    }

public:
    static ECashbackUserPolicy Type;
    static TFactory::TRegistrator<TCashbackDefaultUserConfig> Registrator;
};

class TCashbackConnectedUserConfig : public ICashbackUserPolicyConfig {
    R_READONLY(TString, TagName);
public:
    virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override {
        JREAD_STRING(json, "tag_name", TagName);
        return true;
    }
    virtual NJson::TJsonValue SerializeToJson() const override {
        NJson::TJsonValue result;
        JWRITE(result, "tag_name", TagName);
        return result;
    }
    virtual NDrive::TScheme GetScheme(const IServerBase& server) const override;
    virtual THolder<ICashbackUserPolicy> BuildPolicy() const override;

    virtual ECashbackUserPolicy GetType() const override {
        return Type;
    }

public:
    static ECashbackUserPolicy Type;
    static TFactory::TRegistrator<TCashbackConnectedUserConfig> Registrator;
};

class TCashbackConnectedUser : public ICashbackUserPolicy {
public:
    TCashbackConnectedUser(const TCashbackConnectedUserConfig& config)
        : Config(config)
    {}

    virtual bool GetCashbackUser(const TString& id, TString& targetId, const NDrive::IServer& server, NDrive::TEntitySession& session) const override;
    virtual bool CorrectCashbackAmount(const TString& id, const ui32 originalAmount, ui32& resultAmount, const NDrive::IServer& server, NDrive::TEntitySession& session, const TString& robotId) const override;

protected:
    bool GetConnectionTag(const TString& id, TMaybe<TDBTag>& tag, const NDrive::IServer& server, NDrive::TEntitySession& session) const;

private:
    const TCashbackConnectedUserConfig& Config;
};

class TCashbackReferralUserConfig : public TCashbackConnectedUserConfig {
    R_FIELD(EReferralType, ReferralType, EReferralType::Bonuses);
public:
    virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override {
        return NJson::ParseField(json, "referral_type", NJson::Stringify(ReferralType))
            && TCashbackConnectedUserConfig::DeserializeFromJson(json);
    }
    virtual NJson::TJsonValue SerializeToJson() const override {
        NJson::TJsonValue result = TCashbackConnectedUserConfig::SerializeToJson();
        NJson::InsertField(result, "referral_type", NJson::Stringify(ReferralType));
        return result;
    }
    virtual NDrive::TScheme GetScheme(const IServerBase& server) const override;
    virtual THolder<ICashbackUserPolicy> BuildPolicy() const override;

    virtual ECashbackUserPolicy GetType() const override {
        return ECashbackUserPolicy::Referral;
    }

public:
    static TFactory::TRegistrator<TCashbackReferralUserConfig> Registrator;
};

class TCashbackReferralUser : public TCashbackConnectedUser {
public:
    TCashbackReferralUser(const TCashbackReferralUserConfig& config)
        : TCashbackConnectedUser(config)
        , ReferralType(config.GetReferralType())
    {}

    virtual bool GetCashbackUser(const TString& id, TString& targetId, const NDrive::IServer& server, NDrive::TEntitySession& session) const override;
    virtual TMaybe<TUserPermissionsConstPtr> GetCashbackUserPermissions(const TString& id, const NDrive::IServer& server, NDrive::TEntitySession& session) const override;
    virtual bool CorrectCashbackAmount(const TString& id, const ui32 originalAmount, ui32& resultAmount, const NDrive::IServer& server, NDrive::TEntitySession& session, const TString& robotId) const override;

private:
    EReferralType ReferralType;
};


class TRTCashbackWatcher: public IRTRegularBackgroundProcess {
    R_FIELD(TString, CashbackTag);
    R_FIELD(TSet<TString>, AvalableForCashback, {});
    R_FIELD(ui32, Discount, 1);
    R_FIELD(ui32, EventsPerCycle, 10000);
    R_FIELD(ui32, StartEventId, 0);
    R_FIELD(bool, AutoCreateAccount, true);
    R_FIELD(TInstant, MinimalStartInstant, TInstant::Zero());
    R_FIELD(TDuration, DebtDuration, TDuration::Max());

private:
    using TBase = IRTRegularBackgroundProcess;
    static TFactory::TRegistrator<TRTCashbackWatcher> Registrator;

public:
    using TBase::TBase;

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

    virtual TString GetType() const override {
        return GetTypeName();
    }

    static TString GetTypeName() {
        return "cashback";
    }

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

private:
    THolder<ICashbackUserPolicyConfig> CashbackUserPolicyConfig;
};
