#include "clearing.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/database/drive_api.h>

#include <rtline/library/unistat/cache.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTClearingWatcher> TRTClearingWatcher::Registrator(TRTClearingWatcher::GetTypeName());
IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTRefundsCycleWatcher> TRTRefundsCycleWatcher::Registrator(TRTRefundsCycleWatcher::GetTypeName());

TExpectedState TRTClearingWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    TMessagesCollector errors;
    auto tasks = server.GetDriveAPI()->GetBillingManager().WaitClearingCycle(TasksAtOnce, TasksAtCycle, ClearingInterval, errors);
    TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "cleared-tasks", tasks);
    if (errors.HasMessages()) {
        return MakeUnexpected(errors.GetStringReport());
    }
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TRTClearingWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSDuration>("clearing_interval", "Интервал клиринга");
    scheme.Add<TFSNumeric>("tasks_at_once", "Одновременно задач");
    scheme.Add<TFSNumeric>("tasks_at_cycle", "Одновременно задач в цикле");
    return scheme;
}

bool TRTClearingWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_INT_OPT(jsonInfo, "tasks_at_once", TasksAtOnce);
    JREAD_INT_OPT(jsonInfo, "tasks_at_cycle", TasksAtCycle);
    JREAD_DURATION_OPT(jsonInfo, "clearing_interval", ClearingInterval);
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTClearingWatcher::DoSerializeToJson() const {
    auto jsonInfo = TBase::DoSerializeToJson();
    JWRITE_DURATION(jsonInfo, "clearing_interval", ClearingInterval);
    JWRITE(jsonInfo, "tasks_at_once", TasksAtOnce);
    JWRITE(jsonInfo, "tasks_at_cycle", TasksAtCycle);
    return jsonInfo;
}

TExpectedState TRTRefundsCycleWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    server.GetDriveAPI()->GetBillingManager().WaitRefundCycle();
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TRTRefundsCycleWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    return scheme;
}

bool TRTRefundsCycleWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTRefundsCycleWatcher::DoSerializeToJson() const {
    auto jsonInfo = TBase::DoSerializeToJson();
    return jsonInfo;
}
