#include "compilation.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/offers/actions/abstract.h>

#include <drive/backend/rt_background/manager/state.h>

#include <rtline/library/storage/sql/query.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTBillingCompilation> TRTBillingCompilation::Registrator(TRTBillingCompilation::GetTypeName());
TRTBillingCompilationState::TFactory::TRegistrator<TRTBillingCompilationState> TRTBillingCompilationState::Registrator(TRTBillingCompilation::GetTypeName());

TString TRTBillingCompilationState::GetType() const {
    return TRTBillingCompilation::GetTypeName();
}

TExpectedState TRTBillingCompilation::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    if (!server.GetDriveAPI()->HasBillingManager()) {
        ERROR_LOG << "Undefined billing manager" << Endl;
        return nullptr;
    }
    const TBillingManager& billingManager = server.GetDriveAPI()->GetBillingManager();

    const TRTBillingCompilationState* state = dynamic_cast<const TRTBillingCompilationState*>(stateExt.Get());
    THolder<TRTBillingCompilationState> newState = MakeHolder<TRTBillingCompilationState>();
    ui64 lastEventId = state ? state->GetLastEventId() : FirstEventId;
    newState->SetLastEventId(lastEventId);

    const ui64 historyIdCursor = billingManager.GetHistoryManager().GetLockedMaxEventId();

    TSet<TString> compiledBills;
    do {
        auto session = billingManager.BuildSession(false);
        auto locked = session.TryLock(GetRobotId() + "_final_events");
        if (!locked) {
            return MakeUnexpected("cannot TryLock: " + session.GetStringReport());
        }
        if (!*locked) {
            break;
        }
        auto optionalEvents = billingManager.GetHistoryManager().GetEvents({ lastEventId + 1 }, {}, session);
        if (!optionalEvents) {
            return MakeUnexpected<TString>("cannot GetEvents: " + session.GetStringReport());
        }

        for (auto&& ev : *optionalEvents) {
            if (ev.GetHistoryEventId() <= historyIdCursor) {
                newState->SetLastEventId(ev.GetHistoryEventId());
            }
            if (compiledBills.contains(ev.GetId())) {
                continue;
            }
            if (ev.GetState() == "finished" && ev.GetHistoryAction() == EObjectHistoryAction::UpdateData) {
                auto bill = billingManager.GetCompiledBills().GetFullBillFromDB(ev.GetId(), session);
                if (!bill) {
                    return MakeUnexpected("cannot GetFullBill: " + session.GetStringReport());
                }
                if (!bill->GetFinal()) {
                    if (!billingManager.BuildCompiledBill(*bill, ev, session, true)) {
                        return MakeUnexpected(session.GetStringReport());
                    }
                    compiledBills.insert(ev.GetId());
                }
            }
        }
        if (!session.Commit()) {
            return MakeUnexpected(session.GetStringReport());
        }
        TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "create-compiled", compiledBills.size());
    } while (false);

    auto sessionBuilder = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilderSafe("billing", StartInstant);
    if (!sessionBuilder) {
        ERROR_LOG << GetRobotId() << " " << "cannot create session builder" << Endl;
        return newState;
    }

    TMaybe<TSet<TString>> availableAccounts;
    if (!!AccountFilter) {
        auto accounts =  billingManager.GetAccountsManager().GetAccountsChildren(AccountFilter);
        if (!accounts) {
            ERROR_LOG << GetRobotId() << " " << "cannot filter accounts" << Endl;
            return newState;
        }
        availableAccounts = MakeSet(NContainer::Keys(*accounts));
    }

    TMap<TString, IEventsSession<TCarTagHistoryEvent>::TPtr> partialBill;

    auto sessions = sessionBuilder->GetSessionsActual();
    for (auto& session : sessions) {
        if (compiledBills.contains(session->GetSessionId())) {
            continue;
        }
        if (session->GetClosed() || !session->GetStartTS() || session->GetStartTS() + UpdatePeriod > StartInstant) {
            continue;
        }

        if (!OfferFilter.IsEmpty() || availableAccounts) {
            TBillingSession::TBillingCompilation compilation;
            if (!session->FillCompilation(compilation)) {
                continue;
            }
            if (!compilation.GetCurrentOffer()) {
                continue;
            }
            if ((i32)compilation.GetRevenue() < 0) {
                continue;
            }

            if (!OfferFilter.IsEmpty()) {
                auto action = server.GetDriveAPI()->GetRolesManager()->GetAction(compilation.GetCurrentOffer()->GetBehaviourConstructorId());
                auto offerBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
                if (!offerBuilder) {
                    continue;
                }
                if (!OfferFilter.IsMatching(offerBuilder->GetGrouppingTags())) {
                    continue;
                }
            }
            if (availableAccounts) {
                if (!availableAccounts->contains(compilation.GetCurrentOffer()->GetSelectedCharge())) {
                    continue;
                }
            }
        }
        partialBill.emplace(session->GetSessionId(), session);
    }

    auto session = billingManager.BuildSession(false);
    auto locked = session.TryLock(GetRobotId() + "_current_events");
    if (!locked || !*locked) {
        ERROR_LOG << GetRobotId() << " cannot TryLock: " << session.GetStringReport() << Endl;
        return newState;
    }
    auto bills = billingManager.GetCompiledBills().GetFullBillsFromDB(MakeVector(NContainer::Keys(partialBill)), session);
    if (!bills) {
        ERROR_LOG << GetRobotId() << " cannot GetFullBills: " << session.GetStringReport() << Endl;
        return newState;
    }

    ui32 count = 0;
    ui32 emptyCount = 0;
    for (const auto& [sessionId, carSession] : partialBill) {
        if (count >= PartSize) {
            break;
        }
        auto bill = billingManager.GetCompiledBills().GetFullBill(sessionId, *bills);
        if (bill.GetFinal() || bill.GetFinalTime() + UpdatePeriod > StartInstant) {
            continue;
        }

        auto billingTask = billingManager.GetActiveTasksManager().GetTask(sessionId, session);
        if (!billingTask) {
            ERROR_LOG << GetRobotId() << " " << sessionId << ": " << session.GetStringReport() << Endl;
            break;
        }

        if (!*billingTask) {
            continue;
        }

        TBillingSession::TBillingCompilation compilation;
        if (!carSession->FillCompilation(compilation)) {
            continue;
        }

        if (bill.GetBill() >= compilation.GetRevenue()) {
            continue;
        }

        auto addBill = billingManager.BuildCompiledBill(bill, *billingTask, session, false, (ui32)compilation.GetRevenue());

        if (!addBill) {
            ERROR_LOG << GetRobotId() << " " << sessionId << ": " << session.GetStringReport() << Endl;
            break;
        }
        if (*addBill) {
            count++;
        } else {
            emptyCount++;
            INFO_LOG << GetRobotId() << " " << sessionId << ": empty bill" << Endl;
        }
    }
    if (!session.Commit()) {
        ERROR_LOG << GetRobotId() << " " << session.GetStringReport() << Endl;
    } else {
        TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "create-partial-compiled", count);
        TUnistatSignalsCache::SignalAdd(GetRTProcessName(), "skip-partial-compiled", emptyCount);
    }
    return newState;
}

NDrive::TScheme TRTBillingCompilation::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("offer_filter", "Фильтр офферов");
    scheme.Add<TFSDuration>("update_period", "Период обновления").SetDefault(TDuration::Days(1)).SetRequired(true);
    scheme.Add<TFSNumeric>("first_event_id", "Начать с").SetDefault(0);
    scheme.Add<TFSArray>("parent_ids", "Id родительских кошельков").SetElement<TFSString>();
    const NDrive::IServer* serverImpl = VerifyDynamicCast<const NDrive::IServer*>(&server);
    scheme.Add<TFSVariants>("accounts", "Кошельки").SetVariants(serverImpl->GetDriveAPI()->GetBillingManager().GetKnownAccounts()).SetMultiSelect(true);
    scheme.Add<TFSNumeric>("part_size", "Пакет для частичной выручки").SetDefault(200);
    return scheme;
}

bool TRTBillingCompilation::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    AccountFilter.WithParent = true;
    return NJson::ParseField(jsonInfo, "offer_filter", OfferFilter)
        && NJson::ParseField(jsonInfo, "update_period", UpdatePeriod, true)
        && NJson::ParseField(jsonInfo, "first_event_id", FirstEventId, false)
        && NJson::ParseField(jsonInfo, "parent_ids", AccountFilter.ParentIds)
        && NJson::ParseField(jsonInfo, "accounts", AccountFilter.Accounts)
        && NJson::ParseField(jsonInfo, "part_size", PartSize, false)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTBillingCompilation::DoSerializeToJson() const {
    auto result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "offer_filter", OfferFilter);
    NJson::InsertField(result, "update_period", UpdatePeriod);
    NJson::InsertField(result, "first_event_id", FirstEventId);
    NJson::InsertField(result, "parent_ids", AccountFilter.ParentIds);
    NJson::InsertField(result, "accounts", AccountFilter.Accounts);
    NJson::InsertField(result, "part_size", PartSize);
    return result;
}

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTAddBillingWallets> TRTAddBillingWallets::Registrator(TRTAddBillingWallets::GetTypeName());
TRTAddBillingWalletsState::TFactory::TRegistrator<TRTAddBillingWalletsState> TRTAddBillingWalletsState::Registrator(TRTAddBillingWallets::GetTypeName());

TString TRTAddBillingWalletsState::GetType() const {
    return TRTAddBillingWallets::GetTypeName();
}

TExpectedState TRTAddBillingWallets::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    if (!server.GetDriveAPI()->HasBillingManager()) {
        ERROR_LOG << "Undefined billing manager" << Endl;
        return nullptr;
    }
    const TBillingManager& billingManager = server.GetDriveAPI()->GetBillingManager();

    auto state = dynamic_cast<const TRTAddBillingWalletsState*>(stateExt.Get());
    THolder<TRTBillingCompilationState> newState = MakeHolder<TRTBillingCompilationState>();
    ui64 startEventId = state ? state->GetLastEventId() : FirstEventId;
    ui64 lastEventId = startEventId;
    newState->SetLastEventId(lastEventId);

    const ui64 historyIdCursor = billingManager.GetCompiledBills().GetLockedMaxEventId();

    TSet<TString> compiledBills;
    auto session = billingManager.BuildSession(false);
    while (compiledBills.size() < UpdatePartSize) {
        auto optionalEvents = billingManager.GetCompiledBills().GetEvents(lastEventId + 1, session, NSQL::TQueryOptions(ReadPartSize));
        if (!optionalEvents) {
            return MakeUnexpected<TString>("cannot GetEvents: " + session.GetStringReport());
        }

        if (optionalEvents->empty()) {
            break;
        }

        for (auto&& ev : *optionalEvents) {
            if (ev.GetHistoryEventId() > historyIdCursor) {
                break;
            }
            lastEventId = ev.GetHistoryEventId();
            if (ev.GetBillingWallets()) {
                compiledBills.emplace(ev.GetSessionId());
            }
        }
    }
    auto bills = billingManager.GetCompiledBills().GetBillsFromDB(MakeVector(compiledBills), session);
    if (!bills) {
        return MakeUnexpected(session.GetStringReport());
    }

    TMap<TString, TSet<TString>> sessionWallets;
    for (auto&& bill : *bills) {
        auto wallets = bill.GetBillingWallets();
        sessionWallets[bill.GetSessionId()].insert(wallets.begin(), wallets.end());
    }

    NStorage::ITableAccessor::TPtr table = billingManager.GetCompiledBills().GetDatabase().GetTable(billingManager.GetCompiledBills().GetTableName());
    auto transaction = session.GetTransaction();

    for (auto&& [id, wallets] : sessionWallets) {
        NSQL::TQueryOptions unique;
        unique.SetGenericCondition("history_event_id", TRange<ui64>(startEventId + 1, lastEventId));
        unique.AddGenericCondition("session_id", id);
        NStorage::TTableRecord newRecord;
        newRecord.Set("billing_wallets", JoinSeq(",", wallets));

        auto result = table->UpdateRow(unique.PrintConditions(*transaction), newRecord.BuildSet(*transaction), transaction);
        if (!result || !result->IsSucceed()) {
            return MakeUnexpected(session.GetStringReport());
        }
    }

    newState->SetLastEventId(lastEventId);
    if (!session.Commit()) {
        return MakeUnexpected(session.GetStringReport());
    }
    return newState;
}

NDrive::TScheme TRTAddBillingWallets::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("first_event_id", "Начать с").SetDefault(0);
    scheme.Add<TFSNumeric>("read_part_size", "Размер пакета для чтения").SetDefault(5000);
    scheme.Add<TFSNumeric>("update_part_size", "Размер пакета для записи").SetDefault(100);
    return scheme;
}

bool TRTAddBillingWallets::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return NJson::ParseField(jsonInfo, "first_event_id", FirstEventId, false)
        && NJson::ParseField(jsonInfo, "read_part_size", ReadPartSize, false)
        && NJson::ParseField(jsonInfo, "update_part_size", UpdatePartSize, false)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTAddBillingWallets::DoSerializeToJson() const {
    auto result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "first_event_id", FirstEventId);
    NJson::InsertField(result, "read_part_size", ReadPartSize);
    NJson::InsertField(result, "update_part_size", UpdatePartSize);
    return result;
}
