#include "exports.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/billing/accounts/limited.h>
#include <drive/backend/billing/exports/yt_nodes.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/doc_packages/manager.h>

#include <library/cpp/yson/node/node_io.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/types/uuid.h>

#include <util/string/join.h>

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTFiscalExports> TRTFiscalExports::Registrator(TRTFiscalExports::GetTypeName());
TRTFiscalExportState::TFactory::TRegistrator<TRTFiscalExportState> TRTFiscalExportState::Registrator(TRTFiscalExports::GetTypeName());

IRTRegularBackgroundProcess::TFactory::TRegistrator<TB2BReportExports> TB2BReportExports::Registrator(TB2BReportExports::GetTypeName());
TRTFiscalExportState::TFactory::TRegistrator<TB2BExportState> TB2BExportState::Registrator(TB2BReportExports::GetTypeName());

TString TRTFiscalExportState::GetType() const {
    return TRTFiscalExports::GetTypeName();
}

TString TB2BExportState::GetType() const {
    return TB2BReportExports::GetTypeName();
}

NDrive::TScheme TRTFiscalExports::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("export_type", "Тип экспорта").SetVariants({"income", "refund", "b2b_income", "b2b_refund"});
    scheme.Add<TFSString>("yt_cluster", "Кластер YT");
    scheme.Add<TFSString>("yt_dir", "Директория для экспорта");
    scheme.Add<TFSString>("client_id", "ID клиента для баланса. Если не указан, будет устанавливаться id родительского кошелька");
    scheme.Add<TFSNumeric>("parent_id", "Родительский кошелек");
    scheme.Add<TFSBoolean>("with_user_info", "Выгрузить данные о пользователе").SetDefault(false);
    scheme.Add<TFSBoolean>("with_bonus_accounts", "Подклеить все операции с бонусами").SetDefault(false);
    scheme.Add<TFSNumeric>("first_instant", "Выгружать данные не младше этой секунды");
    scheme.Add<TFSBoolean>("use_op_info", "Использовать ts операции для выбора таблицы" ).SetDefault(false);
    scheme.Add<TFSVariants>("accounts", "Aккаунты для отгрузки").SetMultiSelect(true).SetReference("accounts");
    scheme.Add<TFSVariants>("billing_types", "Виды платежей").InitVariants<EBillingType>().SetMultiSelect(true);
    scheme.Add<TFSVariants>("export_policy", "Как разбивать таблицы").SetVariants({"days", "month"});
    return scheme;
}

bool TRTFiscalExports::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING(jsonInfo, "yt_cluster", YTCluster);
    JREAD_STRING(jsonInfo, "yt_dir", YTDir);
    JREAD_STRING_OPT(jsonInfo, "client_id", ClientId);
    JREAD_STRING_OPT(jsonInfo, "export_type", ExportType);
    JREAD_INT_OPT(jsonInfo, "parent_id", ParentId);
    JREAD_INSTANT_OPT(jsonInfo, "first_instant", FirstInstant);
    JREAD_STRING_OPT(jsonInfo, "export_policy", ExportPolicy);
    JREAD_BOOL(jsonInfo, "with_user_info", WithUserInfo);
    JREAD_BOOL_OPT(jsonInfo, "use_op_info", UseOperationTs);
    JREAD_BOOL_OPT(jsonInfo, "with_bonus_accounts", WithBonusAccounts);

    TJsonProcessor::ReadContainer(jsonInfo, "accounts", Accounts);
    TJsonProcessor::ReadContainer(jsonInfo, "billing_types", BillingTypes);

    AccountFilter.WithBonusAccounts = WithBonusAccounts;
    AccountFilter.WithParent = true;
    if (ParentId > 0) {
        AccountFilter.ParentIds = { ParentId };
    }
    AccountFilter.Accounts = Accounts;

    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    if (ExportType == "income") {
        SetTableName("compiled_bills");
    } else if (ExportType == "refund") {
        SetTableName("compiled_refunds");
    }
    return true;
}

NJson::TJsonValue TRTFiscalExports::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["yt_dir"] = YTDir;
    result["yt_cluster"] = YTCluster;
    result["client_id"] = ClientId;
    result["parent_id"] = ParentId;
    result["with_user_info"] = WithUserInfo;
    result["use_op_info"] = UseOperationTs;
    result["with_bonus_accounts"] = WithBonusAccounts;
    result["export_type"] = ExportType;
    if (FirstInstant != TInstant::Zero()) {
        result["first_instant"] = FirstInstant.Seconds();
    }
    if (ExportPolicy) {
        result["export_policy"] = ExportPolicy;
    }

    TJsonProcessor::WriteContainerArray(result, "accounts", Accounts);
    TJsonProcessor::WriteContainerArrayStrings(result, "billing_types", BillingTypes);
    return result;
}


bool TRTFiscalExports::ProcessRecords(const TRecordsSet& records, const NDrive::IServer& server, ui64& lastEventId, TMessagesCollector& errors) const {
    TUnistatSignalsCache::SignalAdd("billing_exports-" + GetRTProcessName(), "records", 0);
    TUnistatSignalsCache::SignalAdd("billing_exports-" + GetRTProcessName(), "errors", 0);

    auto availableAccounts = server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetAccountsChildren(AccountFilter);
    if (!availableAccounts) {
        errors.AddMessage("TRTFiscalExports", "Cannot fetch accounts");
        return false;
    }

    try {
        auto ytClient = NYT::CreateClient(YTCluster);
        ytClient->Create(YTDir + "/transactions", NYT::NT_MAP, NYT::TCreateOptions().Recursive(true).IgnoreExisting(true));
        THolder<IYTWritersSet> writers;
        if (ExportPolicy == "month") {
            writers = MakeHolder<TYTWritersSet<TTableSelectorMonth>>(ytClient, YTDir + "/transactions");
        } else {
            writers = MakeHolder<TYTWritersSet<TTableSelectorDay>>(ytClient, YTDir + "/transactions");
        }

        for (auto&& record : records) {
            TObjectEvent<NDrive::NBilling::TCompiledImpl> compiledImpl;
            if (!TBaseDecoder::DeserializeFromTableRecord(compiledImpl, record)) {
                continue;
            }

            TObjectEvent<TCompiledBill> compiledBill;
            bool isCompiledBill = TBaseDecoder::DeserializeFromTableRecord(compiledBill, record);

            lastEventId = Max<ui64>(compiledImpl.GetHistoryEventId(), lastEventId);

            if (!BillingTypes.contains(compiledImpl.GetBillingType())) {
                continue;
            }

            TInstant ts = compiledImpl.GetHistoryInstant();
            const TString sessionId = compiledImpl.GetSessionId();
            const TString userId = compiledImpl.GetUserId();

            if (compiledImpl.GetBill() == 0 && (!isCompiledBill || compiledBill.GetCashback() == 0)) {
                continue;
            }

            if (ts < FirstInstant) {
                continue;
            }

            const TString type = NBillingExports::GetOrderType(compiledImpl.GetBillingType());
            for (auto&& item : compiledImpl.GetDetails().GetItems()) {
                auto itAccountParent = availableAccounts->find(item.GetUniqueName());
                if (itAccountParent == availableAccounts->end()) {
                    continue;
                }
                if (item.GetSum() == 0) {
                    continue;
                }
                ui64 promocodeSum = 0;
                TString paymentType = ToString(item.GetType());
                ui64 totalSum = item.GetSum();

                if (item.GetType() == NDrive::NBilling::EAccount::Bonus) {
                    promocodeSum = item.GetSum();
                    paymentType = "card";
                    totalSum = 0;
                }

                if (item.GetType() == NDrive::NBilling::EAccount::YAccount) {
                    paymentType = "yandex_account_withdraw";
                }
                if (item.GetType() == NDrive::NBilling::EAccount::MobilePayment) {
                    paymentType = "card";
                }

                if (TBillingGlobals::CashbackBillingTypes.contains(compiledImpl.GetBillingType())) {
                    promocodeSum = item.GetSum();
                    paymentType = "yandex_account_topup";
                    totalSum = 0;
                }

                TString clientId = ClientId;
                if (!clientId) {
                    NDrive::NBilling::IBillingAccount::TPtr parentAccount = server.GetDriveAPI()->GetBillingManager().GetAccountsManager().GetAccountById(itAccountParent->second);
                    if (parentAccount && parentAccount->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>() && parentAccount->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>()->HasBalanceInfo()) {
                        clientId = ToString(parentAccount->GetRecordAs<NDrive::NBilling::TLimitedAccountRecord>()->GetBalanceInfoRef().GetClientId());
                    }
                }
                if (!clientId) {
                    clientId = ToString(itAccountParent->second);
                }

                NYT::TNode itemNode;
                if (ExportType.StartsWith("b2b_")) {
                    itemNode = NBillingExports::CreateB2BTransactionNode(/*service_order_id=*/sessionId,
                                                                             clientId,
                                                                             /*total_sum=*/totalSum,
                                                                             /*promocode_sum=*/promocodeSum,
                                                                             ts,
                                                                             /*type=*/type,
                                                                             (ExportType == "b2b_refund"),
                                                                             /*external_id=*/item.GetTransactionId(),
                                                                             /*payment_type=*/paymentType);
                } else {
                    if (TBillingGlobals::CashbackBillingTypes.contains(compiledImpl.GetBillingType())) {
                        itemNode = NBillingExports::CreateCashbackTransactionNode(/*service_order_id=*/sessionId,
                                                                             clientId,
                                                                             /*total_sum=*/totalSum,
                                                                             /*promocode_sum=*/promocodeSum,
                                                                             ts,
                                                                             /*type=*/type,
                                                                             (ExportType == "refund"),
                                                                             /*external_id=*/item.GetTransactionId(),
                                                                             paymentType);
                    } else {
                        itemNode = NBillingExports::CreateTransactionNode(/*service_order_id=*/sessionId,
                                                                             clientId,
                                                                             /*total_sum=*/totalSum,
                                                                             /*promocode_sum=*/promocodeSum,
                                                                             ts,
                                                                             /*type=*/type,
                                                                             (ExportType == "refund"),
                                                                             /*external_id=*/item.GetTransactionId(),
                                                                             /*payment_type=*/paymentType);
                    }
                }
                if (WithUserInfo) {
                    itemNode["drive_user_id"] = userId;
                }
                TUnistatSignalsCache::SignalAdd("billing_exports-" + GetRTProcessName(), "records", 1);
                writers->GetWriter(UseOperationTs ? ts : StartInstant)->AddRow(itemNode);
            }
            if (isCompiledBill && compiledBill.GetCashback()) {
                NYT::TNode itemNode = NBillingExports::CreateCashbackTransactionNode(/*service_order_id=*/sessionId,
                                                                             ClientId,
                                                                             /*total_sum=*/0,
                                                                             /*promocode_sum=*/compiledBill.GetCashback(),
                                                                             ts,
                                                                             /*type=*/type,
                                                                             false,
                                                                             /*external_id=*/sessionId);
                if (WithUserInfo) {
                    itemNode["drive_user_id"] = userId;
                }
                TUnistatSignalsCache::SignalAdd("billing_exports-" + GetRTProcessName(), "cashback-records", 1);
                writers->GetWriter(UseOperationTs ? ts : StartInstant)->AddRow(itemNode);
            }
        }
        writers->Finish();
    } catch (const std::exception& e) {
        errors.AddMessage("YTError", FormatExc(e));
        TUnistatSignalsCache::SignalAdd("billing_exports-" + GetRTProcessName(), "errors", 1);
        return false;
    }
    return true;
}

NDrive::TScheme TB2BReportExports::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("accounts", "Aккаунты для отгрузки").SetMultiSelect(true).SetReference("accounts");
    scheme.Add<TFSVariants>("notifier", "Нотификатор").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSString>("doc_constructor", "Конструктор документа");
    scheme.Add<TFSString>("pack_constructor", "Конструктор пакета");
    scheme.Add<TFSString>("doc_name", "Имя итогового файла");
    scheme.Add<TFSNumeric>("first_instant", "Дата первого репорта").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSDuration>("report_interval", "Регулярность репортов");
    return scheme;
}

bool TB2BReportExports::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING(jsonInfo, "notifier", NotifierName);
    JREAD_STRING(jsonInfo, "doc_constructor", DocumentConstructorName);
    JREAD_STRING_OPT(jsonInfo, "pack_constructor", PackConstructorName);
    JREAD_STRING(jsonInfo, "doc_name", DocumentFileName);
    JREAD_INSTANT(jsonInfo, "first_instant", FirstReportInstant);
    JREAD_DURATION(jsonInfo, "report_interval", ReportInterval);

    TJsonProcessor::ReadContainer(jsonInfo, "accounts", Accounts);
    return TBase::DoDeserializeFromJson(jsonInfo);

}

NJson::TJsonValue TB2BReportExports::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    JWRITE(result, "notifier", NotifierName);
    JWRITE(result, "doc_constructor", DocumentConstructorName);
    JWRITE(result, "pack_constructor", PackConstructorName);
    JWRITE(result, "doc_name", DocumentFileName);
    JWRITE_INSTANT(result, "first_instant", FirstReportInstant);
    JWRITE_DURATION(result, "report_interval", ReportInterval);
    TJsonProcessor::WriteContainerArray(result, "accounts", Accounts);
    return result;
}

TExpectedState TB2BReportExports::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    THolder<TB2BExportState> result = MakeHolder<TB2BExportState>();
    const TB2BExportState* state = dynamic_cast<const TB2BExportState*>(stateExt.Get());

    TInstant reportInstant;
    if (state) {
        reportInstant = state->GetLastInstant();
    } else {
        reportInstant = FirstReportInstant;
    }
    result->SetLastInstant(reportInstant);

    TInstant nextReportDate = reportInstant + ReportInterval;
    if (nextReportDate + TDuration::Minutes(10) > StartInstant) {
        return result.Release();
    }

    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();

    TVector<TString> sessions;

    {
        const auto& compiledBills = server.GetDriveAPI()->GetBillingManager().GetCompiledBills();
        auto session = server.GetDriveAPI()->GetBillingManager().BuildSession(true);
        auto events = compiledBills.GetEvents({}, { reportInstant }, session);
        if (!events) {
            return MakeUnexpected<TString>("cannot GetEventsAll from CompiledBills " + session.GetStringReport());
        }

        for (auto&& bill : *events) {
            if (bill.GetFinal()) {
                sessions.emplace_back(bill.GetSessionId());
            }
        }

        auto fullCompiledBills = compiledBills.GetFullBillsFromDB(sessions, session);
        if (!fullCompiledBills) {
            return MakeUnexpected<TString>("cannot GetFillBills from CompiledBills " + session.GetStringReport());
        }

        for (auto&& [sessionId, bill] : *fullCompiledBills) {
            if (bill.GetFinalTime() > nextReportDate) {
                break;
            }
            if (bill.GetBillingType() != EBillingType::CarUsage) {
                continue;
            }
            auto accounts = bill.GetAccounts();
            TSet<TString> intersection;
            std::set_intersection(accounts.begin(), accounts.end(),
                Accounts.begin(), Accounts.end(),
                std::inserter(intersection, intersection.begin()));

            if (!intersection.empty()) {
                sessions.emplace_back(bill.GetSessionId());
            }
        }
    }

    NJson::TJsonValue docInputs;
    docInputs["storage"] = "mds";
    docInputs["document_name"] = PackConstructorName;

    NJson::TJsonValue& parameters = docInputs.InsertValue("document_parameters_" + DocumentConstructorName, NJson::JSON_ARRAY);
    for (auto&& sessionId : sessions) {
        NJson::TJsonValue sData;
        sData["session_id"] = sessionId;
        parameters.AppendValue(std::move(sData));
    }

    if (sessions.empty()) {
        result->SetLastInstant(nextReportDate);
        return result.Release();
    }

    NJson::TJsonValue notifierJson;
    notifierJson["name"] = DocumentFileName + reportInstant.FormatLocalTime("-%Y-%m-%d");
    notifierJson["notifier"] = NotifierName;
    notifierJson["description"] = "Отчет.Драйв";
    docInputs["notifiers"].AppendValue(notifierJson);

    TQueuedDocument newDocument;
    if (!newDocument.SetId(NUtil::CreateUUID()).SetStatus(EAssemblerStage::NotReady).SetAuthor(GetRobotUserId()).SetInputParameters(docInputs)) {
        ERROR_LOG << "incorrect doc input" << Endl;
        return result.Release();
    }

    auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
    if (!server.GetDocumentsManager() || !server.GetDocumentsManager()->UpsertQueuedDocument(newDocument, GetRobotUserId(), session) || !session.Commit()) {
        ERROR_LOG << session.GetStringReport() << Endl;
        return result.Release();
    }
    result->SetLastInstant(nextReportDate);
    return result.Release();
}
