#include "processor.h"

#include <util/generic/serialized_enum.h>

TBlacklistBanActualizer::TFactory::TRegistrator<TBlacklistBanActualizer> TBlacklistBanActualizer::Registrator("blacklist_bans_actualizer");

TExpectedState TBlacklistBanActualizer::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    IUserRegistrationManager::TBlacklistOptions options;
    options.Traits = Traits;
    options.Direct = Direct;
    options.Reverse = Reverse;
    options.SetComment = SetComment;
    options.VerifiedDeviceIdOnly = VerifiedDeviceIdOnly;
    server->GetUserRegistrationManager()->ActualizeBlacklistBans(GetRobotUserId(), BlacklistedTags, SimilarProfilesTag, options);

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TBlacklistBanActualizer::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSString>("blacklisted_tags", "С какими тегами брать пользователей").SetRequired(true);
    scheme.Add<TFSString>("similar_profiles_tag", "Каким тегом маркировать похожих").SetRequired(true);
    scheme.Add<TFSVariants>("mark_policy", "Политики поиска похожих").InitVariants<NUserConnections::ESimilarityTraits>().SetMultiSelect(true);
    scheme.Add<TFSBoolean>("direct").SetDefault(Direct);
    scheme.Add<TFSBoolean>("reverse").SetDefault(Reverse);
    scheme.Add<TFSBoolean>("set_comment").SetDefault(SetComment);
    scheme.Add<TFSBoolean>("verified_device_id_only").SetDefault(VerifiedDeviceIdOnly);

    return scheme;
}

NJson::TJsonValue TBlacklistBanActualizer::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "blacklisted_tags", JoinStrings(BlacklistedTags, ","));
    TJsonProcessor::Write(result, "similar_profiles_tag", SimilarProfilesTag);
    {
        NJson::TJsonValue markPolicy = NJson::JSON_ARRAY;
        for (auto&& trait : GetEnumAllValues<NUserConnections::ESimilarityTraits>()) {
            if (Traits & trait) {
                markPolicy.AppendValue(ToString(trait));
            }
        }
        result["mark_policy"] = std::move(markPolicy);
    }

    NJson::InsertField(result, "direct", Direct);
    NJson::InsertField(result, "reverse", Reverse);
    NJson::InsertField(result, "set_comment", SetComment);
    NJson::InsertField(result, "verified_device_id_only", VerifiedDeviceIdOnly);

    return result;
}

bool TBlacklistBanActualizer::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    {
        TString blacklistedTagsStr;
        JREAD_STRING(jsonInfo, "blacklisted_tags", blacklistedTagsStr);
        BlacklistedTags = SplitString(blacklistedTagsStr, ",");
    }

    JREAD_STRING(jsonInfo, "similar_profiles_tag", SimilarProfilesTag);

    if (jsonInfo.Has("mark_policy") && jsonInfo["mark_policy"].IsArray()) {
        Traits = 0;
        auto traits = jsonInfo["mark_policy"].GetArray();
        for (auto&& element : traits) {
            NUserConnections::ESimilarityTraits traitValue;
            if (!element.IsString() || !TryFromString(element.GetString(), traitValue)) {
                return false;
            }
            Traits |= traitValue;
        }
    } else {
        return false;
    }

    return
        NJson::ParseField(jsonInfo["direct"], Direct) &&
        NJson::ParseField(jsonInfo["reverse"], Reverse) &&
        NJson::ParseField(jsonInfo["set_comment"], SetComment) &&
        NJson::ParseField(jsonInfo["verified_device_id_only"], VerifiedDeviceIdOnly);
}
