#pragma once

#include <drive/backend/rt_background/common/config.h>
#include <drive/backend/rt_background/manager/manager.h>

#include <drive/backend/tags/tags_filter.h>

#include <drive/library/cpp/scheme/scheme.h>

class TRTCarCounter: public IRTCarsProcess {
private:
    using TBase = IRTCarsProcess;
    R_READONLY(TSet<TDuration>, IdleLimits);

private:
    static TFactory::TRegistrator<TRTCarCounter> Registrator;
    THolder<TUnistatSignal<double>> Signal;
    TVector<TAtomicSharedPtr<TUnistatSignal<double>>> SignalIdleLimit;

public:

    virtual bool IsSimultaneousProcess() const override {
        return true;
    }

    virtual TString GetType() const override {
        return "car_counter";
    }

    using TBase::TBase;

    virtual bool DoStart(const TRTBackgroundProcessContainer& container) override {
        if (!TBase::DoStart(container)) {
            return false;
        }
        Signal.Reset(new TUnistatSignal<double>({
            "frontend-" + container.GetName(),
        }, EAggregationType::LastValue, "attx"));
        for (auto&& i : GetIdleLimits()) {
            SignalIdleLimit.emplace_back(new TUnistatSignal<double>({
                "frontend-" + container.GetName() + "-" + ToString(i.Seconds()),
            }, EAggregationType::LastValue, "attx"));
        }
        return true;
    }

    virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const override;

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual NJson::TJsonValue DoSerializeToJson() const override {
        NJson::TJsonValue result = TBase::DoSerializeToJson();
        JWRITE(result, "idle_limits", JoinSeq(",", IdleLimits));
        return result;
    }

    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override {
        if (!TBase::DoDeserializeFromJson(jsonInfo)) {
            return false;
        }
        TString idleLimits;
        JREAD_STRING_OPT(jsonInfo, "idle_limits", idleLimits);
        try {
            StringSplitter(idleLimits).SplitBySet(", ").SkipEmpty().ParseInto(&IdleLimits);
        } catch (const std::exception& e) {
            ERROR_LOG << "cannot deserialize RTCarCounter: " << FormatExc(e) << Endl;
            return false;
        }
        return true;
    }

};
