#include "config.h"

#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/database/drive_api.h>

#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/algorithm/iterator.h>

TRTCarLockers::TFactory::TRegistrator<TRTCarLockers> TRTCarLockers::Registrator("car_lockers");

namespace {
    TMutex MutexLockerSignals;
    TSet<TString> LockerSignals;
}

TExpectedState TRTCarLockers::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    TVector<TTaggedDevice> devices;
    if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetCustomObjectsFromCache(devices, context.GetFilteredCarIds())) {
        return MakeUnexpected<TString>({});
    }
    if (NeedStatusCount) {
        TMap<TString, ui32> statesNewCount;
        {
            auto states = server.GetDriveAPI()->GetStateFiltersDB()->GetObjectStates();
            auto it = states.begin();
            for (auto&& i : devices) {
                if (Advance(it, states.end(), i.GetId())) {
                    ++statesNewCount[it->second];
                }
            }
        }
        for (auto&& i : statesNewCount) {
            TUnistatSignalsCache::SignalLastX("status-count" + (CommonSignalPrefix ? ("-" + CommonSignalPrefix) : ""), i.first, i.second);
        }
    }

    TMap<TString, ui32> tagLockers;
    TMap<TString, ui32> tagLockersPerformed;
    for (auto&& i : devices) {
        TMap<TString, ui32> deviceTags;
        TMap<TString, ui32> deviceTagsPerforming;
        i32 maxPriority = 0;
        for (auto&& tag : i.GetTags()) {
            if (tag->GetTagPriority(0) <= 0) {
                continue;
            }
            if (tag->GetTagPriority(0) > maxPriority) {
                maxPriority = tag->GetTagPriority(0);
                deviceTags.clear();
                ++deviceTags[tag->GetName()];
                if (!!tag->GetPerformer()) {
                    ++deviceTagsPerforming[tag->GetName()];
                }
            } else if (tag->GetTagPriority(0) == maxPriority) {
                ++deviceTags[tag->GetName()];
                if (!!tag->GetPerformer()) {
                    ++deviceTagsPerforming[tag->GetName()];
                }
            }
        }
        if (NeedLockersCount) {
            for (auto&& tagCounter : deviceTags) {
                tagLockers[tagCounter.first] += tagCounter.second;
            }
        }
        if (NeedLockersPerformingCount) {
            for (auto&& tagCounter : deviceTagsPerforming) {
                tagLockersPerformed[tagCounter.first] += tagCounter.second;
            }
        }
    }
    for (auto&& i : tagLockers) {
        TUnistatSignalsCache::SignalLastX(i.first, "locker", i.second);
    }
    for (auto&& i : tagLockersPerformed) {
        TUnistatSignalsCache::SignalLastX(i.first, "locker-performing", i.second);
    }
    TGuard<TMutex> g(MutexLockerSignals);
    for (auto&& i : LockerSignals) {
        if (NeedLockersCount) {
            if (!tagLockers.contains(i)) {
                TUnistatSignalsCache::SignalLastX(i, "locker", 0);
            }
        }
        if (NeedLockersPerformingCount) {
            if (!tagLockersPerformed.contains(i)) {
                TUnistatSignalsCache::SignalLastX(i, "locker-performing", 0);
            }
        }
    }
    TSet<TString> lockerSignals;
    for (auto&& i : tagLockers) {
        lockerSignals.emplace(i.first);
    }
    LockerSignals = lockerSignals;
    return new IRTBackgroundProcessState();
}

NDrive::TScheme TRTCarLockers::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("common_prefix", "общий префикс сигналов").SetRequired(false);
    scheme.Add<TFSBoolean>("need_status_count", "Требуется сигнал по состояниям").SetDefault(true);
    scheme.Add<TFSBoolean>("need_lockers_count", "Требуется сигнал по блокирующим объектам").SetDefault(true);
    scheme.Add<TFSBoolean>("need_lockers_perform", "Требуется сигнал по выполняющимся объектам").SetDefault(true);
    return scheme;
}
