#include "process.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/abstract/base.h>
#include <drive/backend/cars/status/state_filters.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/logging/events.h>

#include <rtline/library/json/cast.h>

TExpectedState TCarStatusWatcherProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    Y_UNUSED(state);
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const TDevicesSnapshotManager& snapshots = server.GetSnapshotsManager();
    const TDriveAPI* api = server.GetDriveAPI();
    if (!api) {
        return MakeUnexpected<TString>("DriveAPI is missing");
    }
    const TStateFiltersDB* states = api->GetStateFiltersDB();
    if (!states) {
        return MakeUnexpected<TString>("StateFiltersDB is missing");
    }
    auto statuses = states->GetObjectStates();
    if (statuses.empty()) {
        return MakeUnexpected<TString>("StateFiltersDB is empty");
    }
    TDeviceLocationOptions dlo = snapshots.GetLocationOptions();
    NJson::TJsonValue result = NJson::JSON_ARRAY;
    for (auto&&[id, status] : statuses) {
        NJson::TJsonValue object;
        object["id"] = id;
        object["status"] = status;
        auto snapshot = snapshots.GetSnapshot(id);
        auto location = snapshot.GetLocation(TDuration::Max(), dlo);
        if (location) {
            object["location"] = location->ToJson();
        }
        const auto& tags = snapshot.GetTagsInPoint();
        if (!tags.empty()) {
            object["location_tags"] = NJson::ToJson(tags);
        }
        result.AppendValue(std::move(object));
    }
    NDrive::TEventLog::Log("ObjectStates", result);
    return MakeAtomicShared<IRTBackgroundProcessState>();
}

TCarStatusWatcherProcess::TFactory::TRegistrator<TCarStatusWatcherProcess> TCarStatusWatcherProcess::Registrator(TCarStatusWatcherProcess::GetTypeName());
