#include "processor.h"

TChatRegularEvolutions::TFactory::TRegistrator<TChatRegularEvolutions> TChatRegularEvolutions::Registrator(GetTypeName());

TExpectedState TChatRegularEvolutions::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    if (!server) {
        return MakeUnexpected<TString>({});
    }
    TUserPermissions::TPtr robotUserPermissions = server->GetDriveAPI()->GetUserPermissions(GetRobotUserId(), TUserPermissionsFeatures());
    if (!robotUserPermissions) {
        return MakeUnexpected<TString>("cannot get permissions for user " + GetRobotUserId());
    }
    ChatFilter.SetPermissions(robotUserPermissions);
    TVector<TFilteredSupportChat> filteredChats;
    TMessagesCollector errors;
    if (!ChatFilter.GetSupportRequests(server, filteredChats, TInstant::Zero(), GetRobotUserId(), errors)) {
        return MakeUnexpected<TString>(errors.GetStringReport());
    }

    TVector<TString> tagsForRemoval;
    auto chatRobots = server->GetChatRobots();
    for (auto&& entry : filteredChats) {
        const TString& userId = entry.GetTag().GetObjectId();
        if (!ShardingPolicy.CheckMatching(userId)) {
            continue;
        }
        auto impl = entry.GetTag().GetTagAs<TSupportChatTag>();
        if (!impl) {
            continue;
        }

        TString chatRobotId;
        TString chatRobotTopic;
        IChatRobot::ParseTopicLink(impl->GetTopicLink(), chatRobotId, chatRobotTopic);

        auto chatRobotIt = chatRobots.find(chatRobotId);
        if (chatRobotIt == chatRobots.end() || !chatRobotIt->second) {
            continue;
        }

        TString stepUser = NodeUser;
        if (stepUser == "$performer$") {
            stepUser = impl->GetPerformer();
        } else if (stepUser == "$chat_owner$") {
            stepUser = userId;
        } else {
            stepUser = GetRobotUserId();
        }

        if (!chatRobotIt->second->MoveToStep(userId, chatRobotTopic, NodeName, nullptr, true, "", stepUser, entry.HasLastMessage() ? entry.GetLastMessageRef().GetHistoryEventId() : 0)) {
            ERROR_LOG << "could not move (" << userId << ", " << chatRobotTopic << ") to " << NodeName << Endl;
            continue;
        }

        if (DoRemoveTag) {
            tagsForRemoval.push_back(entry.GetTag().GetTagId());
        }
    }

    if (server->GetDriveAPI()) {
        TSet<TString> removalPart;
        for (auto&& tag : tagsForRemoval) {
            removalPart.insert(tag);
            if (removalPart.size() == 1000) {
                auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTagsSimple(removalPart, GetRobotUserId(), session, true) || !session.Commit()) {
                    ERROR_LOG << "could not remove tags" << Endl;
                    return MakeUnexpected<TString>({});
                }
                removalPart.clear();
            }
        }
        if (!removalPart.empty()) {
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTagsSimple(removalPart, GetRobotUserId(), session, true) || !session.Commit()) {
                ERROR_LOG << "could not commit tags removal session" << Endl;
                return MakeUnexpected<TString>({});
            }
        }
    }

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TChatRegularEvolutions::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSStructure>("chat_filter", "Фильтр чатов").SetStructure<NDrive::TScheme>(TChatSupportRequestsFilter::GetScheme());
    scheme.Add<TFSStructure>("sharding_policy", "Условия для выбора объекта").SetStructure<NDrive::TScheme>(TObjectSharding::GetScheme());
    scheme.Add<TFSString>("node_name", "В какую вершину переводить чат");
    scheme.Add<TFSString>("node_user", "От кого отправлять сообщения (robot-server, $performer$, $chat_owner$)");
    scheme.Add<TFSBoolean>("do_remove_tag", "Снимать тег обращения");

    return scheme;
}

NJson::TJsonValue TChatRegularEvolutions::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    JWRITE(result, "chat_filter", ChatFilter.SerializeToJson());
    JWRITE(result, "sharding_policy", ShardingPolicy.SerializeToJson());
    JWRITE(result, "node_name", NodeName);
    JWRITE(result, "node_user", NodeUser);
    JWRITE(result, "do_remove_tag", DoRemoveTag);

    return result;
}

bool TChatRegularEvolutions::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    if (jsonInfo.Has("chat_filter") && !ChatFilter.DeserializeFromJson(jsonInfo["chat_filter"])) {
        return false;
    }
    if (jsonInfo.Has("sharding_policy") && !ShardingPolicy.DeserializeFromJson(jsonInfo["sharding_policy"])) {
        return false;
    }
    JREAD_STRING(jsonInfo, "node_name", NodeName);
    JREAD_STRING_OPT(jsonInfo, "node_user", NodeUser);
    JREAD_BOOL_OPT(jsonInfo, "do_remove_tag", DoRemoveTag);

    return true;
}
