#include "states_monitoring.h"

#include <drive/backend/chat_robots/abstract.h>

TChatStatesMonitoring::TFactory::TRegistrator<TChatStatesMonitoring> TChatStatesMonitoring::Registrator(GetTypeName());

TExpectedState TChatStatesMonitoring::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    auto chatRobot = server->GetChatRobot(ChatId);
    if (!chatRobot) {
        ERROR_LOG << "Chat robot monitoring failed: no such chat " << ChatId << Endl;
        return new IRTBackgroundProcessState();
    }

    auto usersInNodes = chatRobot->GetUsersInNodes("", StepsToMonitor);
    TMap<TString, size_t> stats;
    for (auto&& userIt : usersInNodes) {
        stats[userIt.second] += 1;
    }

    for (auto&& signalIt : Signals) {
        const auto& step = signalIt.first;
        ui32 value = stats[step];
        signalIt.second->Signal(value);
    }

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TChatStatesMonitoring::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSString>("chat_id", "id чата").SetRequired(true);
    scheme.Add<TFSArray>("steps_to_monitor", "Какие шаги мониторить").SetElement<TFSString>();

    return scheme;
}

NJson::TJsonValue TChatStatesMonitoring::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "chat_id", ChatId);
    NJson::TJsonValue steps = NJson::JSON_ARRAY;
    for (auto&& step : StepsToMonitor) {
        steps.AppendValue(step);
    }
    result["steps_to_monitor"] = std::move(steps);

    return result;
}

bool TChatStatesMonitoring::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_STRING(jsonInfo, "chat_id", ChatId);
    if (!jsonInfo.Has("steps_to_monitor") || !jsonInfo["steps_to_monitor"].IsArray()) {
        return false;
    }
    StepsToMonitor.clear();
    Signals.clear();
    for (auto&& step : jsonInfo["steps_to_monitor"].GetArray()) {
        if (!step.IsString()) {
            return false;
        }
        auto stepStr = step.GetString();
        StepsToMonitor.insert(stepStr);

        THolder<TChatStateSignal> signalPtr;
        signalPtr.Reset(new TChatStateSignal(ChatId, stepStr));
        Signals[stepStr] = std::move(signalPtr);
    }

    return true;
}
