#include "processor.h"

#include <drive/backend/chat_robots/abstract.h>
#include <drive/backend/data/notifications_tags.h>
#include <drive/backend/data/support_tags.h>
#include <drive/backend/data/user_tags.h>

TChatUnreadPushes::TFactory::TRegistrator<TChatUnreadPushes> TChatUnreadPushes::Registrator(GetTypeName());

TExpectedState TChatUnreadPushes::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    TInstant from = TInstant::Now() - CheckMessagesFrom;

    bool isNight = false;
    ui32 dayOffset = Now().Seconds() % (24 * 60 * 60);
    if (NightStart <= NightEnd) {
        isNight = NightStart <= dayOffset && dayOffset <= NightEnd;
    } else {
        isNight = NightStart <= dayOffset || dayOffset <= NightEnd;
    }

    TSet<TString> tagNames;
    auto tagDescrs = server->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetTagsByType(TSupportChatTag::TypeName);
    for (auto&& descr : tagDescrs) {
        auto descrImpl = descr->GetAs<TSupportChatTag::TDescription>();
        if (!descrImpl) {
            continue;
        }
        if (!descrImpl->GetNeedNotification()) {
            continue;
        }
        tagNames.emplace(descr->GetName());
    }

    TVector<TDBTag> chatTags;
    TMap<TString, TInstant> lastNotificationTimes;
    TSet<TString> currentTagPerformers;
    {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(TSet<TString>(), MakeVector(tagNames), chatTags, session)) {
            return MakeUnexpected<TString>("could not restore support chat tags " + session.GetStringReport());
        }
        TSet<TString> userIds;
        TVector<TDBTag> performedTags;
        for (auto&& tag : chatTags) {
            userIds.emplace(tag.GetObjectId());
        }
        TVector<TDBTag> notificationTags;
        if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags(userIds, TVector<TString>({TUserChatNotificationTag::TypeName}), notificationTags, session)) {
            return MakeUnexpected<TString>("could not restore notification tags " + session.GetStringReport());
        }
        for (auto&& tag : notificationTags) {
            auto impl = tag.GetTagAs<TUserChatNotificationTag>();
            if (!impl) {
                continue;
            }
            lastNotificationTimes.emplace(tag.GetObjectId(), impl->GetLastNotifiedAt());
        }
        if (!server->GetDriveAPI()->GetTagsManager().GetDeviceTags().RestorePerformerTags(MakeVector(userIds), performedTags, session)) {
            return MakeUnexpected<TString>("could not restore performer tags " + session.GetStringReport());
        }
        for (auto&& tag : performedTags) {
            currentTagPerformers.emplace(tag->GetPerformer());
        }
    }

    TMap<TString, TSet<TString>> chatSearchIds;
    TMap<TString, TString> userSearchIds;
    TMap<TString, ui64> lastViewedMessage;
    TMap<TString, TString> topicLinkBySearchId;
    auto chatRobots = server->GetChatRobots();
    for (auto&& tag : chatTags) {
        auto impl = tag.GetTagAs<TSupportChatTag>();
        if (!impl || !tagNames.contains(impl->GetName())) {
            continue;
        }
        auto userId = tag.GetObjectId();
        if (isNight && !currentTagPerformers.contains(userId)) {
            continue;
        }
        TString chatId;
        TString topic;
        const TString topicLink = impl->GetTopicLink();
        IChatRobot::ParseTopicLink(topicLink, chatId, topic);
        auto chatRobotIt = chatRobots.find(chatId);
        if (chatRobotIt == chatRobots.end()) {
            continue;
        }
        auto messageId = chatRobotIt->second->GetLastViewedMessageId(userId, topic, userId);
        lastViewedMessage[userId] = Max(lastViewedMessage[userId], messageId);

        TString searchId = chatRobotIt->second->GetChatSearchId(userId, topic);
        chatSearchIds[chatId].emplace(searchId);
        userSearchIds[searchId] = userId;
        topicLinkBySearchId[searchId] = topicLink;
    }

    TMap<TString, TMap<TString, NDrive::NChat::TMessageEvents>> usersChatMessages;
    {
        NDrive::TEntitySession session = server->GetChatEngine()->BuildSession(true);
        for (auto&& [chatId, searchIds] : chatSearchIds) {
            auto chatRobotIt = chatRobots.find(chatId);
            if (chatRobotIt == chatRobots.end()) {
                continue;
            }
            auto chats = chatRobotIt->second->GetChats(searchIds, session);
            if (!chats) {
                return MakeUnexpected<TString>("could not get chats from robot " + chatId + " error: " + session.GetStringReport());
            }
            if (chats->empty()) {
                continue;
            }
            TVector<NDrive::NChat::TChat> chatsToRequest;
            for (auto&& chat : *chats) {
                if (!chat.GetFlag(NDrive::NChat::TChat::EChatFlags::Muted)) {
                    chatsToRequest.push_back(std::move(chat));
                }
            }
            auto chatMessages = chatRobotIt->second->GetUserChatsMessages(chatsToRequest, session, {}, { from }, NDrive::NChat::TMessage::DefaultUserTraits);
            if (!chatMessages) {
                return MakeUnexpected<TString>("could not get chat messages from robot " + chatId + " error: " + session.GetStringReport());
            }
            for (auto&& [searchId, messages] : *chatMessages) {
                auto userId = userSearchIds.find(searchId);
                if (userId != userSearchIds.end()) {
                    usersChatMessages[userId->second][searchId] = std::move(messages);
                }
            }
        }
    }

    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
    for (auto&& [userId, chats] : usersChatMessages) {
        if (!ShardingPolicy.CheckMatching(userId)) {
            continue;
        }
        TString targetTopicLink;
        TInstant earliestUnreadInstant = TInstant::Max();
        for (auto&& [searchId, messages] : chats) {
            for (auto&& message : messages) {
                if (message.GetType() == NDrive::NChat::TMessage::EMessageType::Separator || message.GetHistoryInstant() <= lastNotificationTimes[userId] || message.GetHistoryEventId() <= lastViewedMessage[userId]) {
                    continue;
                }
                if (earliestUnreadInstant > message.GetHistoryInstant()) {
                    earliestUnreadInstant = message.GetHistoryInstant();
                    targetTopicLink = topicLinkBySearchId[searchId];
                }
                break;
            }
        }
        if (earliestUnreadInstant == TInstant::Max()) {
            // no unread messages since last notification
            continue;
        }
        if (earliestUnreadInstant + WaitPeriod <= Now()) {
            auto tagNotify = IJsonSerializableTag::BuildWithComment<TUserChatNotificationTag>("user_chat_notification", "chat push notification tracking");
            {
                auto tagNotifyImpl = Yensured(dynamic_cast<TUserChatNotificationTag*>(tagNotify.Get()));
                tagNotifyImpl->SetLastNotifiedAt(Now());
            }
            auto tagPush = IJsonSerializableTag::BuildWithComment<TUserPushTag>(PushTagName, "chat push notification tracking");
            {
                auto tagPushImpl = Yensured(dynamic_cast<TUserPushTag*>(tagPush.Get()));
                tagPushImpl->AddMacro("<chat_id>", targetTopicLink);
            }
            if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tagNotify, GetRobotUserId(), userId, server, session) || !server->GetDriveAPI()->GetTagsManager().GetUserTags().AddTag(tagPush, GetRobotUserId(), userId, server, session)) {
                return MakeUnexpected<TString>("unable to add tags on user " + userId + " error: " + session.GetStringReport());
            }
        }
    }
    if (!session.Commit()) {
        return MakeUnexpected<TString>("unable to add tags " + session.GetStringReport());
    }
    return new IRTBackgroundProcessState();
}

NDrive::TScheme TChatUnreadPushes::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSNumeric>("night_start", "Начало ночи (секунды от 00:00 UTC)").SetRequired(true);
    scheme.Add<TFSNumeric>("night_end", "Конец ночи (секунды от 00:00 UTC)").SetRequired(true);
    scheme.Add<TFSString>("push_tag_name", "Тег с пушом").SetRequired(true);
    scheme.Add<TFSDuration>("wait_period", "Сколько ждать").SetRequired(true);
    scheme.Add<TFSDuration>("check_messages_from", "Проверять сообщения за последние n секунд").SetRequired(true);
    scheme.Add<TFSStructure>("sharding_policy", "Условия для выбора объекта").SetStructure<NDrive::TScheme>(TObjectSharding::GetScheme());

    return scheme;
}

NJson::TJsonValue TChatUnreadPushes::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "night_start", NightStart);
    NJson::InsertField(result, "night_end", NightEnd);
    NJson::InsertField(result, "push_tag_name", PushTagName);
    NJson::InsertField(result, "wait_period", NJson::Hr(WaitPeriod));
    NJson::InsertField(result, "check_messages_from", NJson::Hr(CheckMessagesFrom));
    NJson::InsertField(result, "sharding_policy", ShardingPolicy.SerializeToJson());
    return result;
}

bool TChatUnreadPushes::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    if (jsonInfo.Has("sharding_policy") && !ShardingPolicy.DeserializeFromJson(jsonInfo["sharding_policy"])) {
        return false;
    }
    return
        NJson::ParseField(jsonInfo, "night_start", NightStart, true) &&
        NJson::ParseField(jsonInfo, "night_end", NightEnd, true) &&
        NJson::ParseField(jsonInfo, "push_tag_name", PushTagName, true) &&
        NJson::ParseField(jsonInfo, "wait_period", WaitPeriod) &&
        NJson::ParseField(jsonInfo, "check_messages_from", CheckMessagesFrom);
}
