#include "processor.h"

TActiveChatsMonitoring::TFactory::TRegistrator<TActiveChatsMonitoring> TActiveChatsMonitoring::Registrator(GetTypeName());
TRTActiveChatsMonitoringState::TFactory::TRegistrator<TRTActiveChatsMonitoringState> TRTActiveChatsMonitoringState::Registrator(TActiveChatsMonitoring::GetTypeName());

bool TYtDumpingSettings::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_DURATION_OPT(jsonInfo, "window_size", WindowSize);
    JREAD_UINT_OPT(jsonInfo, "chunks_limit", ChunksLimit);
    JREAD_STRING_OPT(jsonInfo, "yt_cluster", YtCluster);
    JREAD_STRING_OPT(jsonInfo, "tables_path", TablesPath);
    return true;
}

NJson::TJsonValue TYtDumpingSettings::SerializeToJson() const {
    NJson::TJsonValue result;
    result["window_size"] = WindowSize.Seconds();
    result["chunks_limit"] = ChunksLimit;
    result["yt_cluster"] = YtCluster;
    result["tables_path"] = TablesPath;
    return result;
}

NDrive::TScheme TYtDumpingSettings::GetScheme() {
    NDrive::TScheme scheme;
    scheme.Add<TFSDuration>("window_size", "Размер окна").SetDefault(TDuration::Days(2));
    scheme.Add<TFSNumeric>("chunks_limit", "Максимальное количество чанков").SetDefault(1024);
    scheme.Add<TFSString>("yt_cluster", "Кластер на YT").SetDefault("hahn");
    scheme.Add<TFSString>("tables_path", "Путь к табличкам").SetDefault("//home/carsharing/production/support/active_chat_monitoring");
    return scheme;
}

TExpectedState TActiveChatsMonitoring::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    const TRTActiveChatsMonitoringState* state = dynamic_cast<const TRTActiveChatsMonitoringState*>(stateExt.Get());
    TUserPermissions::TPtr robotUserPermissions = server->GetDriveAPI()->GetUserPermissions(GetRobotUserId(), TUserPermissionsFeatures());
    if (!robotUserPermissions) {
        return MakeUnexpected<TString>("cannot get permissions for user " + GetRobotUserId());
    }
    ChatsFilter.SetPermissions(robotUserPermissions);

    TVector<TFilteredSupportChat> filteredChats;
    TMessagesCollector errors;
    if (!ChatsFilter.GetSupportRequests(server, filteredChats, TInstant::Zero(), GetRobotUserId(), errors)) {
        return MakeUnexpected<TString>(errors.GetStringReport());
    }

    if (MonitoringAgent != EMonitoringAgent::Telegram) {
        TMap<TString, TVector<ui32>> buckets;
        if (!ChatsFilter.GetMetricFilter().GetSplitByOperator()) {
            for (auto&& tagName : ChatsFilter.GetQueryTagNames(server)) {
                buckets.emplace(tagName, TVector<ui32>());
            }
        }
        for (auto&& chatEntry : filteredChats) {
            auto bucketName = chatEntry.GetYasmBucketName();
            buckets[bucketName].emplace_back(chatEntry.GetMetricValue());
        }
        for (auto&& bucketIt : buckets) {
            PushToAggregatedSensors(bucketIt.first, bucketIt.second);
        }
        {
            TMap<TString, TVector<ui32>> combinedLinesData;
            for (auto&& chatEntry : filteredChats) {
                auto bucketName = chatEntry.GetOverallBucketName();
                combinedLinesData[bucketName].emplace_back(chatEntry.GetMetricValue());
            }
            for (auto&& bucketIt : combinedLinesData) {
                PushToAggregatedSensors(bucketIt.first, bucketIt.second);
            }
        }
    }

    TAtomicSharedPtr<TRTActiveChatsMonitoringState> newState;
    if (state) {
        newState = new TRTActiveChatsMonitoringState(*state);
    } else {
        newState = new TRTActiveChatsMonitoringState();
    }

    if (NotifierName && !filteredChats.empty() && MonitoringAgent == EMonitoringAgent::Telegram) {
        auto notifier = server->GetNotifier(NotifierName);
        if (notifier) {
            TVector<TString> reportLines;
            for (auto&& chat : filteredChats) {
                if (!state || state->GetLastNotifyInstant(chat.GetTag().GetTagId()) + NotificationsDelay <= StartInstant) {
                    reportLines.emplace_back(ChatsFilter.BuildHRReport(server, chat));
                    newState->SetLastNotifyInstant(chat.GetTag().GetTagId(), StartInstant);
                }
            }
            if (!reportLines.empty()) {
                NDrive::INotifier::MultiLinesNotify(notifier, "Чаты, требующие внимания согласно роботу " + GetRTProcessName(), reportLines);
            }
        }
    }

    if (!ReadyNodes.empty()) {
        try {
            FlushYTMonitoringData();
        } catch (const std::exception& e) {
            ERROR_LOG << "dumping error " << FormatExc(e) << Endl;
        }
    }

    return newState;
}

void TActiveChatsMonitoring::StoreSignal(const TString& metricName, const ui32 metricValue) const {
    if (MonitoringAgent == EMonitoringAgent::Yasm) {
        TUnistatSignalsCache::SignalLastX("support-chats", metricName, metricValue);
    } else if (MonitoringAgent == EMonitoringAgent::YT) {
        if (metricValue) {
            NYT::TNode recordNode;
            recordNode["sensor"] = metricName;
            recordNode["value"] = metricValue;
            recordNode["timestamp"] = StartInstant.Seconds();
            ReadyNodes.emplace_back(std::move(recordNode));
        }
    }
}

void TActiveChatsMonitoring::FlushYTMonitoringData() const {
    NYT::TYPath path = YtSettings.GetTablesPath();
    if (!path.EndsWith("/")) {
        path += "/";
    }
    path += GetRTProcessName();
    if (!YtClient->Exists(path)) {
        auto schemaNode = NYT::TNode::CreateList()
            .Add(NYT::TNode()("name", "sensor")("type", "string"))
            .Add(NYT::TNode()("name", "value")("type", "int64"))
            .Add(NYT::TNode()("name", "timestamp")("type", "uint64"));
        YtClient->Create(path, NYT::NT_TABLE, NYT::TCreateOptions().Attributes(NYT::TNode()("schema", schemaNode)));
    }

    bool needsRebuild = false;
    {
        TString infoNodePath = path + "/@resource_usage/chunk_count";
        auto result = YtClient->Get(infoNodePath);
        if (result.IsInt64() && (size_t)result.AsInt64() > YtSettings.GetChunksLimit()) {
            needsRebuild = true;
        }
    }

    auto tx = YtClient->StartTransaction();

    if (needsRebuild) {
        auto reader = YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(path));
        auto writer = tx->CreateTableWriter<NYT::TNode>(NYT::TRichYPath(path));
        for (; reader->IsValid(); reader->Next()) {
            const NYT::TNode& row = reader->GetRow();
            if (row["timestamp"].AsUint64() + YtSettings.GetWindowSize().Seconds() < StartInstant.Seconds()) {
                continue;
            }
            writer->AddRow(row);
        }
    }
    auto writer = tx->CreateTableWriter<NYT::TNode>(NYT::TRichYPath(path).Append(true));
    for (auto&& newNode : ReadyNodes) {
        writer->AddRow(newNode);
    }
    writer->Finish();
    tx->Commit();

    ReadyNodes.clear();
}

void TActiveChatsMonitoring::PushToAggregatedSensors(const TString& group, TVector<ui32>& values) const {
    auto metricName = GetRTProcessName() + "-" + group;
    if (AggregationPolicy == EAggregationPolicy::Count) {
        StoreSignal(metricName, values.size());
        return;
    }
    if (AggregationPolicy == EAggregationPolicy::Mean) {
        double metricValue = 0;
        if (values.size()) {
            double valuesSum = 0;
            for (auto&& value : values) {
                valuesSum += value;
            }
            metricValue = valuesSum / values.size();
            StoreSignal(metricName, metricValue);
        }
        return;
    }
    if (AggregationPolicy == EAggregationPolicy::Quantiles) {
        Sort(values.begin(), values.end());
        for (auto quantile : Quantiles) {
            double metricValue = 0;
            if (values.size()) {
                size_t position = values.size() * quantile / 100;
                if (position > 0) {
                    --position;
                }
                if (position < values.size()) {
                    metricValue = values[position];
                }
            }
            StoreSignal(metricName + "-" + ToString(quantile), metricValue);
        }
        return;
    }
    if (AggregationPolicy == EAggregationPolicy::Buckets) {
        Sort(values.begin(), values.end());
        size_t nextIdx = 0;
        for (auto&& bucketThreshold : Quantiles) {
            size_t countMatched = 0;
            while (nextIdx < values.size() && values[nextIdx] <= bucketThreshold) {
                ++countMatched;
                ++nextIdx;
            }
            StoreSignal(metricName + "-" + ToString(bucketThreshold), countMatched);
        }
        StoreSignal(metricName + "-inf", values.size() - nextIdx);
    }
}

NDrive::TScheme TActiveChatsMonitoring::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSStructure>("chats_filter", "Фильтр чатов").SetStructure(ChatsFilter.GetScheme());
    scheme.Add<TFSVariants>("aggregation_policy", "Как аггрегируем").InitVariants<EAggregationPolicy>();
    scheme.Add<TFSString>("quantiles", "Квантили/бакеты");
    scheme.Add<TFSVariants>("notifier_name", "Дополнительно нотифицировать в канал").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSDuration>("notification_delay", "Интервал между нотификациями по одному чату");

    scheme.Add<TFSVariants>("monitoring_agent", "Куда складывать данные для графиков").InitVariants<EMonitoringAgent>().SetDefault(ToString(EMonitoringAgent::YT));
    scheme.Add<TFSStructure>("yt_settings", "Настройки мониторинга на YT").SetStructure(TYtDumpingSettings::GetScheme());
    return scheme;
}

NJson::TJsonValue TActiveChatsMonitoring::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["chats_filter"] = ChatsFilter.SerializeToJson();
    result["aggregation_policy"] = ToString(AggregationPolicy);
    {
        TString quantiles = "";
        for (auto&& quantile : Quantiles) {
            if (quantiles) {
                quantiles += ",";
            }
            quantiles += ToString(quantile);
        }
        result["quantiles"] = quantiles;
    }
    if (NotifierName) {
        result["notifier_name"] = NotifierName;
    }
    JWRITE_DURATION(result, "notification_delay", NotificationsDelay);
    result["monitoring_agent"] = ToString(MonitoringAgent);
    result["yt_settings"] = YtSettings.SerializeToJson();
    return result;
}

bool TActiveChatsMonitoring::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    if (jsonInfo.Has("chats_filter") && !ChatsFilter.DeserializeFromJson(jsonInfo["chats_filter"])) {
        return false;
    }
    JREAD_FROM_STRING(jsonInfo, "aggregation_policy", AggregationPolicy);
    {
        Quantiles.clear();
        TString quantilesStr;
        JREAD_STRING_OPT(jsonInfo, "quantiles", quantilesStr);
        auto quantilesStringArr = SplitString(quantilesStr, ",");
        for (auto&& quantileStr : quantilesStringArr) {
            ui32 quantile;
            if (!TryFromString(quantileStr, quantile)) {
                return false;
            }
            if (AggregationPolicy == EAggregationPolicy::Quantiles && quantile > 100) {
                return false;
            }
            Quantiles.push_back(quantile);
        }
    }
    JREAD_STRING_OPT(jsonInfo, "notifier_name", NotifierName);
    JREAD_DURATION_OPT(jsonInfo, "notification_delay", NotificationsDelay);
    JREAD_FROM_STRING_OPT(jsonInfo, "monitoring_agent", MonitoringAgent);
    if (jsonInfo.Has("yt_settings") && !YtSettings.DeserializeFromJson(jsonInfo["yt_settings"])) {
        return false;
    }

    ReadyNodes.clear();
    YtClient = NYT::CreateClient(YtSettings.GetYtCluster(), NYT::TCreateClientOptions());

    return true;
}

void TRTActiveChatsMonitoringState::SerializeToProto(NDrive::NProto::TActiveChatMonitoringState& proto) const {
    for (auto&& it : LastNotification) {
        NDrive::NProto::TMutedCommunication* mutedChat = proto.AddMutedChats();
        mutedChat->SetTagId(it.first);
        mutedChat->SetLastNotifyTS(it.second.Seconds());
    }
}

bool TRTActiveChatsMonitoringState::DeserializeFromProto(const NDrive::NProto::TActiveChatMonitoringState& proto) {
    LastNotification.clear();
    for (auto&& mutedChat : proto.GetMutedChats()) {
        LastNotification.emplace(mutedChat.GetTagId(), TInstant::Seconds(mutedChat.GetLastNotifyTS()));
    }
    return true;
}

NJson::TJsonValue TRTActiveChatsMonitoringState::GetReport() const {
    NJson::TJsonValue result = TBase::GetReport();
    NJson::TJsonValue items = NJson::JSON_ARRAY;
        for (auto&& it : LastNotification) {
        NJson::TJsonValue item;
        item["tag_id"] = it.first;
        item["last_notify_ts"] = it.second.Seconds();
        items.AppendValue(std::move(item));
    }
    result["chats"] = std::move(items);
    return result;
}

NDrive::TScheme TRTActiveChatsMonitoringState::DoGetScheme() const {
    NDrive::TScheme scheme = TBase::DoGetScheme();

    NDrive::TScheme itemScheme;
    itemScheme.Add<TFSString>("tag_id");
    itemScheme.Add<TFSNumeric>("last_notify_ts");

    scheme.Add<TFSArray>("chats").SetElement(itemScheme);
    return scheme;
}

TString TRTActiveChatsMonitoringState::GetType() const {
    return TActiveChatsMonitoring::GetTypeName();
}

TInstant TRTActiveChatsMonitoringState::GetLastNotifyInstant(const TString& tagId) const {
    auto it = LastNotification.find(tagId);
    if (it != LastNotification.end()) {
        return it->second;
    }
    return it->second;
}

void TRTActiveChatsMonitoringState::SetLastNotifyInstant(const TString& tagId, const TInstant instant) {
    LastNotification[tagId] = instant;
}
