#include "config.h"

#include <drive/backend/compiled_riding/manager.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/device_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/tags/tags_manager.h>

#include <rtline/util/algorithm/container.h>

#include <util/string/join.h>

TRTCleaningWatcher::TFactory::TRegistrator<TRTCleaningWatcher> TRTCleaningWatcher::Registrator(TRTCleaningWatcher::GetTypeName());
TString TRTCleaningWatcher::BaseReport = "<mileage><rides_count><rides_duration><sessions_duration><critical_period><empty_report>";

class TRTCleaningWatcher::TCarMatching: public IRTCarsProcess::TCarTagsModification {
    R_READONLY(bool, ForRemove, false);
    R_OPTIONAL(double, Mileage);
    R_OPTIONAL(ui32, RidesCount);
    R_OPTIONAL(TDuration, RidesDuration);
    R_OPTIONAL(TDuration, SessionsDuration);
    R_OPTIONAL(TDuration, IdleDuration);

    TVector<ui64> CompiledRideIds;
    TSet<TString> Users;
    const TRTCleaningWatcher& Owner;

public:
    TCarMatching(const TDriveCarInfo& info, const bool forRemove, const TRTCleaningWatcher& owner)
        : TCarTagsModification(info.GetId(), info.GetHRReport())
        , ForRemove(forRemove)
        , Owner(owner)
    {
    }

    TString GetHRInfo(const IServerBase& server, const TRTCleaningWatcher& watcher) const {
        TString report = watcher.GetHRReportTemplate() ? watcher.GetHRReportTemplate() : TRTCleaningWatcher::BaseReport;
        bool hasReport = false;
        if (HasMileage() && watcher.HasCriticalMileage()) {
            SubstGlobal(report, "<mileage>", "пробег " + server.GetLocalization()->DistanceFormatKm(ELocalization::Rus, GetMileageUnsafe()) + "\n");
            hasReport = true;
        } else {
            SubstGlobal(report, "<mileage>", "");
        }
        if (HasRidesCount() && watcher.HasCriticalRidesCount()) {
            SubstGlobal(report, "<rides_count>", "пользователи " + ::ToString(GetRidesCountUnsafe()) + "\n");
            hasReport = true;
        } else {
            SubstGlobal(report, "<rides_count>", "");
        }
        if (HasRidesDuration() && watcher.HasCriticalRidesDuration()) {
            hasReport = true;
            SubstGlobal(report, "<rides_duration>", "в поездке " + server.GetLocalization()->FormatDuration(ELocalization::Rus, GetRidesDurationUnsafe()) + "\n");
        } else {
            SubstGlobal(report, "<rides_duration>", "");
        }
        if (HasSessionsDuration() && watcher.HasCriticalSessionsDuration()) {
            hasReport = true;
            SubstGlobal(report, "<sessions_duration>", "фактическое использование " + server.GetLocalization()->FormatDuration(ELocalization::Rus, GetSessionsDurationUnsafe()) + "\n");
        } else {
            SubstGlobal(report, "<sessions_duration>", "");
        }
        if (HasIdleDuration() && watcher.HasCriticalPeriod()) {
            SubstGlobal(report, "<critical_period>", "прошло " + server.GetLocalization()->FormatDuration(ELocalization::Rus, GetIdleDurationUnsafe()) + "\n");
            hasReport = true;
        } else {
            SubstGlobal(report, "<critical_period>", "");
        }
        if (!hasReport && (watcher.HasCriticalMileage() || watcher.HasCriticalRidesCount() || watcher.HasCriticalRidesDuration() || watcher.HasCriticalPeriod())) {
            SubstGlobal(report, "<empty_report>", "Нет информации за запрошенный период");
        } else {
            SubstGlobal(report, "<empty_report>", "");
        }
        return report;
    }

    bool FetchData(const NDrive::IServer& server, TSessionsBuilder<TCarTagHistoryEvent>::TPtr bSessions, const TConstDevicesSnapshot& snapshots, const TStopEvents& stopEvents, TVector<ui64>& historyIds, TInstant now, NDrive::TEntitySession& session) {
        const TString& carId = GetObjectId();
        const TSet<EObjectHistoryAction> startEvents = ForRemove ? TSet<EObjectHistoryAction>({EObjectHistoryAction::Add, EObjectHistoryAction::TagEvolve}) : TSet<EObjectHistoryAction>({EObjectHistoryAction::Remove});
        const auto& deviceTagManager = server.GetDriveAPI()->GetTagsManager().GetDeviceTags();

        TMaybe<TInstant> closeLastInstant;
        TMaybe<double> mileageLastInstant;
        ui64 sinceEventId = 0;

        auto p = stopEvents.find(carId);
        if (p != stopEvents.end()) {
            const auto& ev = p->second;
            sinceEventId = ev.GetHistoryEventId();
            closeLastInstant = ev.GetHistoryTimestamp();
            auto snapshot = ev->GetObjectSnapshotAs<THistoryDeviceSnapshot>();
            mileageLastInstant = snapshot ? snapshot->GetMileage() : mileageLastInstant;
        }

        if (!closeLastInstant || !mileageLastInstant) {

        INFO_LOG << Owner.GetRobotId() << ": GetEventsByObject " << carId << Endl;
        auto optionalEvents = deviceTagManager.GetEventsByObject(carId, session, sinceEventId);
        if (!optionalEvents) {
            ERROR_LOG << "RTCleaningWatcher::CarMatching::FetchData: cannot GetEventsByObject: " << session.GetStringReport() << Endl;
            return false;
        }

        for (auto it = optionalEvents->rbegin(); it != optionalEvents->rend(); ++it) {
            const auto& ev = *it;
            closeLastInstant = ev.GetHistoryInstant();
            const THistoryDeviceSnapshot* ds = dynamic_cast<const THistoryDeviceSnapshot*>(ev->GetObjectSnapshot().Get());
            double mileage;
            if (ds && ds->GetMileage(mileage)) {
                mileageLastInstant = mileage;
            }
            if (Owner.GetTagsForCheck().contains(ev->GetName()) && startEvents.contains(ev.GetHistoryAction())) {
                break;
            }
        }

        }

        {
            const TRTDeviceSnapshot* ds = snapshots.GetSnapshot(carId);
            if (ds) {
                double mileage;
                if (!!mileageLastInstant && ds->GetMileage(mileage)) {
                    SetMileage(mileage - *mileageLastInstant);
                }
            }
        }
        if (closeLastInstant) {
            SetIdleDuration(now - *closeLastInstant);
        }
        if (Owner.Match(*this)) {
            return true;
        }
        if (Owner.HasCriticalRidesCount() || Owner.HasCriticalRidesDuration() || Owner.HasCriticalSessionsDuration()) {
            auto since = closeLastInstant.GetOrElse(TInstant::Zero());
            auto ydbTx = server.GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("cleaning_fetch_data", &server);
            auto optionalCompiledSessions = server.GetDriveAPI()->GetMinimalCompiledRides().GetObject<TMinimalCompiledRiding>({carId}, session, ydbTx, since);
            if (!optionalCompiledSessions) {
                return false;
            }

            TDuration rideDuration = TDuration::Zero();
            TDuration sessionDuration = TDuration::Zero();
            TSet<TString> compileRideIds;
            for (auto&& compiledSession : *optionalCompiledSessions) {
                compileRideIds.emplace(compiledSession.GetSessionId());
                if (compiledSession.GetStartInstant() < since) {
                    continue;
                }

                CompiledRideIds.emplace_back(compiledSession.GetHistoryEventId());
            }
            IEventsSession<TCarTagHistoryEvent>::TPtr lastSession = bSessions->GetLastObjectSession(carId);
            if (!!lastSession && !compileRideIds.contains(lastSession->GetSessionId()) && lastSession->GetStartTS() >= since) {
                TBillingSession::TBillingCompilation bCompilation;
                if (lastSession->FillCompilation(bCompilation)) {
                    if (bCompilation.GetCurrentOffer()) {
                        auto action = server.GetDriveAPI()->GetRolesManager()->GetAction(bCompilation.GetCurrentOffer()->GetBehaviourConstructorId());
                        if (Owner.GetOfferAttributesFilter().IsEmpty() || (action && Owner.GetOfferAttributesFilter().IsMatching((*action)->GetGrouppingTags()))) {
                            rideDuration += bCompilation.GetRideDuration();
                            sessionDuration += bCompilation.GetActiveDuration();
                            Users.emplace(lastSession->GetUserId());
                        }
                    }
                }
            }
            SetRidesCount(Users.size());
            SetRidesDuration(rideDuration);
            SetSessionsDuration(sessionDuration);
        }
        if (Owner.Match(*this)) {
            return true;
        }
        historyIds.insert(historyIds.end(), CompiledRideIds.begin(), CompiledRideIds.end());
        return true;
    }

    void FetchRidingDuration(const TMap<ui64, TObjectEvent<TCompiledRiding>>& cRides, const NDrive::IServer& server) {
        Y_UNUSED(server);
        TDuration rideDuration = OptionalRidesDuration().GetOrElse(TDuration::Zero());
        TDuration sessionDuration = OptionalSessionsDuration().GetOrElse(TDuration::Zero());
        for (auto&& i : CompiledRideIds) {
            auto it = cRides.find(i);
            if (it == cRides.end()) {
                continue;
            }
            rideDuration += it->second.GetRidingDurationDef(TDuration::Zero());
            sessionDuration += it->second.GetRidingDurationDef(TDuration::Zero()) + it->second.GetAcceptanceDurationDef(TDuration::Zero()) + it->second.GetParkingDurationDef(TDuration::Zero());
            Users.emplace(it->second.GetHistoryUserId());
        }
        SetRidesCount(Users.size());
        SetRidesDuration(rideDuration);
        SetSessionsDuration(sessionDuration);
    }

    void FetchRidingDuration(const TMap<ui64, TObjectEvent<TFullCompiledRiding>>& cRides, const NDrive::IServer& server) {
        TDuration rideDuration = OptionalRidesDuration().GetOrElse(TDuration::Zero());
        TDuration sessionDuration = OptionalSessionsDuration().GetOrElse(TDuration::Zero());
        for (auto&& i : CompiledRideIds) {
            auto it = cRides.find(i);
            if (it == cRides.end()) {
                continue;
            }
            if (!Owner.GetOfferAttributesFilter().IsEmpty()) {
                if (!it->second.GetOffer()) {
                    continue;
                }
                auto action = server.GetDriveAPI()->GetRolesManager()->GetAction(it->second.GetOffer()->GetBehaviourConstructorId());
                if (!action || !Owner.GetOfferAttributesFilter().IsMatching((*action)->GetGrouppingTags())) {
                    continue;
                }
            }
            rideDuration += it->second.GetRidingDurationDef(TDuration::Zero());
            sessionDuration += it->second.GetRidingDurationDef(TDuration::Zero()) + it->second.GetAcceptanceDurationDef(TDuration::Zero()) + it->second.GetParkingDurationDef(TDuration::Zero());
            Users.emplace(it->second.GetHistoryUserId());
        }
        SetRidesCount(Users.size());
        SetRidesDuration(rideDuration);
        SetSessionsDuration(sessionDuration);
    }
};

TMaybe<TRTCleaningWatcher::TStopEvents> TRTCleaningWatcher::GetStopEvents(const TSet<TString>& carIds, const IEntityTagsManager& tagManager, NDrive::TEntitySession& session) const {
    const auto startEvents = RemoveTag ? TSet<EObjectHistoryAction>({EObjectHistoryAction::Add, EObjectHistoryAction::TagEvolve}) : TSet<EObjectHistoryAction>({EObjectHistoryAction::Remove});
    auto optionalEvents = tagManager.GetEvents(TInstant::Zero(), session, IEntityTagsManager::TQueryOptions(carIds.size(), true)
        .SetObjectIds(carIds)
        .SetActions(startEvents)
        .SetTags(TagsForCheck)
    );
    if (!optionalEvents) {
        return {};
    }
    TStopEvents result;
    if (optionalEvents->empty()) {
        return result;
    }
    for (auto&& ev : *optionalEvents) {
        auto objectId = ev.GetObjectId();
        if (result.contains(objectId)) {
            continue;
        }
        result.emplace(std::move(objectId), std::move(ev));
    }
    TSet<TString> remainders;
    for (auto&& id : carIds) {
        if (!result.contains(id)) {
            remainders.insert(id);
        }
    }
    if (!remainders.empty()) {
        INFO_LOG << GetRobotId() << ": GetStopEvents fetching extra " << remainders.size() << " ids" << Endl;
        auto extraEvents = GetStopEvents(remainders, tagManager, session);
        if (!extraEvents) {
            return {};
        }
        result.insert(extraEvents->begin(), extraEvents->end());
    }
    return result;
}

bool TRTCleaningWatcher::Match(const TCarMatching& info) const {
    if (HasCriticalMileage()) {
        if (info.HasMileage() && info.GetMileageUnsafe() + 1e-5 >= GetCriticalMileageUnsafe()) {
            return true;
        }
    }

    if (HasCriticalRidesCount()) {
        if (info.HasRidesCount() && info.GetRidesCountUnsafe() >= GetCriticalRidesCountUnsafe()) {
            return true;
        }
    }

    if (HasCriticalRidesDuration()) {
        if (info.HasRidesDuration() && info.GetRidesDurationUnsafe() >= GetCriticalRidesDurationUnsafe()) {
            return true;
        }
    }

    if (HasCriticalSessionsDuration()) {
        if (info.HasSessionsDuration() && info.GetSessionsDurationUnsafe() >= GetCriticalSessionsDurationUnsafe()) {
            return true;
        }
    }

    if (HasCriticalPeriod()) {
        if (!info.HasIdleDuration() || info.GetIdleDurationUnsafe() >= GetCriticalPeriodUnsafe()) {
            return true;
        }
    }
    return false;
}

TExpectedState TRTCleaningWatcher::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    auto result = MakeAtomicShared<IRTBackgroundProcessState>();

    const TMinimalRidingHistoryManager& compiledSessionManager = server.GetDriveAPI()->GetMinimalCompiledRides();
    const IDriveTagsManager& tagsManager = server.GetDriveAPI()->GetTagsManager();

    if (!tagsManager.GetTagsMeta().GetTagTypeByName(GetTagName())) {
        return MakeUnexpected("incorrect tag name: " + GetTagName());
    }

    TVector<TDBTag> dbTags;
    TSet<TString> taggedCars;
    {
        auto session = tagsManager.GetDeviceTags().BuildTx<NSQL::ReadOnly>();
        if (!tagsManager.GetDeviceTags().RestoreTags({}, MakeVector(GetTagsForCheck()), dbTags, session)) {
            return MakeUnexpected("cannot restore tags: " + session.GetStringReport());
        }
        for (auto&& i : dbTags) {
            taggedCars.emplace(i.GetObjectId());
        }
    }

    TSessionsBuilder<TCarTagHistoryEvent>::TPtr bSessions;
    for (ui32 i = 0; i < 7; ++i) {
        bSessions = server.GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilderSafe("billing", Now());
        if (!!bSessions) {
            break;
        }
    }
    if (!bSessions) {
        return MakeUnexpected<TString>("no billing session builder");
    }
    auto snapshots = server.GetSnapshotsManager().GetSnapshots();

    TMap<TString, TCarMatching> carsMatching;
    TMap<TString, TCarMatching> carsNotMatching;
    {
        TMap<TString, TDriveCarInfo> allCarsInfo = context.GetFetchedCarsData();
        TVector<ui64> historyIds;
        TMap<TString, TCarMatching> matchingMap;
        {
            TSet<TString> carIds;
            for (auto&& [id, data] : allCarsInfo) {
                if (RemoveTag && !taggedCars.contains(id)) {
                    continue;
                }
                carIds.insert(id);
            }
            auto session = tagsManager.GetDeviceTags().BuildTx<NSQL::ReadOnly>();
            INFO_LOG << GetRobotId() << ": GetStopEvents start" << Endl;
            auto optionalStopEvents = GetStopEvents(carIds, tagsManager.GetDeviceTags(), session);
            INFO_LOG << GetRobotId() << ": GetStopEvents finish" << Endl;
            if (!optionalStopEvents) {
                return MakeUnexpected("cannot GetStopEvents: " + session.GetStringReport());
            }
            INFO_LOG << GetRobotId() << ": FetchData start" << Endl;
            for (auto [id, data] : allCarsInfo) {
                if (RemoveTag && !taggedCars.contains(id)) {
                    continue;
                }
                TCarMatching matching(data, RemoveTag, *this);
                if (!matching.FetchData(server, bSessions, snapshots, *optionalStopEvents, historyIds, StartInstant, session)) {
                    return MakeUnexpected("cannot FetchData for " + id + ": " + session.GetStringReport());
                }
                matchingMap.emplace(id, std::move(matching));
            }
            INFO_LOG << GetRobotId() << ": FetchData finish" << Endl;
        }
        for (ui32 i = 0; i < historyIds.size(); ) {
            INFO_LOG << GetRobotId() << ": FetchRidingDuration " << i << " start" << Endl;
            auto tx = tagsManager.GetDeviceTags().BuildTx<NSQL::ReadOnly>();
            const ui32 iNext = Min<ui32>(i + 50000, historyIds.size());
            const TVector<ui64> currentIds(historyIds.begin() + i, historyIds.begin() + iNext);
            if (OfferAttributesFilter.IsEmpty()) {
                auto compiledSessions = compiledSessionManager.GetEvents<TCompiledRiding>(currentIds, tx);
                if (!compiledSessions) {
                    ERROR_LOG << GetRobotId() << "cannot GetEventsCompiledRiding: " << tx.GetStringReport() << Endl;
                    continue;
                }
                TMap<ui64, TObjectEvent<TCompiledRiding>> cRides;
                for (auto&& session : *compiledSessions) {
                    auto eventId = session.GetHistoryEventId();
                    cRides.emplace(eventId, std::move(session));
                }
                for (auto&& [id, matching] : matchingMap) {
                    matching.FetchRidingDuration(cRides, server);
                }
            } else {
                auto compiledSessions = compiledSessionManager.GetEvents<TFullCompiledRiding>(currentIds, tx);
                if (!compiledSessions) {
                    ERROR_LOG << GetRobotId() << "cannot GetEventsCompiledRiding: " << tx.GetStringReport() << Endl;
                    continue;
                }
                TMap<ui64, TObjectEvent<TFullCompiledRiding>> cRides;
                for (auto&& session : *compiledSessions) {
                    auto eventId = session.GetHistoryEventId();
                    cRides.emplace(eventId, std::move(session));
                }
                for (auto&& [id, matching] : matchingMap) {
                    matching.FetchRidingDuration(cRides, server);
                }
            }
            INFO_LOG << GetRobotId() << ": FetchRidingDuration " << i << " finish" << Endl;
            i = iNext;
        }
        for (auto i : matchingMap) {
            if (Match(i.second)) {
                carsMatching.emplace(i.first, std::move(i.second));
            } else if (taggedCars.contains(i.first) && !RemoveTag) {
                carsNotMatching.emplace(i.first, std::move(i.second));
            }
        }
    }

    if (RemoveTag) {
        for (auto&& i : dbTags) {
            if (!GetPerformingModification() && !!i->GetPerformer()) {
                continue;
            }
            auto it = carsMatching.find(i.GetObjectId());
            if (it != carsMatching.end()) {
                it->second.MutableTagsForRemove().emplace_back(std::move(i));
            }
        }
    } else {

        for (auto&& i : dbTags) {
            carsMatching.erase(i.GetObjectId());
        }

        for (auto it = carsNotMatching.begin(); it != carsNotMatching.end();) {
            if (!taggedCars.contains(it->first)) {
                it = carsNotMatching.erase(it);
            } else {
                ++it;
            }
        }

        for (auto&& i : dbTags) {
            if (!GetPerformingModification() && !!i->GetPerformer()) {
                continue;
            }
            auto it = carsNotMatching.find(i.GetObjectId());
            if (it != carsNotMatching.end()) {
                it->second.MutableTagsForRemove().emplace_back(i);
            }
        }

        for (auto&& i : carsMatching) {
            ITag::TPtr newTag = server.GetDriveAPI()->GetTagsManager().GetTagsMeta().CreateTag(GetTagName(), GetRTProcessName() + "\n" + i.second.GetHRInfo(server, *this));
            if (!newTag) {
                return MakeUnexpected<TString>("cannot create tag: " + GetTagName());
            }
            newTag->SetTagPriority(TagPriority);
            i.second.MutableTagsForAdd().emplace_back(newTag);
        }
    }

    for (auto&& it = carsMatching.begin(); it != carsMatching.end();) {
        if (it->second.IsEmpty()) {
            it = carsMatching.erase(it);
        } else {
            ++it;
        }
    }

    for (auto&& it = carsNotMatching.begin(); it != carsNotMatching.end(); ++it) {
        if (!it->second.IsEmpty() && IsCriteriaCondition && !RemoveTag) {
            carsMatching.emplace(it->first, it->second);
        }
    }

    const auto doReport = [this, &carsMatching, &server, &context]() {
        context.NotifyModifications(carsMatching, GetDryRunMode(), GetEmptyReport(), server.GetNotifier(GetNotifierName()), [this, &server](const TAbstractTagsModification* tagsModification) {
            auto carMatching = dynamic_cast<const TCarMatching*>(tagsModification);
            return tagsModification ? tagsModification->GetHRReport() + (carMatching ? "\n" + carMatching->GetHRInfo(server, *this) : "") : "";
        });
    };

    if (!context.ApplyModification(carsMatching, doReport, GetDryRunMode(), GetRTProcessName())) {
        return MakeUnexpected<TString>("cannot ApplyModification");
    }
    return result;
}

NDrive::TScheme TRTCleaningWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("tag_name", "[+/-]название тега для использования при срабатывании условий");
    scheme.Add<TFSDuration>("critical_period", "Время до действия");
    scheme.Add<TFSDuration>("critical_rides_duration", "Продолжительность поездок действия");
    scheme.Add<TFSDuration>("critical_sessions_duration", "Продолжительность сессий(без резерва)");
    scheme.Add<TFSNumeric>("critical_rides_count", "количество поездок до действия");
    scheme.Add<TFSNumeric>("critical_mileage", "пробег до действия (км)");
    scheme.Add<TFSNumeric>("tag_priority", "приоритет навешиваемого тега").SetDefault(0);
    scheme.Add<TFSString>("equal_tags", "эквивалентные теги tag1, tag2, tag3, ...");
    scheme.Add<TFSString>("offer_attribute_filter", "фильтр офферов по тегам группировки");
    scheme.Add<TFSString>("hr_report_template", "Шаблон комментария").SetDefault("<mileage><rides_count><rides_duration><critical_period><empty_report>");

    scheme.Add<TFSBoolean>("dry_run_mode", "только нотификации").SetDefault(false);
    scheme.Add<TFSBoolean>("is_criteria_condition", "условие является критерием наличия тега").SetDefault(false);
    scheme.Add<TFSBoolean>("performing_modification", "Модифицировать исполняемые").SetDefault(false);

    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSBoolean>("empty_report", "нотификация о холостом исполнении").SetDefault(false);
    return scheme;
}

bool TRTCleaningWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_STRING_OPT(jsonInfo, "notifier", NotifierName);
    JREAD_BOOL_OPT(jsonInfo, "dry_run_mode", DryRunMode);
    JREAD_BOOL_OPT(jsonInfo, "is_criteria_condition", IsCriteriaCondition);
    JREAD_BOOL_OPT(jsonInfo, "performing_modification", PerformingModification);

    JREAD_INT_OPT(jsonInfo, "tag_priority", TagPriority);
    JREAD_STRING(jsonInfo, "tag_name", TagName);

    AssertCorrectConfig(!!TagName, "incorrect TagName field in config");
    if (TagName.StartsWith("-")) {
        RemoveTag = true;
        TagName = TagName.substr(1);
    }
    if (TagName.StartsWith("+")) {
        RemoveTag = false;
        TagName = TagName.substr(1);
    }

    TString equalTagNames;
    JREAD_STRING_OPT(jsonInfo, "equal_tags", equalTagNames);
    StringSplitter(equalTagNames).SplitBySet(", ").SkipEmpty().Collect(&TagsForCheck);
    TagsForCheck.emplace(TagName);

    TString offerFilter;
    JREAD_STRING_OPT(jsonInfo, "offer_attribute_filter", offerFilter);
    OfferAttributesFilter = TTagsFilter::BuildFromString(offerFilter);

    JREAD_STRING_OPT(jsonInfo, "hr_report_template", HRReportTemplate);

    CriticalPeriod.Clear();
    CriticalRidesDuration.Clear();
    CriticalSessionsDuration.Clear();
    CriticalRidesCount.Clear();
    CriticalMileage.Clear();

    JREAD_DURATION_OPT(jsonInfo, "critical_period", CriticalPeriod);
    JREAD_DURATION_OPT(jsonInfo, "critical_rides_duration", CriticalRidesDuration);
    JREAD_DURATION_OPT(jsonInfo, "critical_sessions_duration", CriticalSessionsDuration);
    JREAD_INT_OPT(jsonInfo, "critical_rides_count", CriticalRidesCount);
    JREAD_DOUBLE_OPT(jsonInfo, "critical_mileage", CriticalMileage);
    JREAD_BOOL_OPT(jsonInfo, "empty_report", EmptyReport);
    return true;
}

NJson::TJsonValue TRTCleaningWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    JWRITE(result, "dry_run_mode", DryRunMode);
    JWRITE(result, "is_criteria_condition", IsCriteriaCondition);
    JWRITE(result, "notifier", NotifierName);
    JWRITE(result, "tag_priority", TagPriority);
    JWRITE(result, "tag_name", RemoveTag ? ("-" + TagName) : ("+" + TagName));
    JWRITE(result, "performing_modification", PerformingModification);
    JWRITE(result, "hr_report_template", HRReportTemplate);

    TSet<TString> tagsNamesSerial = TagsForCheck;
    tagsNamesSerial.erase(TagName);
    JWRITE(result, "equal_tags", JoinSeq(",", tagsNamesSerial));
    JWRITE(result, "offer_attribute_filter", OfferAttributesFilter.ToString());

    TJsonProcessor::WriteDurationString(result, "critical_period", CriticalPeriod);
    TJsonProcessor::WriteDurationString(result, "critical_rides_duration", CriticalRidesDuration);
    TJsonProcessor::WriteDurationString(result, "critical_sessions_duration", CriticalSessionsDuration);
    TJsonProcessor::Write(result, "critical_rides_count", CriticalRidesCount);
    TJsonProcessor::Write(result, "critical_mileage", CriticalMileage);
    JWRITE(result, "empty_report", EmptyReport);
    return result;
}
