#include "config.h"

#include <rtline/util/json_processing.h>

TExpectedState IRTCarsProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    TTagsModificationContext nextContext(*this, context);
    const NDrive::IServer& frServer = context.GetServerAs<NDrive::IServer>();

    if (!frServer.GetDriveAPI()->GetCarsData()->RefreshCache(DataActuality)) {
        return MakeUnexpected<TString>("cannot refresh cache");
    }
    if (!!ObjectsFilter) {
        if (!ObjectsFilter->GetAllowedCarIds(nextContext.MutableFilteredCarIds(), &frServer, DataActuality)) {
            return MakeUnexpected<TString>("cannot restore objects from cache");
        }
        nextContext.MutableFetchedCarsData() = frServer.GetDriveAPI()->GetCarsData()->GetCachedObjectsMap(nextContext.GetFilteredCarIds());
    } else {
        nextContext.MutableFetchedCarsData() = frServer.GetDriveAPI()->GetCarsData()->GetCachedObjectsMap();
        nextContext.MutableFilteredCarIds() = MakeSet(NContainer::Keys(nextContext.MutableFetchedCarsData()));
    }
    return DoExecuteFiltered(state, frServer, nextContext);
}

NDrive::TScheme IRTCarsProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    auto gTab = result.StartTabGuard("filter");
    result.Add<TFSStructure>("filter", "Фильтр объектов", 80000).SetStructure(TCarsFilter::GetScheme(server.GetAsSafe<NDrive::IServer>()));
    return result;
}

NJson::TJsonValue IRTCarsProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    if (!!ObjectsFilter) {
        JWRITE(result, "filter", ObjectsFilter->SerializeToJson());
    }
    return result;
}

bool IRTCarsProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (jsonInfo.Has("filter")) {
        TCarsFilter filter;
        if (!filter.DeserializeFromJson(jsonInfo["filter"])) {
            return false;
        }
        ObjectsFilter = std::move(filter);
    } else if (jsonInfo.Has("tags_filter")) {
        TTagsFilter tagsFilter;
        TString tagsFilterStr;
        JREAD_STRING_OPT(jsonInfo, "tags_filter", tagsFilterStr);
        if (!tagsFilter.DeserializeFromString(tagsFilterStr)) {
            return false;
        }

        TCarsFilter filter;
        filter.SetIncludeTagsFilter(std::move(tagsFilter));
        ObjectsFilter = std::move(filter);
    }
    return TBase::DoDeserializeFromJson(jsonInfo);
}

const TDriveCarInfo* TCarsProcessContext::GetFetchedCarsData(const TString& carId) const {
    auto it = FetchedCarsData.find(carId);
    if (it == FetchedCarsData.end()) {
        return nullptr;
    } else {
        return &it->second;
    }
}


TDateTimeFilterConfig::operator bool() const noexcept {
    return HasRestrictions();
}

bool TDateTimeFilterConfig::HasRestrictions() const {
    return Since || Until || For || To;
}

NDrive::TScheme TDateTimeFilterConfig::GetScheme() {
    NDrive::TScheme scheme;
    scheme.Add<TFSNumeric>("since", "Не раньше чем").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSNumeric>("until", "Не позже чем").SetVisual(TFSNumeric::EVisualType::DateTime);
    scheme.Add<TFSDuration>("for", "На столько до текущего момента");
    scheme.Add<TFSDuration>("to", "До стольки с текущего момента");
    return scheme;
}

TRange<TInstant> TDateTimeFilterConfig::GetFilter() const {
    TRange<TInstant> range;
    if (Since && Since != TInstant::Max()) {
        range.From = Since;
    } else if (For && For != TDuration::Max()) {
        range.From = ModelingNow() - For;
    }
    if (Until && Until != TInstant::Max()) {
        range.To = Until;
    } else if (To && To != TDuration::Max()) {
        range.To = ModelingNow() - To;
    }
    return range;
}

bool TDateTimeFilterConfig::DeserializeFromJson(const NJson::TJsonValue& data) {
    return NJson::ParseField(data["since"], Since)
        && NJson::ParseField(data["until"], Until)
        && NJson::ParseField(data["for"], For)
        && NJson::ParseField(data["to"], To);
}

NJson::TJsonValue TDateTimeFilterConfig::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "since", NJson::Seconds(Since));
    NJson::InsertNonNull(result, "until", NJson::Seconds(Until));
    if (For) {
        NJson::InsertField(result, "for", NJson::Hr(For));
    }
    if (To) {
        NJson::InsertField(result, "to", NJson::Hr(To));
    }
    return result;
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& data, TDateTimeFilterConfig& result) {
    return result.DeserializeFromJson(data);
}

template <>
NJson::TJsonValue NJson::ToJson(const TDateTimeFilterConfig& config) {
    return config.SerializeToJson();
}
