#pragma once

#include <drive/backend/rt_background/manager/manager.h>

#include <drive/backend/cars/car.h>
#include <drive/backend/cars_filter/cars_filter.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tag_modification.h>
#include <drive/backend/tags/tags_filter.h>

#include <drive/library/cpp/scheme/scheme.h>

class TCarsProcessContext: public IRTBackgroundProcess::TExecutionContext {
private:
    using TBase = IRTBackgroundProcess::TExecutionContext;
    using TCarsFetchedData = TMap<TString, TDriveCarInfo>;

private:
    R_FIELD(TSet<TString>, FilteredCarIds);
    R_FIELD(TCarsFetchedData, FetchedCarsData);

    const NDrive::IServer& Server;

public:
    const TDriveCarInfo* GetFetchedCarsData(const TString& carId) const;

    TCarsProcessContext(const TBase& baseContext)
        : TBase(baseContext)
        , Server(GetServerAs<NDrive::IServer>())
    {
    }

    const NDrive::IServer& GetServer() const {
        return GetServerAs<NDrive::IServer>();
    }
};

class IRTCarsProcess: public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;

public:
    using TCarTagsModification = TAbstractTagsModification;

    class TTagsModificationContext: public TCarsProcessContext, public TCarTagsModificationContext {
    private:
        using TBase = TCarsProcessContext;

    public:
        TTagsModificationContext(const IRTRegularBackgroundProcess& process, const TExecutionContext& context)
            : TBase(context)
            , TCarTagsModificationContext(process.GetRTProcessName(), process.GetRobotUserId(), context.GetServerAs<NDrive::IServer>())
        {
        }

        virtual TMaybe<TDriveCarInfo> GetFetchedObjectData(const TString& objectId) const override {
            auto result = GetFetchedCarsData(objectId);
            if (result) {
                return *result;
            } else {
                return {};
            }
        }
    };

protected:
    TMaybe<TCarsFilter> ObjectsFilter;

protected:
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual NJson::TJsonValue DoSerializeToJson() const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const final;
    virtual TExpectedState DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> state, const NDrive::IServer& server, TTagsModificationContext& context) const = 0;
};

class TDateTimeFilterConfig {
    R_FIELD(TInstant, Since, TInstant::Zero());
    R_FIELD(TInstant, Until, TInstant::Zero());
    R_FIELD(TDuration, For, TDuration::Zero());
    R_FIELD(TDuration, To, TDuration::Zero());

public:
    virtual ~TDateTimeFilterConfig() = default;

    static NDrive::TScheme GetScheme();

    explicit operator bool() const noexcept;
    virtual bool HasRestrictions() const;

    virtual bool DeserializeFromJson(const NJson::TJsonValue& data);
    virtual NJson::TJsonValue SerializeToJson() const;

    virtual TRange<TInstant> GetFilter() const;
};
