#include "database.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/history/sequential.h>

#include <rtline/util/types/expected.h>

NDrive::TScheme TDBTableScanner::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("start_from", "Начать с id").SetDefault(0);
    scheme.Add<TFSString>("table_name", "Имя таблицы в базе");
    scheme.Add<TFSString>("condition", "SQL условие на данные");
    scheme.Add<TFSNumeric>("quantum", "Максимальный размер пакета для обработки").SetDefault(0);
    scheme.Add<TFSNumeric>("lock_timeout_new", "Таймаут на лок (секунды)").SetDefault(LockTimeout.Seconds());
    return scheme;
}

bool TDBTableScanner::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_UINT_OPT(jsonInfo, "start_from", StartFromId)
    JREAD_STRING(jsonInfo, "table_name", TableName);
    JREAD_STRING_OPT(jsonInfo, "condition", Condition);
    JREAD_UINT_OPT(jsonInfo, "quantum", Quantum);
    JREAD_DURATION_OPT(jsonInfo, "lock_timeout_new", LockTimeout);
    return true;
}

NJson::TJsonValue TDBTableScanner::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["table_name"] = TableName;
    result["start_from"] = StartFromId;
    result["condition"] = Condition;
    result["quantum"] = Quantum;
    result["lock_timeout_new"] = LockTimeout.Seconds();
    return result;
}

TExpectedState TDBTableScanner::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    THolder<TRTHistoryWatcherState> result = THolder(BuildState());

    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const TRTHistoryWatcherState* state = dynamic_cast<const TRTHistoryWatcherState*>(stateExt.Get());
    auto lastEventId = state ? state->GetLastEventId() : StartFromId;

    auto table = GetTable();
    if (!table) {
        return MakeUnexpected<TString>("cannot find SequentialTable for " + TableName);
    }

    auto maxEventId = table->GetLockedMaxEventId(LockTimeout);
    auto eventIdColumn = table->GetEventIdFieldName();
    TRecordsSet records;
    {
        auto tx = table->BuildTx<NSQL::Writable>(LockTimeout);
        TStringStream selector;
        selector << "SELECT * FROM " << TableName
            << " WHERE " << eventIdColumn << " > " << lastEventId
            << " AND " << eventIdColumn << " <= " << maxEventId;
        if (Condition) {
            selector << " AND " << '(' << Condition << ')';
        }
        if (Quantum) {
            selector << " ORDER BY " << eventIdColumn;
            selector << " LIMIT " << Quantum;
        }

        auto queryResult = tx->Exec(selector.Str(), &records);
        if (!queryResult || !queryResult->IsSucceed()) {
            return MakeUnexpected<TString>(tx.GetStringReport());
        }
    }
    TMessagesCollector errors;
    if (!ProcessRecords(records, server, lastEventId, errors)) {
        return MakeUnexpected<TString>("ProcessRecordsFailure: " + errors.GetStringReport());
    }

    result->SetLastEventId(lastEventId);
    return result;
}

const IBaseSequentialTableImpl* TDBTableScanner::GetTable() const {
    auto result = IBaseSequentialTableImpl::Instance(TableName);
    if (!result) {
        ALERT_LOG << GetRobotId() << ": cannot get SequentialTable for " << TableName << Endl;
    }
    return result;
}
