#include "process.h"

#include <drive/backend/rt_background/manager/state.h>

#include <drive/backend/alerts/container.h>
#include <drive/backend/data/common/serializable.h>
#include <drive/backend/database/drive_api.h>

#include <util/string/join.h>


IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTCommonAlertsProcess> TRTCommonAlertsProcess::Registrator(TRTCommonAlertsProcess::GetTypeName());

TRTCommonAlertsState::TFactory::TRegistrator<TRTCommonAlertsState> TRTCommonAlertsState::Registrator(TRTCommonAlertsProcess::GetTypeName());

TString TRTCommonAlertsState::GetType() const {
    return TRTCommonAlertsProcess::GetTypeName();
}

TExpectedState TRTCommonAlertsProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    if (!AlertImpl) {
        return MakeUnexpected<TString>({});
    }

    TAtomicSharedPtr<TRTCommonAlertsState> stateImpl = std::dynamic_pointer_cast<TRTCommonAlertsState>(state);
    if (!stateImpl) {
        stateImpl = MakeAtomicShared<TRTCommonAlertsState>();
    }
    if (!AlertImpl->Execute(&server, StartInstant, nullptr, GetRTProcessName(), stateImpl)) {
        stateImpl->SetLastError(AlertImpl->GetErrors().GetStringReport());
    }
    if (AlertImpl->GetErrors().HasMessages()) {
        ERROR_LOG << "CommonAlertsProcess " << GetRTProcessName() << " error: " << AlertImpl->GetErrors().GetStringReport() << Endl;
    }
    return stateImpl;
}

NDrive::TScheme TRTCommonAlertsProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("entity_type", "С чем работаем").InitVariants<NAlerts::EAlertEntityType>();
    NAlerts::TAlertContainer::GetScheme(server, scheme);
    return scheme;
}


bool TRTCommonAlertsProcess::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }

    JREAD_FROM_STRING_OPT(jsonInfo, "entity_type", EntityType);
    AlertImpl = MakeHolder<NAlerts::TAlertContainer>(EntityType, GetRobotUserId());
    return AlertImpl->DeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTCommonAlertsProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    JWRITE(result, "entity_type", ::ToString(EntityType));
    if (AlertImpl) {
        AlertImpl->SerializeToJson(result);
    }
    return result;
}
