#include "manager.h"

#include <drive/backend/registrar/manager.h>
#include <drive/backend/database/drive/private_data.h>

#include <drive/backend/processors/yang_proxy/callback_persdata.h>

TRTDatasyncManager::TFactory::TRegistrator<TRTDatasyncManager> TRTDatasyncManager::Registrator(TRTDatasyncManager::GetTypeName());

namespace {
    TMutex Mutex;
}

bool TRTDatasyncManager::TYangJsonParser::ParseChecks(const TString& assignmentId, const TMap<TString, TSet<TString>>& checksSettings, const TDriveAPI* driveApi) {
    if (!UserId && SecretId) {
        auto session = driveApi->BuildTx<NSQL::ReadOnly>();
        auto optionalAssignment = TRTDatasyncManager::GetAssignment(SecretId, driveApi, session);
        if (optionalAssignment) {
            UserId = optionalAssignment->GetUserId();
        } else {
            throw TInternalDatasyncManagerError()
            << "failed to get userId for assignment " << assignmentId
            << ": " << session.GetStringReport();
        }
    }

    if (!UserId || !Verdicts || !Verdicts->IsMap()) {
        return false;
    }

    for (const auto& [name, status] : Verdicts->GetMap()) {
        if (!status.IsArray() || status.GetArray().empty() || !status.GetArray().front().IsString()) {
            return false;
        }
        if (name == "fraud_reasons") {
            continue;
        } else {
            TString statusString = status.GetArray().front().GetString();
            VerdictsParsed.emplace(name, TUserDocumentsCheck(
                UserId, name, statusString, SecretId, WorkerId, PoolId,
                "", assignmentId, "", Timestamp.Seconds(), Comment)
            );
        }
    }

    for (const auto& [name, check] : VerdictsParsed) {
        if (!checksSettings.contains(name) || !checksSettings.at(name).contains(check.GetStatus())) {
            return false;
        }
    }

    if (!(PassportBiographical && PassportBiographical->IsDefined()) &&
        !(PassportRegistration && PassportRegistration->IsDefined()) &&
        !(LicenseBack && LicenseBack->IsDefined()) &&
        !(LicenseFront && LicenseFront->IsDefined())
    ) {
        INFO_LOG << TRTDatasyncManager::GetTypeName() << ": no documents data parsed, only verdicts" << Endl;
    }

    return true;
}

void TRTDatasyncManager::TDocumentsSuite::AddYangData(TYangJsonParser&& parser, const TInstant& createdAt, const TString& operatorUserId) {
    OperatorUserId = operatorUserId;
    auto assignmentId = parser.YangJson["id"].GetString();

    TMessagesCollector errors;
    const auto* passportBiographical = parser.PassportBiographical;
    const auto* passportRegistration = parser.PassportRegistration;
    const auto* licenseBack = parser.LicenseBack;
    const auto* licenseFront = parser.LicenseFront;

    if ((passportBiographical && passportBiographical->IsDefined()) ||
        (passportRegistration && passportRegistration->IsDefined())
    ) {
        TUserPassportData patch;
        patch.ParseFromYang(
            passportBiographical ? *passportBiographical : NJson::EJsonValueType::JSON_NULL,
            passportRegistration ? *passportRegistration : NJson::EJsonValueType::JSON_NULL,
            errors
        );

        if (PassportDataPatch) {
            PassportDataPatch->Patch(patch);
        } else {
            PassportDataPatch = std::move(patch);
        }

        if (PassportTimestamp < createdAt) {
            PassportTimestamp = createdAt;
            PassportRevision = assignmentId;
        }
    }

    if ((licenseBack && licenseBack->IsDefined()) ||
        (licenseFront && licenseFront->IsDefined())
    ) {
        TUserDrivingLicenseData patch;
        patch.ParseFromYang(
            licenseBack ? *licenseBack : NJson::EJsonValueType::JSON_NULL,
            licenseFront ? *licenseFront : NJson::EJsonValueType::JSON_NULL,
            errors
        );

        if (DrivingLicenseDataPatch) {
            DrivingLicenseDataPatch->Patch(patch);
        } else {
            DrivingLicenseDataPatch = patch;
        }

        if (LicenseTimestamp < createdAt) {
            LicenseTimestamp = createdAt;
            LicenseRevision = assignmentId;
        }
    }

    for (const auto& [name, status] : parser.VerdictsParsed) {
        if (Verdicts.contains(name)) {
            if (Verdicts.at(name).second < createdAt) {
                Verdicts[name] = {status, createdAt};
            }
        } else {
            Verdicts[name] = {status, createdAt};
        }
    }

    YangData[assignmentId] = std::move(parser.YangJson);
}

TString TRTDatasyncManager::TDocumentsSuite::GetPassportRevision() const {
    return PassportRevision;
}

TString TRTDatasyncManager::TDocumentsSuite::GetLicenseRevision() const {
    return LicenseRevision;
}

TMaybe<TYangDocumentVerificationAssignment> TRTDatasyncManager::GetAssignment(const TString& secretId, const TDriveAPI* driveApi, NDrive::TEntitySession& session) {
        const auto& photoManager = Yensured(driveApi)->GetDocumentPhotosManager();
        auto fetchAssignmentResult = photoManager.GetDocumentVerificationAssignments().FetchInfo(secretId, session);
        if (fetchAssignmentResult.size() != 1) {
            return Nothing();
        }
        return fetchAssignmentResult.begin()->second;
}

TString TRTDatasyncManager::TYangJsonParser::GetUserId(const NJson::TJsonValue& yangData, const TYangJsonParserConfig& config) {
    return yangData.GetValueByPath(config.UserIdPath)
            ? yangData.GetValueByPath(config.UserIdPath)->GetString()
            : "";
}

TDriveUserData TRTDatasyncManager::GetUser(const TString& userId, const TDriveAPI* driveApi, NDrive::TEntitySession& session) {
    auto fetchUserResult = driveApi->GetUsersData()->FetchInfo(userId, session);
    if (!fetchUserResult || fetchUserResult.size() != 1) {
        throw TInternalDatasyncManagerError() << "Failed to find user";
    }

    return fetchUserResult.begin()->second;
}

NThreading::TFuture<TRTDatasyncManager::TDocumentsSuite*> TRTDatasyncManager::TDocumentsSuite::FillDocumentsData(const NDrive::IServer* server) {
    auto DriveApi = server->GetDriveAPI();

    NThreading::TFuture<TUserPassportData> passportDataFuture;
    NThreading::TFuture<TUserDrivingLicenseData> licenseDataFuture;
    if (User.GetPassportDatasyncRevision() && PassportDataPatch) {
        passportDataFuture = DriveApi->GetPrivateDataClient().GetPassport(User, User.GetPassportDatasyncRevision());
    }
    if (User.GetDrivingLicenseDatasyncRevision() && DrivingLicenseDataPatch) {
        licenseDataFuture = DriveApi->GetPrivateDataClient().GetDrivingLicense(User, User.GetDrivingLicenseDatasyncRevision());
    }

    TVector<NThreading::TFuture<void>> futures;
    if (passportDataFuture.Initialized()) {
        futures.push_back(passportDataFuture.IgnoreResult());
    }
    if (licenseDataFuture.Initialized()) {
        futures.push_back(licenseDataFuture.IgnoreResult());
    }
    auto waiter = NThreading::WaitExceptionOrAll(futures);
    return waiter.Apply([
        server,
        this,
        passportDataFuture = std::move(passportDataFuture),
        licenseDataFuture = std::move(licenseDataFuture)
    ](const NThreading::TFuture<void>&) -> TDocumentsSuite* {
        if (passportDataFuture.Initialized()) {
            try {
                PassportData = passportDataFuture.GetValue();
            } catch (const TDatasyncNotFound&) {
            }
        }
        if (licenseDataFuture.Initialized()) {
            try {
                DrivingLicenseData = licenseDataFuture.GetValue();
            } catch (const TDatasyncNotFound&) {
            }
        }

        TMessagesCollector errors;
        if (PassportDataPatch) {
            if (PassportData) {
                PassportData->Patch(*PassportDataPatch);
            } else {
                PassportData = PassportDataPatch;
            }
            User.SetPassportDatasyncRevision(GetPassportRevision());
        }

        if (DrivingLicenseDataPatch) {
            if (DrivingLicenseData) {
                DrivingLicenseData->Patch(*DrivingLicenseDataPatch);
            } else {
                DrivingLicenseData = DrivingLicenseDataPatch;
            }
            User.SetDrivingLicenseDatasyncRevision(GetLicenseRevision());
        }

        auto userRegistrationManager = server->GetUserRegistrationManager();
        if (PassportData && PassportData->HasBirthDate()) {
            if (!userRegistrationManager) {
                throw TInternalDatasyncManagerError() << "UserRegistrationManager is missing";
            }
            User.SetPassportNamesHash(userRegistrationManager->GetDocumentNumberHash(PassportData->GetNamesForHash()));
        }
        if (PassportData && PassportData->GetNumber()) {
            if (!userRegistrationManager) {
                throw TInternalDatasyncManagerError() << "UserRegistrationManager is missing";
            }
            User.SetPassportNumberHash(userRegistrationManager->GetDocumentNumberHash(PassportData->GetNumber()));
        }
        if (DrivingLicenseData && DrivingLicenseData->GetNumber()) {
            if (!userRegistrationManager) {
                throw TInternalDatasyncManagerError() << "UserRegistrationManager is missing";
            }
            User.SetDrivingLicenseNumberHash(userRegistrationManager->GetDocumentNumberHash(DrivingLicenseData->GetNumber()));
        }

        if (PassportData && PassportData->HasFirstName()) {
            User.SetFirstName(NRegistrarUtil::ToTitle(PassportData->GetFirstName()));
            User.SetLastName(NRegistrarUtil::ToTitle(PassportData->GetLastName()));
            User.SetPName(NRegistrarUtil::ToTitle(PassportData->GetMiddleName()));
        } else if (DrivingLicenseData && DrivingLicenseData->HasFirstName()) {
            User.SetFirstName(NRegistrarUtil::ToTitle(DrivingLicenseData->GetFirstName()));
            User.SetLastName(NRegistrarUtil::ToTitle(DrivingLicenseData->GetLastName()));
            User.SetPName(NRegistrarUtil::ToTitle(DrivingLicenseData->GetMiddleName()));
        }
        return this;
    });
}

NThreading::TFuture<void> TRTDatasyncManager::WriteToDatasync(const NThreading::TFuture<TRTDatasyncManager::TDocumentsSuite*>& documentsSuiteFuture, const TUserDocumentsChecksManager::TChecksConfiguration& checksConfig, const NDrive::IServer* server) {
    auto documentsSuite = documentsSuiteFuture.GetValue();
    auto& user = documentsSuite->User;
    auto& passportData = documentsSuite->PassportData;
    auto& licenseData = documentsSuite->DrivingLicenseData;

    auto DriveApi = server->GetDriveAPI();
    const auto& privateDataClient = DriveApi->GetPrivateDataClient();
    auto updatePassportFuture = passportData ? privateDataClient.UpdatePassport(user, documentsSuite->GetPassportRevision(), *passportData) : NThreading::MakeFuture();
    auto updateDrivingLicenseFuture = licenseData ? privateDataClient.UpdateDrivingLicense(user, documentsSuite->GetLicenseRevision(), *licenseData) : NThreading::MakeFuture();

    auto waiter = NThreading::WaitAll(updatePassportFuture, updateDrivingLicenseFuture);
    return waiter.Apply([
        documentsSuite,
        &user,
        &checksConfig,
        server,
        passportFuture = std::move(updatePassportFuture),
        licenseFuture = std::move(updateDrivingLicenseFuture)
    ](const NThreading::TFuture<void>&) -> NThreading::TFuture<void> {
        auto driveApi = server->GetDriveAPI();
        if (passportFuture.HasException()) {
            ERROR_LOG << GetTypeName() << ": Failed to update passport for : " << NThreading::GetExceptionInfo(passportFuture);
        }
        if (licenseFuture.HasException()) {
            ERROR_LOG << GetTypeName() << ": Failed to update license for: " << NThreading::GetExceptionInfo(licenseFuture);
        }
        if (passportFuture.HasException() || licenseFuture.HasException()) {
            throw TInternalDatasyncManagerError() << "Failed to upload user " << user.GetUserId() << " assignment data into Datasync";
        }

        TString operatorUserId = documentsSuite->OperatorUserId;
        auto session = driveApi->template BuildTx<NSQL::Writable>();
        if (!driveApi->GetUsersData()->UpdateUser(user, operatorUserId, session)) {
            session.Check();
        }

        for (const auto& yangAssignment: documentsSuite->YangData) {
            const auto& yangData = yangAssignment.second;

            TString secretId;
            if (!NJson::TryFromJson(yangData["tasks"][0]["input_values"]["secret"], secretId)) {
                throw TInternalDatasyncManagerError() << "Failed to parse secret id";
            }

            auto optionalAssignment = GetAssignment(secretId, driveApi, session);
            if (!optionalAssignment) {
                throw TInternalDatasyncManagerError() << "Failed to restore assignment by secret id " << secretId;
            }
            auto assignment = *optionalAssignment;
            assignment.SetProcessedAt(Now());

            if (assignment.GetFinalizedAt() != TInstant::Zero()) {
                assignment.SetNeedReprocessing(true);
            }

            if (!driveApi->GetDocumentPhotosManager().GetDocumentVerificationAssignments().Upsert(assignment, session)) {
                throw TInternalDatasyncManagerError() << "Can't upsert assignment " << assignment.GetId() << ": " << session.GetReport();
            }
        }

        const TUserDocumentsChecksManager* checksManager;
        if (!(checksManager = server->GetUserDocumentsChecksManager())) {
            throw TInternalDatasyncManagerError() << "User documents manager is not configured";
        }

        auto statusSetter = [&](NDrive::TEntitySession& chatSession, NDrive::TEntitySession& tagSession) {
            for (const auto& [_, checkPair] : documentsSuite->Verdicts) {
                const auto& check = checkPair.first;
                auto optionalPhotoIds = TUserDocumentsChecksManager::GetPhotoIds(check.GetType(), check.GetSecretId(), checksConfig.GenericMap, *server, tagSession);
                if (!optionalPhotoIds) {
                    throw TInternalDatasyncManagerError() << "Invalid photo status: " << chatSession.GetStringReport();
                }
                if (!checksManager->SetStatus(check, user.GetUserId(), *optionalPhotoIds, checksConfig, *server, chatSession, tagSession)) {
                    throw TInternalDatasyncManagerError() << "Failed to set status. Chat session: " << chatSession.GetStringReport() << " Tag session: " << tagSession.GetStringReport();
                }
            }
        };

        if (server->GetChatEngine()->GetDatabaseName() != server->GetDriveAPI()->GetDatabaseName()) {
            auto chatSession = server->GetChatEngine()->BuildSession(false);
            statusSetter(chatSession, session);
            if (!chatSession.Commit()) {
                throw TInternalDatasyncManagerError() << "Can't chat commit update: " << chatSession.GetReport();
            }
        } else {
            statusSetter(session, session);
        }


        if (!session.Commit()) {
            throw TInternalDatasyncManagerError() << "Can't commit update: " << session.GetReport();
        }
        return NThreading::MakeFuture();
    });
}

TExpectedState TRTDatasyncManager::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /* stateExt */, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    if (!server || !server->GetDriveAPI() || !server->GetDriveAPI()->HasDatasyncQueueClient()) {
        return nullptr;
    }
    const auto driveApi = server->GetDriveAPI();
    const auto& queueClient = *driveApi->GetDatasyncQueueClient();

    auto yangClient = server->GetYangClient();
    if (!yangClient) {
        return nullptr;
    }

    if (!server->GetDriveAPI()->HasDocumentPhotosManager()) {
        return nullptr;
    }

    TMap<TString, TDatasyncQueueEntry> queueEntries;
    {
        auto session = queueClient.BuildSession(true);
        queueEntries = queueClient.Get(BatchSize, session);
    }
    if (queueEntries.empty()) {
        // queue is empty - nothing to do
        return MakeAtomicShared<IRTBackgroundProcessState>();
    }

    TUserDocumentsChecksManager::TChecksConfiguration checksConfig;
    TMessagesCollector errors;
    if (!checksConfig.Init(
        ChecksSettingsGvars,
        ChecksGenericMapGvars,
        PhotoChecksPriorityGvars,
        ChatsToMoveGvars,
        *server,
        errors)
    ) {
        return MakeUnexpected<TString>(TStringBuilder() << "Failed init checks config: " << errors.GetStringReport());
    }

    TMap<TString, TDocumentsSuite> suites;

    {
        TMap<TString, TString> failedTasks;
        TVector<NThreading::TFuture<void>> yangFutures;
        TMap<TString, TString> assignmentsUserIds;
        for (const auto& queueEntry: queueEntries) {
            auto assignmentId = queueEntry.first;
            auto yangDataFuture = yangClient->RequestAssignmentAsync(assignmentId);
            auto future = yangDataFuture.Apply([
                this,
                assignmentId,
                driveApi,
                &suites,
                &checksConfig,
                &assignmentsUserIds,
                createdAt = queueEntry.second.GetCreatedAt()
            ](const NThreading::TFuture<NJson::TJsonValue>& r) -> void {
                auto yangData = r.GetValue();

                TYangJsonParser parser(std::move(yangData), ParserConfig);
                if (!parser.ParseChecks(assignmentId, checksConfig.Settings, driveApi)) {
                    if (parser.UserId) {
                        TGuard<TMutex> g(Mutex);
                        assignmentsUserIds[assignmentId] = parser.UserId;
                    }
                    throw TInternalDatasyncManagerError() << "Failed to get any documents data from the assignment " << assignmentId << ". Does it have different scheme?";
                }

                TGuard<TMutex> g(Mutex);
                assignmentsUserIds[assignmentId] = parser.UserId;
                suites[parser.UserId].AddYangData(std::move(parser), createdAt, GetRobotUserId());
            }).Apply([assignmentId, &failedTasks](const NThreading::TFuture<void>& r) -> void {
                if (r.HasException()) {
                    TGuard<TMutex> g(Mutex);
                    const TString exceptionMessage = "Failed to process assignment: " + assignmentId + ": " + NThreading::GetExceptionMessage(r);
                    ERROR_LOG << GetTypeName() << ": " << exceptionMessage << Endl;
                    failedTasks[assignmentId] = exceptionMessage;
                }
            });
            yangFutures.push_back(std::move(future));
        }
        NThreading::WaitAll(yangFutures).Wait();

        if (failedTasks) {
            // process queue entries that haven't made it to a suite
            auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
            for (const auto& task: failedTasks) {
                const auto& assignment = queueEntries[task.first]
                    .SetReturnReason(task.second)
                    .SetUserId(assignmentsUserIds[task.first]);
                if (!ReturnAssignment(assignment, queueClient, server, session)) {
                    return MakeUnexpected<TString>(TStringBuilder() << "Datasync queue manager failed for " << task.first << ": " << session.GetReport());
                }
            }
            if (!session.Commit()) {
                return MakeUnexpected<TString>(TStringBuilder() << "Datasync manager failed to commit changes to the queue: " << session.GetReport());
            }
        }
    }

    {
        auto session = driveApi->template BuildTx<NSQL::ReadOnly>();
        for (auto& suite: suites) {
            suite.second.User = std::move(GetUser(suite.first, driveApi, session));
        }
    }

    {
        TMap<TString, TString> failedSuites;
        TVector<NThreading::TFuture<void>> suitesFutures;
        for (auto& suite : suites) {
            auto userId = suite.first;

            auto taskFuture = suite.second.FillDocumentsData(server)
                .Apply([server, &checksConfig](const NThreading::TFuture<TDocumentsSuite*>& documentsSuiteFuture) -> NThreading::TFuture<void> {
                    return WriteToDatasync(documentsSuiteFuture, checksConfig, server);
                })
                .Apply([&userId, &failedSuites](const NThreading::TFuture<void>& r) -> NThreading::TFuture<void> {
                    if (r.HasException()) {
                        TGuard<TMutex> g(Mutex);
                        auto exceptionMessage = NThreading::GetExceptionMessage(r);
                        ERROR_LOG << GetTypeName() << ": Failed to process user " << userId << " assignment: " << exceptionMessage << Endl;
                        failedSuites[userId] = exceptionMessage;
                    }
                    return NThreading::MakeFuture();
                });

            suitesFutures.push_back(std::move(taskFuture));
        }
        NThreading::WaitAll(suitesFutures).Wait();

        auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
        for (const auto& suite: suites) {
            bool failed = failedSuites.contains(suite.first);
            for (const auto& yangResponse: suite.second.YangData) {
                auto assignmentId = yangResponse.first;
                auto queueEntry = queueEntries[assignmentId]
                    .SetReturnReason(failedSuites[suite.first])
                    .SetUserId(suite.first);
                if (!(failed
                    ? ReturnAssignment(queueEntry, queueClient, server, session)
                    : queueClient.Delete(assignmentId, session)))
                {
                    return MakeUnexpected<TString>(TStringBuilder() << "Datasync queue manager failed for " << assignmentId << ": " << session.GetReport());
                }
            }
        }

        if (!session.Commit()) {
            return MakeUnexpected<TString>(TStringBuilder() << "Datasync manager failed to commit changes to the queue: " << session.GetReport());
        }
    }


    if (AlertSize > 0) {
        auto session = server->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        auto size = queueClient.Size(session);
        if (size > AlertSize) {
            TUnistatSignalsCache::SignalAdd("datasync_queue", "size", size);
        }
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

bool TRTDatasyncManager::ReturnAssignment(const TDatasyncQueueEntry& entry, const TDatasyncQueueClient& client, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (entry.GetAttempt() < FallbackPolicy.GetAttemptLimit()) {
        return client.Return(entry, session);
    }

    const auto driveApi = Yensured(server->GetDriveAPI());
    switch (FallbackPolicy.GetType()) {
        case TFallbackPolicy::EType::AddTag: {
            if (!entry.HasUserId() || !entry.GetUserIdUnsafe()) {
                WARNING_LOG << "Failed to add tag for " << entry.GetAssignmentId()
                    << ": missing user id" << Endl;
                break;
            }

            if (!entry.HasReturnReason()) {
                session.SetErrorInfo(
                    "TRTDatasyncManager::ReturnAssignment",
                    "No user id or return reason specified for queue entry"
                );
                return false;
            }
            const auto& userId = entry.GetUserIdUnsafe();
            const auto& returnReason = entry.GetReturnReasonUnsafe();

            const auto& tagsManager =  driveApi->GetTagsManager();
            auto tag = tagsManager.GetTagsMeta().CreateTag(FallbackPolicy.GetTagName(), returnReason);
            if (!tag) {
                session.SetErrorInfo(
                    "TRTDatasyncManager::ReturnAssignment",
                    "Failed to create tag " + FallbackPolicy.GetTagName()
                );
                return false;
            }

            if (!tagsManager.GetUserTags().AddTag(tag, GetRobotUserId(), userId, server, session)) {
                session.AddErrorMessage(
                    "TRTDatasyncManager::ReturnAssignment",
                    "Failed to add tag"
                );
                return false;
            }
            break;
        }
        default: {
        }
    }

    return FallbackPolicy.GetRemoveFromQueue()
    ? client.Delete(entry.GetAssignmentId(), session)
    : client.Return(entry, session);
}

NDrive::TScheme TRTDatasyncManager::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSNumeric>("batch_size", "Размер батча").SetDefault(1).SetRequired(true);
    scheme.Add<TFSDuration>("request_timeout", "Таймаут запросов").SetDefault(TDuration::Seconds(10)).SetRequired(true);
    scheme.Add<TFSNumeric>("alert_size", "Лимит размера очереди для сигнала (0 - сигнал неактивен)").SetMin(0).SetRequired(false);

    {
        auto tab = scheme.StartTabGuard("paths");
        scheme.Add<TFSString>("pb_path", "Json-путь к лицевой стороне паспорта").SetDefault("solutions.[0].output_values.documents.passport_biographical.data").SetRequired(true);
        scheme.Add<TFSString>("pr_path", "Json-путь к странице регистрации паспорта").SetDefault("solutions.[0].output_values.documents.passport_registration.data").SetRequired(true);
        scheme.Add<TFSString>("lb_path", "Json-путь к обратной стороне ВУ").SetDefault("solutions.[0].output_values.documents.license_back.data").SetRequired(true);
        scheme.Add<TFSString>("lf_path", "Json-путь к лицевой стороне ВУ").SetDefault("solutions.[0].output_values.documents.license_front.data").SetRequired(true);
        scheme.Add<TFSString>("verdicts_path", "Json-путь к вердиктам проверок").SetDefault("solutions.[0].output_values.verdicts").SetRequired(true);
        scheme.Add<TFSString>("user_id_path", "Json-путь к user id").SetDefault("tasks.[0].input_values.user_id").SetRequired(true);

        scheme.Add<TFSString>("pool_id_path", "Json-путь к pool id").SetDefault("pool_id").SetRequired(false);
        scheme.Add<TFSString>("secret_id_path", "Json-путь к secret id").SetDefault("tasks.[0].input_values.secret").SetRequired(false);
        scheme.Add<TFSString>("worker_id_path", "Json-путь к worker id").SetDefault("worker_id").SetRequired(false);
        scheme.Add<TFSString>("timestamp_path", "Json-путь к timestamp").SetDefault("submitted").SetRequired(false);
        scheme.Add<TFSString>("comment_path", "Json-путь к комменту other").SetDefault("solutions.[0].output_values.other").SetRequired(false);
    }

    {
        auto tab = scheme.StartTabGuard("gvars");
        scheme.Add<TFSString>("checks_settings", "Имя gvars-переменной с типами проверок").SetDefault("user_documents_checks.settings").SetRequired(true);
        scheme.Add<TFSString>("generic_checks", "Имя gvars-переменной с соответствием проверок документам").SetDefault("user_documents_checks.generic_checks").SetRequired(true);
        scheme.Add<TFSString>("checks_priority", "Имя gvars-переменной с приоритетом вердиктов и с их соответствием вердиктам фото").SetDefault("user_documents_checks.photo_settings").SetRequired(true);
        scheme.Add<TFSString>("chats_to_move", "Имя gvars-переменной с id чатов, которые необходимо обновить при выставлении проверок").SetDefault("user_documents_checks.chats_to_move").SetRequired(true);
    }

    {
        auto tab = scheme.StartTabGuard("fallback");
        scheme.Add<TFSVariants>("fallback_type", "Политика возврата в очередь").InitVariants<TFallbackPolicy::EType>().SetDefault(::ToString(TFallbackPolicy::EType::None)).SetRequired(true);
        scheme.Add<TFSBoolean>("remove_from_queue", "Удалять из очереди").SetDefault(false);
        scheme.Add<TFSNumeric>("attempt_limit", "Допустимое кол-во попыток").SetDefault(Max<ui64>());
        {
            TSet<TString> tagNames;
            if (auto impl = server.GetAs<NDrive::IServer>()) {
                tagNames = impl->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TSimpleUserTag::TypeName });
            }
            scheme.Add<TFSVariants>("fallback_tag_name", "Тип навешиваемого тега").SetVariants(tagNames);
        }
    }
    return scheme;
}

void TRTDatasyncManager::TYangJsonParserConfig::SerializeToJson(NJson::TJsonValue& result) const {
    NJson::InsertField(result, "pb_path", PassportBiographicalPath);
    NJson::InsertField(result, "pr_path", PassportRegistrationPath);
    NJson::InsertField(result, "lf_path", LicenseFrontPath);
    NJson::InsertField(result, "lb_path", LicenseBackPath);
    NJson::InsertField(result, "user_id_path", UserIdPath);
    NJson::InsertField(result, "verdicts_path", VerdictsPath);
    NJson::InsertField(result, "pool_id_path", PoolIdPath);
    NJson::InsertField(result, "secret_id_path", SecretIdPath);
    NJson::InsertField(result, "worker_id_path", WorkerIdPath);
    NJson::InsertField(result, "timestamp_path", TimestampPath);
    NJson::InsertField(result, "comment_path", CommentPath);
}

void TRTDatasyncManager::TFallbackPolicy::SerializeToJson(NJson::TJsonValue& result) const {
    NJson::InsertField(result, "fallback_type", NJson::Stringify(Type));
    NJson::InsertField(result, "remove_from_queue", RemoveFromQueue);
    NJson::InsertField(result, "attempt_limit", AttemptLimit);
    NJson::InsertField(result, "fallback_tag_name", TagName);
}

NJson::TJsonValue TRTDatasyncManager::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "batch_size", BatchSize);
    NJson::InsertField(result, "request_timeout", GetRequestTimeout().ToString());
    NJson::InsertField(result, "alert_size", AlertSize);
    NJson::InsertField(result, "checks_settings", ChecksSettingsGvars);
    NJson::InsertField(result, "generic_checks", ChecksGenericMapGvars);
    NJson::InsertField(result, "checks_priority", PhotoChecksPriorityGvars);
    NJson::InsertField(result, "chats_to_move", ChatsToMoveGvars);
    ParserConfig.SerializeToJson(result);
    FallbackPolicy.SerializeToJson(result);
    return result;
}

bool TRTDatasyncManager::TYangJsonParserConfig::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return
        NJson::ParseField(jsonInfo, "pb_path", PassportBiographicalPath, true) &&
        NJson::ParseField(jsonInfo, "pr_path", PassportRegistrationPath, true) &&
        NJson::ParseField(jsonInfo, "lf_path", LicenseFrontPath, true) &&
        NJson::ParseField(jsonInfo, "lb_path", LicenseBackPath, true) &&
        NJson::ParseField(jsonInfo, "verdicts_path", VerdictsPath, true) &&
        NJson::ParseField(jsonInfo, "user_id_path", UserIdPath, false) &&
        NJson::ParseField(jsonInfo, "pool_id_path", PoolIdPath, false) &&
        NJson::ParseField(jsonInfo, "secret_id_path", SecretIdPath, false) &&
        NJson::ParseField(jsonInfo, "worker_id_path", WorkerIdPath, false) &&
        NJson::ParseField(jsonInfo, "timestamp_path", TimestampPath, false) &&
        NJson::ParseField(jsonInfo, "comment_path", CommentPath, false);
}

bool TRTDatasyncManager::TFallbackPolicy::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return
        NJson::ParseField(jsonInfo, "fallback_type", NJson::Stringify(Type), false) &&
        NJson::ParseField(jsonInfo, "fallback_tag_name", TagName, Type == EType::AddTag) &&
        NJson::ParseField(jsonInfo, "remove_from_queue", RemoveFromQueue, false) &&
        NJson::ParseField(jsonInfo, "attempt_limit", AttemptLimit, false);
}

bool TRTDatasyncManager::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return
        TBase::DoDeserializeFromJson(jsonInfo) &&
        NJson::ParseField(jsonInfo, "batch_size", BatchSize, true) &&
        NJson::ParseField(jsonInfo, "request_timeout", MutableProtectedRequestTimeout(), true) &&
        NJson::ParseField(jsonInfo, "alert_size", AlertSize, false) &&
        NJson::ParseField(jsonInfo, "checks_settings", ChecksSettingsGvars, true) &&
        NJson::ParseField(jsonInfo, "generic_checks", ChecksGenericMapGvars, false) &&
        NJson::ParseField(jsonInfo, "checks_priority", PhotoChecksPriorityGvars, false) &&
        NJson::ParseField(jsonInfo, "chats_to_move", ChatsToMoveGvars, false) &&
        ParserConfig.DoDeserializeFromJson(jsonInfo) &&
        FallbackPolicy.DoDeserializeFromJson(jsonInfo);

}
