#pragma once

#include <drive/backend/rt_background/manager/manager.h>
#include <drive/backend/users/user_documents_check.h>

#include <drive/backend/database/drive_api.h>
#include <drive/library/cpp/yang/client.h>
#include <drive/library/cpp/datasync_queue/client.h>

struct TInternalDatasyncManagerError : public virtual yexception {
};

class TRTDatasyncManager: public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;
    static TFactory::TRegistrator<TRTDatasyncManager> Registrator;

private:
    struct TYangJsonParserConfig {
        TString PassportBiographicalPath;
        TString PassportRegistrationPath;
        TString LicenseBackPath;
        TString LicenseFrontPath;

        TString VerdictsPath;

        TString UserIdPath;
        TString PoolIdPath;
        TString SecretIdPath;
        TString WorkerIdPath;
        TString TimestampPath;
        TString CommentPath;

        bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo);
        void SerializeToJson(NJson::TJsonValue& json) const;
    };

    class TYangJsonParser {
    public:
        TYangJsonParser(NJson::TJsonValue&& yangJson, const TYangJsonParserConfig& config)
        : YangJson(std::move(yangJson))
        , PassportBiographical(YangJson.GetValueByPath(config.PassportBiographicalPath))
        , PassportRegistration(YangJson.GetValueByPath(config.PassportRegistrationPath))
        , LicenseBack(YangJson.GetValueByPath(config.LicenseBackPath))
        , LicenseFront(YangJson.GetValueByPath(config.LicenseFrontPath))
        , Verdicts(YangJson.GetValueByPath(config.VerdictsPath))
        , PoolId(
            YangJson.GetValueByPath(config.PoolIdPath)
            ? YangJson.GetValueByPath(config.PoolIdPath)->GetString()
            : "")
        , SecretId(
            YangJson.GetValueByPath(config.SecretIdPath)
            ? YangJson.GetValueByPath(config.SecretIdPath)->GetString()
            : "")
        , WorkerId(
            YangJson.GetValueByPath(config.WorkerIdPath)
            ? YangJson.GetValueByPath(config.WorkerIdPath)->GetString()
            : "")
        , UserId(GetUserId(YangJson, config))
        , Timestamp(
            YangJson.GetValueByPath(config.TimestampPath) &&
            YangJson.GetValueByPath(config.TimestampPath)->IsString()
            ? TInstant::ParseIso8601(YangJson.GetValueByPath(config.TimestampPath)->GetString())
            : TInstant::Zero())
        , Comment(
            YangJson.GetValueByPath(config.CommentPath)
            ? YangJson.GetValueByPath(config.CommentPath)->GetString()
            : "")
        {
        }

        static TString GetUserId(const NJson::TJsonValue& yangData, const TYangJsonParserConfig& config);
        bool ParseChecks(const TString& assignmentId, const TMap<TString, TSet<TString>>& checksSettings, const TDriveAPI* driveApi);

    public:
        const NJson::TJsonValue YangJson;

        const NJson::TJsonValue* PassportBiographical;
        const NJson::TJsonValue* PassportRegistration;
        const NJson::TJsonValue* LicenseBack;
        const NJson::TJsonValue* LicenseFront;

        const NJson::TJsonValue* Verdicts;
        TMap<TString, TUserDocumentsCheck> VerdictsParsed;

        const TString PoolId;
        const TString SecretId;
        const TString WorkerId;
        TString UserId;
        const TInstant Timestamp;
        const TString Comment;
    };

    struct TDocumentsSuite {
    public:
        TDriveUserData User;
        TString OperatorUserId;

        TMap<TString, NJson::TJsonValue> YangData;
        TMap<TString, std::pair<TUserDocumentsCheck, TInstant>> Verdicts;
        TVector<TString> FraudReasons;

        TMaybe<TUserPassportData> PassportDataPatch;
        TMaybe<TUserDrivingLicenseData> DrivingLicenseDataPatch;

        TString PassportRevision;
        TInstant PassportTimestamp = TInstant::Zero();

        TString LicenseRevision;
        TInstant LicenseTimestamp = TInstant::Zero();

        TMaybe<TUserPassportData> PassportData;
        TMaybe<TUserDrivingLicenseData> DrivingLicenseData;

    public:
        NThreading::TFuture<TDocumentsSuite*> FillDocumentsData(const NDrive::IServer* server);
        void AddYangData(TYangJsonParser&& parser, const TInstant& createdAt, const TString& operatorUserId);

        TString GetPassportRevision() const;
        TString GetLicenseRevision() const;
    };

public:
    class TFallbackPolicy {
    public:
        enum class EType {
            None /* "none" */,
            AddTag /* "add_tag" */
        };

    public:
        bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo);
        void SerializeToJson(NJson::TJsonValue& json) const;

    private:
        R_READONLY(EType, Type, EType::None);
        R_READONLY(bool, RemoveFromQueue, false);
        R_READONLY(ui64, AttemptLimit, Max<ui64>());
        R_READONLY(TString, TagName);
    };

private:
    R_READONLY(TString, ChecksSettingsGvars);
    R_READONLY(TString, ChecksGenericMapGvars);
    R_READONLY(TString, PhotoChecksPriorityGvars);
    R_READONLY(TString, ChatsToMoveGvars);

    R_READONLY(ui64, BatchSize, 1);
    R_READONLY(TDuration, RequestTimeout, TDuration::Seconds(10));
    R_READONLY(TYangJsonParserConfig, ParserConfig);
    R_READONLY(ui64, AlertSize, 0);
    R_READONLY(TFallbackPolicy, FallbackPolicy);

private:
    bool ReturnAssignment(const TDatasyncQueueEntry& entry, const TDatasyncQueueClient& client, const NDrive::IServer* server, NDrive::TEntitySession& session) const;

    static TDriveUserData GetUser(const TString& userId, const TDriveAPI* driveApi, NDrive::TEntitySession& session);
    static TMaybe<TYangDocumentVerificationAssignment> GetAssignment(const TString& secretId, const TDriveAPI* driveApi, NDrive::TEntitySession& session);
    static NThreading::TFuture<void> WriteToDatasync(const NThreading::TFuture<TDocumentsSuite*>& documentsSuiteFuture, const TUserDocumentsChecksManager::TChecksConfiguration& checksConfig, const NDrive::IServer* server);

public:
    using TBase::TBase;

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /* stateExt */, const TExecutionContext& context) const override;

    static TString GetTypeName() {
        return "yang_to_datasync_queue_manager";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;

    virtual NJson::TJsonValue DoSerializeToJson() const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
};
