#include "config.h"

#include <drive/backend/doc_packages/builder.h>
#include <drive/backend/doc_packages/manager.h>

#include <drive/library/cpp/mds/client.h>

#include <rtline/util/types/uuid.h>

TRTDocumentBuilderBackground::TFactory::TRegistrator<TRTDocumentBuilderBackground> TRTDocumentBuilderBackground::Registrator(TRTDocumentBuilderBackground::GetTypeName());

TExpectedState TRTDocumentBuilderBackground::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& frServer = context.GetServerAs<NDrive::IServer>();
    if (!frServer.GetDocumentsManager()) {
        ERROR_LOG << "Document manager not initialized" << Endl;
        return MakeUnexpected<TString>({});
    }
    if (!frServer.GetDriveAPI()->HasMDSClient()) {
        ERROR_LOG << "MDS client not initialized" << Endl;
        return MakeUnexpected<TString>({});
    }

    const TS3Client::TBucket* bucket = frServer.GetDriveAPI()->GetMDSClient().GetBucket(BucketName);
    if (!bucket) {
        ERROR_LOG << "Incorrect bucket name" << Endl;
        return MakeUnexpected<TString>({});
    }

    TVector<TQueuedDocument> documents;
    if (!frServer.GetDocumentsManager()->GetQueuedDocuments(documents, EAssemblerStage::NotReady)) {
        ERROR_LOG << "Cannot fetch queue's documents from cache" << Endl;
        return MakeUnexpected<TString>({});
    }
    for (auto&& doc : documents) {
        TString queue = doc.GetDocumentQueue();
        TString type = doc.GetDocumentFormat();
        {
            auto document = frServer.GetDocumentsManager()->GetQueuedDocument(doc.GetId());
            if (!document || document->GetStatus() != EAssemblerStage::NotReady) {
                continue;
            }
            queue = document->GetDocumentQueue();
        }
        {
            TVector<TDocumentDescriptionPtr> documents;
            if (!frServer.GetDocumentsManager()->GetRegisteredDocuments(documents, { doc.GetDocumentName() }, true)) {
                return MakeUnexpected<TString>({});
            }
            if (documents.empty() || !documents.front()) {
                continue;
            }
            type = (type ? type : documents.front()->GetDocumentFormat());
            queue = (queue ? queue : documents.front()->GetQueue());
        }
        if (!queue || !Queues.contains(queue)) {
            continue;
        }
        type = (type ? type : "txt");

        TAtomicSharedPtr<IDocumentAssembler> builderPtr = nullptr;
        if (type == "pdf") {
            THolder<TTexServerDocBuilder> texBuilder(new TTexServerDocBuilder(TexConfig));
            if (!frServer.GetDocumentsManager()->InitializeTexBuilder(*texBuilder)) {
                return MakeUnexpected<TString>({});
            }

            TStringBuilder header;
            header << texBuilder->GetHeader() << Endl;

            //add watermark
            header << "\\newsavebox\\userwatermark" << Endl;
            header << "\\savebox\\userwatermark{ \\tikz[color = red!33!green!33!blue!33,opacity = 0.5]\\node{" + doc.GetAuthor() + "}; }" << Endl;
            texBuilder->SetHeader(header);

            builderPtr = TAtomicSharedPtr<IDocumentAssembler>(texBuilder.Release());
        } else if (type == "csv") {
            builderPtr = TAtomicSharedPtr<IDocumentAssembler>(new TCsvDocBuilder());
        } else {
            builderPtr = TAtomicSharedPtr<IDocumentAssembler>(new TRawDocBuilder());
        }

        TUserPermissions::TPtr userPermissions = frServer.GetDriveAPI()->GetUserPermissions(doc.GetAuthor(), TUserPermissionsFeatures());
        builderPtr->SetAuthorId(userPermissions->GetUid());

        TMessagesCollector errors;
        TBlob document;
        try {
            document = frServer.GetDocumentsManager()->BuildDocument(doc.GetInputParameters(), frServer, *builderPtr, userPermissions, errors);
        } catch(const std::exception& e) {
            errors.AddMessage("document_builder", "fail to build document: " + FormatExc(e));
            document = Default<TBlob>();
        }
        NDrive::TEventLog::Log("BuildDocumentFinished", NJson::TMapBuilder
            ("doc_id", doc.GetId())
            ("post_data", doc.GetInputParameters())
        );

        TString documentStr = TString((char*)document.Data(), document.Size());
        if (document.Empty()) {
            doc.SetStatus(EAssemblerStage::Error).SetError(errors.GetStringReport());
        } else {
            NJson::TJsonValue resultOutput;
            TString errorMessage;
            for (auto&& eStrorage : doc.GetStorages()) {
                switch (eStrorage) {
                case EDocumentStorage::MDS: {
                    TString mdsKey = doc.GetDocumentName() + "_" + doc.GetId();
                    if (type == "raw") {
                        mdsKey += ".txt";
                    } else {
                        mdsKey += "." + type;
                    }

                    if (bucket->PutKey(mdsKey, documentStr, errors) != HTTP_OK) {
                        errorMessage = "mds not available " + errors.GetStringReport();
                    } else {
                        resultOutput.InsertValue("link", frServer.GetDriveAPI()->GetMDSClient().GetTmpFilePath(BucketName, mdsKey));
                        if (errors.HasMessages()) {
                            resultOutput.InsertValue("warnings", errors.GetStringReport());
                        }
                    }
                    break;
                }
                case EDocumentStorage::Base64:
                    resultOutput.InsertValue("data", Base64Encode(documentStr));
                    break;
                case EDocumentStorage::Raw:
                    resultOutput.InsertValue("data", documentStr);
                    break;
                case EDocumentStorage::Json:
                    if (!NJson::ReadJsonFastTree(documentStr, &resultOutput)) {
                        errorMessage = "incorrect json " + documentStr;
                    }
                    break;
                default:
                    break;
                }
            }
            if (!errorMessage) {
                for (const auto& notifierConfig : doc.GetNotifiers()) {
                    NDrive::INotifier::TMessage message(notifierConfig.GetName(), documentStr);
                    message.SetAdditionalInfo(notifierConfig.GetDescription());

                    TString mimeType;
                    message.SetTitle(notifierConfig.GetTitle() + "." + type);
                    if (type == "pdf") {
                        mimeType = "application/pdf";
                    }
                    if (type == "csv") {
                        mimeType = "text/csv";
                    }
                    if (!mimeType) {
                        mimeType = "text/plain";
                    }
                    if (!NDrive::INotifier::SendDocument(frServer.GetNotifier(notifierConfig.GetNotifier()), message, mimeType, NDrive::INotifier::TContext().SetServer(&frServer))) {
                        resultOutput.InsertValue("notifier_error", true);
                    }
                }
                doc.SetStatus(EAssemblerStage::Done).SetResultOutput(resultOutput);
            } else {
                doc.SetStatus(EAssemblerStage::Error).SetError(errorMessage);
            }
        }

        auto session = frServer.GetDriveAPI()->template BuildTx<NSQL::Writable>();
        if (!frServer.GetDocumentsManager()->UpsertQueuedDocument(doc, GetRobotUserId(), session) || !session.Commit()) {
            ERROR_LOG << session.GetStringReport() << Endl;
            return MakeUnexpected<TString>({});
        }
    }
    return new IRTBackgroundProcessState();
}

NDrive::TScheme GetTexConfigScheme() {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("folder").SetDefault(".");
    scheme.Add<TFSDuration>("convert_timeout").SetDefault(TDuration::Seconds(5));
    return scheme;
}

NDrive::TScheme TRTDocumentBuilderBackground::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("bucket");
    scheme.Add<TFSStructure>("tex_config").SetStructure(GetTexConfigScheme());
    scheme.Add<TFSArray>("queues").SetElement<TFSString>().SetRequired(true);
    return scheme;
}

bool TRTDocumentBuilderBackground::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING(jsonInfo, "bucket", BucketName);
    Queues.clear();
    JREAD_CONTAINER_OPT(jsonInfo, "queues", Queues);
    return TexConfig.DeserializeFromJson(jsonInfo["tex_config"]) && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTDocumentBuilderBackground::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "bucket", BucketName);
    TJsonProcessor::Write(result, "tex_config", TexConfig.SerializeToJson());
    TJsonProcessor::WriteContainerArray(result, "queues", Queues);
    return result;
}


TRTDocumentCleanerBackground::TFactory::TRegistrator<TRTDocumentCleanerBackground> TRTDocumentCleanerBackground::Registrator(TRTDocumentCleanerBackground::GetTypeName());

TExpectedState TRTDocumentCleanerBackground::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& frServer = context.GetServerAs<NDrive::IServer>();
    if (!frServer.GetDocumentsManager()) {
        ERROR_LOG << "Document manager not initialized" << Endl;
        return MakeUnexpected<TString>({});
    }
    if (!frServer.GetDriveAPI()->HasMDSClient()) {
        ERROR_LOG << "MDS client not initialized" << Endl;
        return MakeUnexpected<TString>({});
    }

    const TS3Client::TBucket* bucket = frServer.GetDriveAPI()->GetMDSClient().GetBucket(BucketName);
    if (!bucket) {
        ERROR_LOG << "Incorrect bucket name" << Endl;
        return MakeUnexpected<TString>({});
    }

    TString defaultLinkPrefix = frServer.GetDriveAPI()->GetMDSClient().GetTmpFilePath(BucketName, "");
    defaultLinkPrefix = defaultLinkPrefix.substr(defaultLinkPrefix.find_first_of(':') + 1);
    TVector<TQueuedDocument> documents;
    if (!frServer.GetDocumentsManager()->GetQueuedDocuments(documents, {})) {
        ERROR_LOG << "Cannot fetch queue documents from cache" << Endl;
        return MakeUnexpected<TString>({});
    }
    for (auto&& doc : documents) {
        if (doc.GetStatus() == EAssemblerStage::NotReady || doc.GetLastUpdate() + CriticalTime > StartInstant) {
            continue;
        }
        TString failedMessage;
        if (doc.GetResultOutput().Has("link")) {
            TString key = doc.GetResultOutput()["link"].GetStringRobust();
            key = key.substr(key.find_first_of(':') + 1);
            key = key.substr(defaultLinkPrefix.size());

            TMessagesCollector errors;
            if (bucket->DeleteKey(key, errors) != HTTP_NO_CONTENT) {
                failedMessage = errors.GetStringReport();
            }
        }
        {
            auto session = frServer.GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!failedMessage && (!frServer.GetDocumentsManager()->RemoveQueuedDocument(doc.GetId(), GetRobotUserId(), session) || !session.Commit())) {
                ERROR_LOG << session.GetStringReport() << Endl;
                failedMessage += " " + session.GetStringReport();
            }
        }
        if (failedMessage) {
            doc.SetStatus(EAssemblerStage::FailedDeletion);
            doc.SetError(failedMessage);
            auto session = frServer.GetDriveAPI()->template BuildTx<NSQL::Writable>();
            if (!frServer.GetDocumentsManager()->UpsertQueuedDocument(doc, GetRobotUserId(), session) || !session.Commit()) {
                ERROR_LOG << session.GetStringReport() << Endl;
            }
        }
    }
    return new IRTBackgroundProcessState();
}

NDrive::TScheme TRTDocumentCleanerBackground::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("bucket");
    scheme.Add<TFSDuration>("critical_time").SetDefault(TDuration::Hours(3));
    return scheme;
}

bool TRTDocumentCleanerBackground::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING(jsonInfo, "bucket", BucketName);
    JREAD_DURATION(jsonInfo, "critical_time", CriticalTime);
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTDocumentCleanerBackground::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "bucket", BucketName);
    TJsonProcessor::WriteDurationString(result, "critical_time", CriticalTime);
    return result;
}

TRTFiltredSessionsDocumentShedulerBackground::TFactory::TRegistrator<TRTFiltredSessionsDocumentShedulerBackground> TRTFiltredSessionsDocumentShedulerBackground::Registrator(TRTFiltredSessionsDocumentShedulerBackground::GetTypeName());
TDocumentShedulerState::TFactory::TRegistrator<TDocumentShedulerState> TDocumentShedulerState::Registrator(TRTFiltredSessionsDocumentShedulerBackground::GetTypeName());

TString TDocumentShedulerState::GetType() const {
    return TRTFiltredSessionsDocumentShedulerBackground::GetTypeName();
}

TExpectedState TRTFiltredSessionsDocumentShedulerBackground::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const NDrive::IServer& frServer = context.GetServerAs<NDrive::IServer>();

    THolder<TDocumentShedulerState> result = MakeHolder<TDocumentShedulerState>();
    const TDocumentShedulerState* state = dynamic_cast<const TDocumentShedulerState*>(stateExt.Get());

    TInstant reportInstant = state ? state->GetLastInstant() : FirstReportInstant;
    result->SetLastInstant(reportInstant);

    if (reportInstant > ModelingNow()) {
        return result.Release();
    }

    if (!frServer.GetDocumentsManager()) {
        ERROR_LOG << "Document manager not initialized" << Endl;
        return nullptr;
    }

    TFiltredSessionsDocument::TContext localContext = Context;
    localContext.SetSince(reportInstant - StartShift);
    localContext.SetUntil(reportInstant - FinishShift);

    NJson::TJsonValue docInputs = localContext.SerializeToJson();
    docInputs["document_name"] = DocumentName;
    docInputs["storage"] = "mds";

    for (const auto& notifier : Notifiers) {
        TQueuedDocument::TNotifier localNotifier = notifier;
        localNotifier.SetName(notifier.GetName() + reportInstant.FormatLocalTime("-%Y-%m-%d"));
        localNotifier.SetTitle(notifier.GetTitle() + reportInstant.FormatLocalTime("-%Y-%m-%d"));
        localNotifier.SetDescription(notifier.GetDescription() + " Период" + localContext.GetSince().FormatLocalTime(" %Y-%m-%d %R") + localContext.GetUntil().FormatLocalTime(" - %Y-%m-%d %R"));

        docInputs["notifiers"].AppendValue(localNotifier.SerializeToJson());
    }

    TQueuedDocument newDocument;
    if (!newDocument.SetId(NUtil::CreateUUID()).SetStatus(EAssemblerStage::NotReady).SetAuthor(GetRobotUserId()).SetInputParameters(docInputs)) {
        ERROR_LOG << "incorrect doc input" << Endl;
        return result.Release();
    }

    auto session = frServer.GetDriveAPI()->template BuildTx<NSQL::Writable>();
    if (!frServer.GetDocumentsManager()->UpsertQueuedDocument(newDocument, GetRobotUserId(), session) || !session.Commit()) {
        ERROR_LOG << session.GetStringReport() << Endl;
        return result.Release();
    }
    result->SetLastInstant(reportInstant + ReportInterval);
    return result.Release();
}

NDrive::TScheme TRTFiltredSessionsDocumentShedulerBackground::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);

    auto serverImpl = server.GetAs<NDrive::IServer>();
    TVector<TDocumentDescriptionPtr> documentsPtr;
    if (serverImpl && serverImpl->GetDocumentsManager()) {
        serverImpl->GetDocumentsManager()->GetRegisteredDocuments(documentsPtr, true);
    }
    TSet<TString> names;
    for (const auto& doc : documentsPtr) {
        names.insert(doc.GetInternalId());
    }
    scheme.Add<TFSVariants>("document_name", "Имя документа").SetVariants(names).SetRequired(true);

    TFiltredSessionsDocument::TContext::AddScheme(scheme.Add<TFSStructure>("context", "Параметры документа").SetStructure<NDrive::TScheme>(), server);
    scheme.Add<TFSArray>("notifiers", "Нотификация").SetElement(TQueuedDocument::TNotifier::GetScheme(server)).SetRequired(true);

    scheme.Add<TFSNumeric>("first_report_time", "Время построения первого отчета").SetVisual(TFSNumeric::EVisualType::DateTime).SetRequired(true);
    scheme.Add<TFSDuration>("report_interval", "Интервал между репортами").SetRequired(true);
    scheme.Add<TFSDuration>("start_shift", "Сдвиг начала").SetRequired(true);
    scheme.Add<TFSDuration>("finish_shift", "Сдвиг окончания").SetRequired(true);
    return scheme;
}

bool TRTFiltredSessionsDocumentShedulerBackground::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING(jsonInfo, "document_name", DocumentName);
    JREAD_INSTANT(jsonInfo, "first_report_time", FirstReportInstant);
    JREAD_DURATION(jsonInfo, "report_interval", ReportInterval);
    JREAD_DURATION(jsonInfo, "start_shift", StartShift);
    JREAD_DURATION(jsonInfo, "finish_shift", FinishShift);
    if (jsonInfo.Has("notifier")) {
        TQueuedDocument::TNotifier notifier;
        if (!notifier.DeserializeFromJson(jsonInfo["notifier"])) {
            return false;
        }
        Notifiers.emplace_back(std::move(notifier));
    }
    TMessagesCollector errors;
    return Context.DeserializeFromJson(jsonInfo["context"], errors) && NJson::ParseField(jsonInfo, "notifiers", Notifiers) && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTFiltredSessionsDocumentShedulerBackground::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "context", Context.SerializeToJson());
    TJsonProcessor::Write(result, "document_name", DocumentName);
    TJsonProcessor::Write(result, "notifiers", NJson::ToJson(Notifiers));
    TJsonProcessor::WriteInstant(result, "first_report_time", FirstReportInstant);
    TJsonProcessor::WriteDurationString(result, "report_interval", ReportInterval);
    TJsonProcessor::WriteDurationString(result, "start_shift", StartShift);
    TJsonProcessor::WriteDurationString(result, "finish_shift", FinishShift);
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TRTDocumentDeadlineWatcherBackground::TDocumentNotification& object) {
    return object.ToJson();
}
template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TRTDocumentDeadlineWatcherBackground::TDocumentNotification& result) {
    return result.FromJson(value);
}

TRTDocumentDeadlineWatcherBackground::TFactory::TRegistrator<TRTDocumentDeadlineWatcherBackground> TRTDocumentDeadlineWatcherBackground::Registrator(TRTDocumentDeadlineWatcherBackground::GetTypeName());

TExpectedState TRTDocumentDeadlineWatcherBackground::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*stateExt*/, const TExecutionContext& context) const {
    const NDrive::IServer& frServer = context.GetServerAs<NDrive::IServer>();

    if (!frServer.GetDocumentsManager()) {
        ERROR_LOG << "Document manager not initialized" << Endl;
        return nullptr;
    }

    TSet<TString> names;
    for(const auto& doc : Documents) {
        names.insert(doc.GetName());
    }

    TVector<TDocumentDescriptionPtr> result;
    if (!frServer.GetDocumentsManager()->GetRegisteredDocuments(result, names) || result.size() != names.size()) {
        ERROR_LOG << "Cannot restore documents" << Endl;
        return nullptr;
    }

    for (const auto& docPtr : result) {
        for (const auto& docDeadline : Documents) {
            if (docPtr->GetName() == docDeadline.GetName() && (ModelingNow() + docDeadline.GetCriticalTime() > docPtr->GetDeadline())) {
                NDrive::INotifier::Notify(frServer.GetNotifier(NotifierName), "Истекает время жизни документа " + docPtr->GetComment() + " " + docPtr->GetDeadline().FormatLocalTime("%d.%m.%Y"));
            }
        }
    }

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TRTDocumentDeadlineWatcherBackground::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier_name", "Нотифаер").SetVariants(server.GetNotifierNames()).SetRequired(true);
    scheme.Add<TFSArray>("documents", "Документы").SetElement(TDocumentNotification::GetScheme(server)).SetRequired(true);
    return scheme;
}

bool TRTDocumentDeadlineWatcherBackground::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return NJson::ParseField(jsonInfo, "notifier_name", NotifierName, true)
        && NJson::ParseField(jsonInfo, "documents", Documents, true)
        && TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TRTDocumentDeadlineWatcherBackground::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "notifier_name", NotifierName);
    TJsonProcessor::Write(result, "documents", NJson::ToJson(Documents));
    return result;
}

