#pragma once
#include <drive/backend/rt_background/common/config.h>
#include <drive/backend/rt_background/common/state.h>

#include <drive/backend/background/manager/regular.h>
#include <drive/backend/doc_packages/manager.h>
#include <drive/backend/doc_packages/impl/document_repeated.h>
#include <drive/backend/user_document_photos/manager.h>

#include <rtline/util/types/accessor.h>

class TRTDocumentBuilderBackground : public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;
    R_READONLY(TString, BucketName);
    R_READONLY(NTexBuilder::TTexBuilderConfig, TexConfig);
    R_READONLY(TSet<TString>, Queues, { "default" });
private:
    static TFactory::TRegistrator<TRTDocumentBuilderBackground> Registrator;
protected:
    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;
public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    using TBase::TBase;

    static TString GetTypeName() {
        return "document_builder";
    }
};

class TRTDocumentCleanerBackground : public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;
    R_READONLY(TString, BucketName);
    R_READONLY(TDuration, CriticalTime, TDuration::Minutes(30));
private:
    static TFactory::TRegistrator<TRTDocumentCleanerBackground> Registrator;
protected:
    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;
public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    using TBase::TBase;

    static TString GetTypeName() {
        return "document_cleaner";
    }
};

class TDocumentShedulerState : public TRTInstantWatcherState {
    static TFactory::TRegistrator<TDocumentShedulerState> Registrator;
public:
    virtual TString GetType() const override;
};

class TRTFiltredSessionsDocumentShedulerBackground : public IRTRegularBackgroundProcess {
private:
    using TBase = IRTRegularBackgroundProcess;
    R_READONLY(TFiltredSessionsDocument::TContext, Context);
    R_READONLY(TString, DocumentName);
    R_READONLY(TVector<TQueuedDocument::TNotifier>, Notifiers);
    R_READONLY(TInstant, FirstReportInstant);
    R_READONLY(TDuration, ReportInterval);
    R_READONLY(TDuration, StartShift);
    R_READONLY(TDuration, FinishShift);

private:
    static TFactory::TRegistrator<TRTFiltredSessionsDocumentShedulerBackground> Registrator;
protected:
    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;
public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    using TBase::TBase;

    static TString GetTypeName() {
        return "filtred_sessions_document_scheduler";
    }
};

class TRTDocumentDeadlineWatcherBackground : public IRTRegularBackgroundProcess {
public:
    using TBase = IRTRegularBackgroundProcess;
    class TDocumentNotification {
        R_READONLY(TString, Name);
        R_READONLY(TDuration, CriticalTime);

    public:
        NJson::TJsonValue ToJson() const {
            NJson::TJsonValue result;
            NJson::InsertField(result, "name", Name);
            NJson::InsertField(result, "critical_time", CriticalTime);
            return result;
        }

        bool FromJson(const NJson::TJsonValue& json) {
            return NJson::ParseField(json, "name", Name, true)
                && NJson::ParseField(json, "critical_time", CriticalTime, true);
        }

        static NDrive::TScheme GetScheme(const IServerBase& server) {
            NDrive::TScheme result;
            auto serverImpl = server.GetAs<NDrive::IServer>();
            TVector<TDocumentDescriptionPtr> documentsPtr;
            if (serverImpl && serverImpl->GetDocumentsManager()) {
                serverImpl->GetDocumentsManager()->GetRegisteredDocuments(documentsPtr);
            }
            TSet<TString> names;
            for (const auto& doc : documentsPtr) {
                names.insert(doc.GetInternalId());
            }

            result.Add<TFSVariants>("name", "Имя документа").SetVariants(names).SetRequired(true);
            result.Add<TFSDuration>("critical_time", "Уведомить за").SetRequired(true);
            return result;
        }
    };

    R_READONLY(TString, NotifierName);
    R_READONLY(TVector<TDocumentNotification>, Documents);
private:
    static TFactory::TRegistrator<TRTDocumentDeadlineWatcherBackground> Registrator;
protected:
    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;
    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;
public:
    virtual TString GetType() const override {
        return GetTypeName();
    }

    using TBase::TBase;

    static TString GetTypeName() {
        return "document_deadline_watcher";
    }
};
