#include "processor.h"

#include "state.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/documents_verification/manager.h>

#include <mapreduce/yt/interface/client.h>

#include <util/generic/size_literals.h>

TDocumentPhotosDownload::TFactory::TRegistrator<TDocumentPhotosDownload> TDocumentPhotosDownload::Registrator(GetTypeName());

TExpectedState TDocumentPhotosDownload::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> stateExt, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();
    const TRTDocumentPhotosDownloadState* state = dynamic_cast<const TRTDocumentPhotosDownloadState*>(stateExt.Get());
    ui32 processedRows = state ? state->GetProcessedRows() : 0;

    const auto ytClient = NYT::CreateClient(YtCluster, NYT::TCreateClientOptions());
    const auto readRange = NYT::TReadRange::FromRowIndices(processedRows, processedRows + BatchSize);
    const auto inputTablePath = NYT::TRichYPath(InputPhotoIdsTablePath).AddRange(readRange);
    const auto reader = ytClient->CreateTableReader<NYT::TNode>(inputTablePath);

    const auto now = TInstant::Now();

    if (reader->IsValid()) {
        const auto tx = ytClient->StartTransaction();
        const auto outputTablePath = NYT::TRichYPath(OutputPhotoBytesTablePath).Append(true);
        const auto writerOptions = NYT::TTableWriterOptions().Config(NYT::TNode()("max_row_weight", 128_MB));
        const auto writer = tx->CreateTableWriter<NYT::TNode>(outputTablePath, writerOptions);
        const auto& documentPhotosManager = server.GetDriveAPI()->GetDocumentPhotosManager();

        for (; reader->IsValid(); reader->Next()) {
            const NYT::TNode& inputRow = reader->GetRow();
            const auto& photoId = inputRow["photo_id"].AsString();
            TString photoBytes;

            Y_ENSURE(documentPhotosManager.GetDocumentPhoto(photoId, photoBytes, server),
                     "Failed to get content for photo_id " << photoId);

            NYT::TNode outputRow;
            outputRow("photo_id", photoId);
            outputRow("photo_bytes", photoBytes);
            outputRow("timestamp", now.MicroSeconds());

            writer->AddRow(outputRow);
            processedRows++;
        }

        writer->Finish();
        tx->Commit();
    }

    auto newState = MakeHolder<TRTDocumentPhotosDownloadState>();
    newState->SetProcessedRows(processedRows);
    return newState.Release();
}

NDrive::TScheme TDocumentPhotosDownload::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("yt_cluster", "Кластер YT").SetDefault("hahn");
    scheme.Add<TFSString>("input_photo_ids_table_path", "Входная таблица с айдишниками фотографий для скачивания").SetRequired(true);
    scheme.Add<TFSString>("output_photo_bytes_table_path", "Выходная таблица для записи результата").SetRequired(true);
    scheme.Add<TFSNumeric>("batch_size", "Размер пачки обрабатываемых фотографий").SetDefault(100);
    return scheme;
}

NJson::TJsonValue TDocumentPhotosDownload::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    NJson::InsertField(result, "yt_cluster", YtCluster);
    NJson::InsertField(result, "input_photo_ids_table_path", InputPhotoIdsTablePath);
    NJson::InsertField(result, "output_photo_bytes_table_path", OutputPhotoBytesTablePath);
    NJson::InsertField(result, "batch_size", BatchSize);
    return result;
}

bool TDocumentPhotosDownload::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    return NJson::ParseField(jsonInfo["yt_cluster"], YtCluster) &&
           NJson::ParseField(jsonInfo["input_photo_ids_table_path"], InputPhotoIdsTablePath) &&
           NJson::ParseField(jsonInfo["output_photo_bytes_table_path"], OutputPhotoBytesTablePath) &&
           NJson::ParseField(jsonInfo["batch_size"], BatchSize);
}
