#include "config.h"

#include <drive/backend/database/drive_api.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/container.h>

#include <util/string/join.h>

TRTDropCarTagPerformer::TFactory::TRegistrator<TRTDropCarTagPerformer> TRTDropCarTagPerformer::Registrator(TRTDropCarTagPerformer::GetTypeName());
TRTDropUserTagPerformer::TFactory::TRegistrator<TRTDropUserTagPerformer> TRTDropUserTagPerformer::Registrator(TRTDropUserTagPerformer::GetTypeName());

TVector<TString> IRTPerformerDropper::GetPerformers(const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    if (!PerformerTagNames) {
        return {};
    }

    if (!server->GetDriveAPI()) {
        throw yexception() << "DriveAPI is not defined";
    }

    const auto& userTagsManager = server->GetDriveAPI()->GetEntityTagsManager(NEntityTagsManager::EEntityType::User);

    TVector<TDBTag> offDutyTags;
    if (!userTagsManager.RestoreTags({}, PerformerTagNames, offDutyTags, session)) {
        session.Check();
        return {};
    }

    TSet<TString> performersIds;
    for (const auto& performerTag: offDutyTags) {
        performersIds.insert(performerTag.GetObjectId());
    }

    return TVector<TString>(performersIds.begin(), performersIds.end());
}

TVector<TDBTag> IRTPerformerDropper::GetTagsToDropPerformer(const IRTBackgroundProcess::TExecutionContext& context, NDrive::TEntitySession& session) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    if (!server) {
        throw yexception() << "failed to get server from a context";
    }

    IEntityTagsManager::TQueryOptions options;
    auto performersIds = GetPerformers(server, session);
    if (performersIds) {
        options.SetPerformers(performersIds);
    }
    auto objectsIds = GetFilteredObjects(context);
    if (!!objectsIds) {
        options.SetObjectIds(objectsIds);
    }
    if (!TagNames) {
        return {};
    }
    options.SetTags(TagNames);

    const auto& objectTagsManager = GetObjectsTagsManager(server);
    auto tags = objectTagsManager.RestoreTags(session, std::move(options));
    if (!tags) {
        session.Check();
        return {};
    }

    if (ActivityPeriod && !FilterTagsByActivity(*tags, server, session)) {
        return {};
    }

    return *tags;
}

bool IRTPerformerDropper::FilterTagsByActivity(TVector<TDBTag>& tags, const NDrive::IServer* server, NDrive::TEntitySession& session) const {
    const auto& objectTagsManager = GetObjectsTagsManager(server);
    IEntityTagsManager::TQueryOptions options;

    TVector<TString> tagIds;
    for (const auto& tag: tags) {
        tagIds.push_back(tag.GetTagId());
    }
    options.SetTagIds(tagIds);

    auto activityLimit = Now() - *ActivityPeriod;
    auto events = objectTagsManager.GetEvents(activityLimit, session, std::move(options));
    if (!events || events->empty()) {
        session.Check();
        return !tags.empty();
    }

    TMap<TString, TDBTag> tagsById;
    for (const auto& tag: tags) {
        tagsById[tag.GetTagId()] = tag;
    }
    for (const auto& event: *events) {
        tagsById.erase(event.GetTagId());
    }

    if (tagsById.empty()) {
        return false;
    }

    tags = MakeVector(NContainer::Values(tagsById));
    return true;
}

TMaybe<TVector<TDBTag>> IRTPerformerDropper::DropPerformers(const TString& robotUserId, const IRTBackgroundProcess::TExecutionContext& context, NDrive::TEntitySession& session) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    if (!server) {
        throw yexception() << "failed to get server from a context";
    }

    auto performedTags = GetTagsToDropPerformer(context, session);
    if (!performedTags) {
        return {TVector<TDBTag>()};
    }

    auto notifier = GetNotifier(*server);
    if (HasActivityPeriod() && !FilterTagsByActivity(performedTags, server, session)) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Все теги типа " + JoinSeq(",", GetTagNames()) + " выполняются оперативно (в пределах " + TJsonProcessor::FormatDurationString(GetActivityPeriodUnsafe()) + ")"));
        }
        return {TVector<TDBTag>()};
    }

    const auto& objectTagsManager = GetObjectsTagsManager(server);
    if (!objectTagsManager.DropTagsPerformer(performedTags, robotUserId, session, true)) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Не получается сбросить исполнителя тегов " + JoinSeq(",", GetTagNames()) + ": " + session.GetStringReport()));
        }
        return Nothing();
    }

    return performedTags;
}

NDrive::INotifier::TPtr IRTPerformerDropper::GetNotifier(const NDrive::IServer& server) const {
    if (!HasNotifierName()) {
        return nullptr;
    }
    NDrive::INotifier::TPtr notifier = server.GetNotifier(GetNotifierNameUnsafe());
    return notifier;
}

const IEntityTagsManager& IRTPerformerDropper::GetObjectsTagsManager(const NDrive::IServer* server) const {
    const auto* driveApi = server->GetDriveAPI();
    if (!driveApi) {
        throw yexception() << "DriveAPI is undefined";
    }
    return driveApi->GetEntityTagsManager(GetObjectEntityType());
}

NDrive::TScheme IRTPerformerDropper::PatchScheme(NDrive::TScheme& scheme, const IServerBase& server) const {
    scheme.Add<TFSArray>("performer_tag_names", "Теги исполнителя").SetElement<TFSString>();
    scheme.Add<TFSArray>("tag_names", "Имена тегов").SetElement<TFSString>().SetRequired(true);
    scheme.Add<TFSDuration>("duration", "Максимальный период неактивности").SetMin(TDuration::Minutes(1)).SetDefault(TDuration::Hours(12)).SetRequired(false);
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    return scheme;
}

bool IRTPerformerDropper::Parse(const NJson::TJsonValue& jsonInfo) {
    return
        NJson::ParseField(jsonInfo, "performer_tag_names", PerformerTagNames) &&
        NJson::ParseField(jsonInfo, "tag_names", TagNames, true) &&
        NJson::ParseField(jsonInfo, "notifier", NotifierName) &&
        NJson::ParseField(jsonInfo, "duration", ActivityPeriod) &&
        (!!TagNames || !!PerformerTagNames);
}

NJson::TJsonValue IRTPerformerDropper::PatchJson(NJson::TJsonValue& result) const {
    NJson::InsertField(result, "tag_names", TagNames);
    NJson::InsertField(result, "performer_tag_names", PerformerTagNames);
    if (NotifierName) {
        NJson::InsertField(result, "notifier", *NotifierName);
    }
    if (ActivityPeriod) {
        NJson::InsertField(result, "duration", *ActivityPeriod);
    }
    return result;
}

TExpectedState TRTDropCarTagPerformer::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    if (!server.GetDriveAPI()) {
        return nullptr;
    }

    TMaybe<TVector<TDBTag>> cleanTags;
    {
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
        cleanTags = DropPerformers(GetRobotUserId(), context, session);
        if (!cleanTags || !session.Commit()) {
            return MakeUnexpected<TString>("cannot drop performer by robot: " + session.GetStringReport());
        }
    }

    NDrive::INotifier::TPtr notifier = GetNotifier(server);
    if (notifier) {
        TMap<TString, TString> users;
        TSet<TString> objects;
        for (const auto& tag : *cleanTags) {
            users.emplace(tag.GetTagId(), tag->GetPerformer());
            objects.emplace(tag.GetObjectId());
        }
        auto session = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
        auto gUsers = server.GetDriveAPI()->GetUsersData()->FetchInfo(NContainer::Values(users), session);
        if (!gUsers) {
            session.AddErrorMessage("RTDropPerformerWatcherConfig::DoExecuteFiltered", "cannot FetchInfo for UsersData");
            session.Check();
        }

        const TString header = "У тегов типа " + JoinSeq(",", GetTagNames()) + " сброшен исполнитель у объектов: \n";
        const auto& fetchedCarsData = context.GetFetchedCarsData();
        TVector<TString> droppedPerformers;
        for (auto&& i : *cleanTags) {
            TString performer;
            auto userInfo = gUsers.GetResultPtr(users[i.GetTagId()]);
            if (userInfo) {
                performer = userInfo->GetHRReport();
            }
            if (fetchedCarsData.contains(i.GetObjectId())) {
                droppedPerformers.emplace_back(fetchedCarsData.at(i.GetObjectId()).GetHRReport() + " " + performer);
            } else {
                droppedPerformers.emplace_back(i.GetObjectId() + " " + performer);
            }
        }

        NDrive::INotifier::MultiLinesNotify(notifier, header, droppedPerformers);
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TRTDropCarTagPerformer::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme = TDropper::PatchScheme(scheme, server);
    return scheme;
}

bool TRTDropCarTagPerformer::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return
        TBase::DoDeserializeFromJson(jsonInfo) &&
        TDropper::Parse(jsonInfo) && HasNotifierName() &&
        (HasActivityPeriod() && GetActivityPeriodUnsafe() > TDuration::Minutes(1));
}

NJson::TJsonValue TRTDropCarTagPerformer::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result = TDropper::PatchJson(result);
    return result;
}

TExpectedState TRTDropUserTagPerformer::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /* stateExt */, const TExecutionContext& context) const {
    const NDrive::IServer* server = &context.GetServerAs<NDrive::IServer>();
    if (!server || !server->GetDriveAPI()) {
        return nullptr;
    }

    auto session = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();
    if (!DropPerformers(GetRobotUserId(), context, session) || !session.Commit()) {
        return MakeUnexpected<TString>("cannot drop performer by robot: " + session.GetStringReport());
    }

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

NDrive::TScheme TRTDropUserTagPerformer::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme = TDropper::PatchScheme(scheme, server);
    return scheme;
}

NJson::TJsonValue TRTDropUserTagPerformer::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result = TDropper::PatchJson(result);
    return result;
}

bool TRTDropUserTagPerformer::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return
        TBase::DoDeserializeFromJson(jsonInfo) &&
        TDropper::Parse(jsonInfo);
}
