#include "deprecated_config.h"

#include <drive/backend/database/drive_api.h>

#include <rtline/util/json_processing.h>
#include <rtline/util/algorithm/container.h>

#include <util/string/join.h>

TRTDropPerformerWatcherConfig::TFactory::TRegistrator<TRTDropPerformerWatcherConfig> TRTDropPerformerWatcherConfig::Registrator(TRTDropPerformerWatcherConfig::GetTypeName());

TExpectedState TRTDropPerformerWatcherConfig::DoExecuteFiltered(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const NDrive::IServer& server, TTagsModificationContext& context) const {
    auto session = server.GetDriveAPI()->template BuildTx<NSQL::Writable>();
    TVector<TDBTag> tags;
    if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().RestoreTags(context.GetFilteredCarIds(), GetTagNames(), tags, session)) {
        return MakeUnexpected<TString>("сannot restore tags for " + JoinSeq(",", GetTagNames()) + ": " + session.GetStringReport());
    }
    TMap<TString, TDBTag> tagsById;

    {
        TSet<TString> tagIds;
        for (auto&& i : tags) {
            if (!!i->GetPerformer()) {
                tagIds.emplace(i.GetTagId());
                tagsById.emplace(i.GetTagId(), i);
            }
        }

        if (!tagsById.empty()) {
            TRecordsSet records;
            auto result = session->Exec("SELECT tag_id FROM car_tags_history WHERE tag_id IN ('" + JoinSeq("','", tagIds) + "') AND history_timestamp > " + ToString((StartInstant - GetDuration()).Seconds()), &records);
            if (!!result && result->IsSucceed()) {
                for (auto&& i : records) {
                    tagsById.erase(i.Get("tag_id"));
                }
            } else {
                return MakeUnexpected<TString>("cannot select tags for activity check: " + session.GetStringReport());
            }
        }
    }

    NDrive::INotifier::TPtr notifier = server.GetNotifier(GetNotifierName());
    if (tagsById.empty()) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Все теги типа " + JoinSeq(",", GetTagNames()) + " выполняются оперативно (в пределах " + TJsonProcessor::FormatDurationString(GetDuration()) + ")"));
        }
        return new IRTBackgroundProcessState();
    }

    TVector<TDBTag> cleanTags;
    TMap<TString, TString> users;
    TSet<TString> objects;
    for (auto&& i : tagsById) {
        cleanTags.emplace_back(i.second);
        if (i.second->GetPerformer()) {
            users.emplace(i.first, i.second->GetPerformer());
        }
        objects.emplace(i.second.GetObjectId());
    }

    session.SetComment("too long passive (" + GetDuration().ToString() + ")");
    if (!server.GetDriveAPI()->GetTagsManager().GetDeviceTags().DropTagsPerformer(cleanTags, GetRobotUserId(), session, true) || !session.Commit()) {
        if (notifier) {
            notifier->Notify(NDrive::INotifier::TMessage("Не получается сбросить исполнителя тегов " + JoinSeq(",", GetTagNames()) + ": " + session.GetStringReport()));
        }
        return MakeUnexpected<TString>("cannot drop performer by robot: " + session.GetStringReport());
    }

    auto tx = server.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
    auto gCars = server.GetDriveAPI()->GetCarsData()->FetchInfo(objects, tx);
    if (!gCars) {
        tx.AddErrorMessage("RTDropPerformerWatcherConfig::DoExecuteFiltered", "cannot FetchInfo for CarsData");
        tx.Check();
    }
    auto gUsers = server.GetDriveAPI()->GetUsersData()->FetchInfo(NContainer::Values(users), tx);
    if (!gUsers) {
        tx.AddErrorMessage("RTDropPerformerWatcherConfig::DoExecuteFiltered", "cannot FetchInfo for UsersData");
        tx.Check();
    }

    const TString header = "У тегов типа " + JoinSeq(",", GetTagNames()) + " сброшен исполнитель (слишком длительное время простоя: > " + GetDuration().ToString() + ") у объектов: \n";
    TVector<TString> droppedPerformers;
    for (auto&& i : cleanTags) {
        TString performer;
        auto userInfo = gUsers.GetResultPtr(users[i.GetTagId()]);
        if (userInfo) {
            performer = userInfo->GetHRReport();
        }
        auto* carInfo = gCars.GetResultPtr(i.GetObjectId());
        if (carInfo) {
            droppedPerformers.emplace_back(carInfo->GetHRReport() + " " + performer);
        } else {
            droppedPerformers.emplace_back(i.GetObjectId() + " " + performer);
        }
    }
    NDrive::INotifier::MultiLinesNotify(notifier, header, droppedPerformers);

    return new IRTBackgroundProcessState();
}

NDrive::TScheme TRTDropPerformerWatcherConfig::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("tag_names", "Теги");
    scheme.Add<TFSDuration>("duration", "Продолжительность, необходимая для сброса").SetMin(TDuration::Minutes(1)).SetDefault(TDuration::Hours(12));
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    return scheme;
}

bool TRTDropPerformerWatcherConfig::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!TBase::DoDeserializeFromJson(jsonInfo)) {
        return false;
    }
    JREAD_STRING(jsonInfo, "notifier", NotifierName);
    JREAD_DURATION(jsonInfo, "duration", Duration);

    if (!NotifierName || Duration < TDuration::Minutes(1)) {
        return false;
    }

    TString tagNames;
    JREAD_STRING(jsonInfo, "tag_names", tagNames);

    StringSplitter(tagNames).SplitBySet(", ").SkipEmpty().Collect(&TagNames);
    return true;
}

NJson::TJsonValue TRTDropPerformerWatcherConfig::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::WriteDurationString(result, "duration", Duration);
    JWRITE(result, "notifier", NotifierName);
    JWRITE(result, "tag_names", JoinSeq(",", TagNames));
    return result;
}
