#include "config.h"

#include <drive/backend/data/user_tags.h>

TUsingExternalPromoWatcher::TFactory::TRegistrator<TUsingExternalPromoWatcher> TUsingExternalPromoWatcher::Registrator(TUsingExternalPromoWatcher::GetTypeName());

TExpectedState TUsingExternalPromoWatcher::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /*state*/, const TExecutionContext& context) const {
    const NDrive::IServer& frServer = context.GetServerAs<NDrive::IServer>();
    TSet<TString> usedPromoCodes;
    try {
        auto reader = YtClient->CreateTableReader<NYT::TNode>(NYT::TRichYPath(GetYtPath()));
        for (; reader->IsValid(); reader->Next()) {
            const NYT::TNode& row = reader->GetRow();

            auto promoKey = row[GetPromoKey()].AsString();
            usedPromoCodes.insert(std::move(promoKey));
        }
    } catch (const yexception& e) {
        NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), e.what());
        return MakeUnexpected<TString>({});
    }

    TVector<TDBTag> dbTags;
    {
        auto session = frServer.GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();

        if (!frServer.GetDriveAPI()->GetTagsManager().GetUserTags().RestoreTags({}, { TagName }, dbTags, session)) {
            NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), session.GetStringReport());
            return MakeUnexpected<TString>({});
        }
    }

    auto description = dynamic_cast<const TExternalPromoTag::TDescription*>(frServer.GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(TagName).Get());
    if (!description) {
        NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), "incorrect description for tag " + TagName);
        return MakeUnexpected<TString>({});
    }

    TVector<TDBTag> forRemove;
    for (const auto& tag : dbTags) {
        auto tagPtr = tag.GetTagAs<TExternalPromoTag>();
        if (!tagPtr) {
            continue;
        }
        if (usedPromoCodes.contains(tagPtr->GetCode()) || description->GetDeadline() < ModelingNow()) {
            forRemove.emplace_back(std::move(tag));
        }
    }

    const IDriveTagsManager& manager = frServer.GetDriveAPI()->GetTagsManager();
    ui32 delta = 100;
    for (size_t i = 0; i < forRemove.size(); ) {
        const size_t next = Min<size_t>(i + delta, forRemove.size());
        bool fail = false;
        for (ui32 att = 0; att < 7; ++att) {
            {
                TVector<TDBTag> tagsForRemove(forRemove.begin() + i, forRemove.begin() + next);
                NDrive::TEntitySession session = manager.GetUserTags().BuildTx<NSQL::Writable>();
                fail = false;
                if (!frServer.GetDriveAPI()->GetTagsManager().GetUserTags().RemoveTags(tagsForRemove, GetRobotUserId(), &frServer, session, true) || !session.Commit()) {
                    NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), session.GetStringReport());
                    fail = true;
                } else {
                    break;
                }
            }
            Sleep(TDuration::Seconds(5));
        }
        if (fail) {
            return MakeUnexpected<TString>({});
        }
        i = next;
    }

    if (forRemove.size()) {
        NDrive::INotifier::Notify(frServer.GetNotifier(GetNotifierName()), "remove " + ToString(forRemove.size()) + " used promo codes out of " + ToString(dbTags.size()));
    }
    return new IRTBackgroundProcessState();
}

NDrive::TScheme TUsingExternalPromoWatcher::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSVariants>("notifier", "Способ нотификации").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSString>("yt_cluster", "Кластер").SetDefault("hahn");
    scheme.Add<TFSString>("yt_path", "Путь к таблице").SetRequired(true);
    scheme.Add<TFSString>("promo_key", "Ключ, по которому лежит промокод").SetDefault("promocode");
    const auto impl = server.GetAs<NDrive::IServer>();
    const auto api = impl ? impl->GetDriveAPI() : nullptr;
    ITagsMeta::TTagDescriptions descriptions;
    if (api) {
        descriptions = api->GetTagsManager().GetTagsMeta().GetTagsByType(TExternalPromoTag::TypeName);
    }
    TSet<TString> tagNames;
    for (auto&& i : descriptions) {
        const TExternalPromoTag::TDescription* descr = dynamic_cast<const TExternalPromoTag::TDescription*>(i.Get());
        if (descr) {
            tagNames.emplace(descr->GetName());
        }
    }
    scheme.Add<TFSVariants>("tag_name", "Имя тега").SetVariants(tagNames).SetRequired(true);
    return scheme;
}

bool TUsingExternalPromoWatcher::DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING(jsonInfo, "notifier", NotifierName);
    JREAD_STRING(jsonInfo, "yt_cluster", YtCluster);
    JREAD_STRING(jsonInfo, "yt_path", YtPath);
    JREAD_STRING(jsonInfo, "promo_key", PromoKey);
    JREAD_STRING(jsonInfo, "tag_name", TagName);
    YtClient.Reset(NYT::CreateClient(YtCluster));
    return TBase::DoDeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TUsingExternalPromoWatcher::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    TJsonProcessor::Write(result, "notifier", NotifierName);
    TJsonProcessor::Write(result, "yt_cluster", YtCluster);
    TJsonProcessor::Write(result, "yt_path", YtPath);
    TJsonProcessor::Write(result, "promo_key", PromoKey);
    TJsonProcessor::Write(result, "tag_name", TagName);
    return result;
}
