#include "process.h"

#include <drive/backend/data/device_tags.h>
#include <drive/backend/data/feedback.h>

#include <rtline/library/json/parse.h>

class TFeedbackAggregationProcessState : public TJsonRTBackgroundProcessState {
public:
    TFeedbackAggregationProcessState() = default;
    TFeedbackAggregationProcessState(ui64 lastEventId)
        : LastEventId(lastEventId)
    {
    }

    ui64 GetLastEventId() const {
        return LastEventId;
    }

    static TString GetTypeName() {
        return TFeedbackAggregationProcess::GetTypeName();
    }
    TString GetType() const override {
        return GetTypeName();
    }

protected:
    NJson::TJsonValue SerializeToJson() const override {
        NJson::TJsonValue result = NJson::JSON_MAP;
        result["last_event_id"] = LastEventId;
        return result;
    }
    bool DeserializeFromJson(const NJson::TJsonValue& value) override {
        return NJson::TryFromJson(value["last_event_id"], LastEventId);
    }

private:
    ui64 LastEventId = 0;

private:
    static TFactory::TRegistrator<TFeedbackAggregationProcessState> Registrator;
};

TExpectedState TFeedbackAggregationProcess::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    const TDriveAPI* api = server.GetDriveAPI();
    if (!api) {
        return MakeUnexpected<TString>("DriveAPI is missing");
    }

    auto feedbackState = dynamic_cast<TFeedbackAggregationProcessState*>(state.Get());
    auto lastEventId = feedbackState ? feedbackState->GetLastEventId() : StartEventId;

    const TDeviceTagsManager& deviceTagManager = api->GetTagsManager().GetDeviceTags();
    const TTraceTagsManager& traceTagsManager = api->GetTagsManager().GetTraceTags();

    TOptionalTagHistoryEventId lockedMaxEventId;
    TOptionalTagHistoryEvents optionalEvents;
    {
        ui64 limit = 1000;
        auto session = api->template BuildTx<NSQL::Writable>();
        lockedMaxEventId = traceTagsManager.GetLockedMaxEventId(session);
        if (!lockedMaxEventId) {
            return MakeUnexpected<TString>(TStringBuilder() << "cannot acquire LockedMaxEventId: " << session.GetStringReport());
        }
        IEntityTagsManager::TQueryOptions options(limit);
        options.SetTags({ TraceTagName });
        optionalEvents = traceTagsManager.GetEvents(lastEventId + 1, {}, session, options);
        if (!optionalEvents) {
            return MakeUnexpected<TString>(TStringBuilder() << "cannot acquire events since " << lastEventId + 1 << ": " << session.GetStringReport());
        }
    }

    auto proposalManager = deviceTagManager.GetPropositionsPtr();
    if (!proposalManager) {
        return MakeUnexpected<TString>("cannot acquire proposal manager");
    }

    auto tagDescription = api->GetTagsManager().GetTagsMeta().GetDescriptionByName(DeviceTagName);
    auto description = dynamic_cast<const TConfirmableTag::TDescription*>(tagDescription.Get());
    ui32 confirmationCount = description ? description->GetConfirmationsCount() : proposalManager->GetConfig().GetDefaultConfirmationsNeed();

    TMap<TString, TString> deviceProposals;
    {
        auto session = api->template BuildTx<NSQL::ReadOnly>();
        auto allProposals = proposalManager->Get(session);
        if (!allProposals) {
            return MakeUnexpected<TString>("cannot get propositions");
        }
        for (auto&& [proposalId, proposal]: *allProposals) {
            if (proposal->GetName() != DeviceTagName) {
                continue;
            }
            const TString& deviceId = proposal.GetObjectId();
            deviceProposals[deviceId] = proposalId;
        }
    }

    const auto& events = *optionalEvents;
    for (auto&& ev : events) {
        if (ev.GetHistoryEventId() > *lockedMaxEventId) {
            break;
        }
        auto prevLastEventId = lastEventId;
        lastEventId = std::max(lastEventId, ev.GetHistoryEventId());
        if (ev.GetHistoryAction() != EObjectHistoryAction::Add) {
            continue;
        }
        if (ev->GetName() != TraceTagName) {
            continue;
        }
        auto feedback = ev.GetTagAs<TFeedbackTraceTag>();
        if (!feedback) {
            WARNING_LOG << GetRobotId() << ": cannot cast tag " << ev.GetTagId() << " as FeedbackTraceTag" << Endl;
            continue;
        }
        const TString& deviceId = feedback->GetDeviceId();
        const TString& userId = feedback->GetUserId();
        auto session = api->template BuildTx<NSQL::Writable>();
        if (!feedback->IsFeedbackEnabled(&server, session)) {
            INFO_LOG << GetRobotId() << ": feedback disabled for " << feedback->GetUserId() << Endl;
            continue;
        }

        auto p = deviceProposals.find(deviceId);
        if (p != deviceProposals.end()) {
            const TString& proposalId = p->second;
            if (!deviceTagManager.ConfirmPropositions({ proposalId }, userId, &server, session)) {
                ERROR_LOG << GetRobotId() << ": cannot confirm proposal " << proposalId << ' ' << session.GetStringReport() << Endl;
                return MakeAtomicShared<TFeedbackAggregationProcessState>(prevLastEventId);
            }
        } else {
            TDBTag tag;
            tag.SetData(api->GetTagsManager().GetTagsMeta().CreateTag(DeviceTagName));
            tag.SetObjectId(deviceId);
            tag->SetObjectSnapshot(ev->GetObjectSnapshot());
            if (confirmationCount) {
                TPropositionId proposalId = deviceTagManager.ProposeTag(tag, userId, session);
                if (!proposalId) {
                    ERROR_LOG << GetRobotId() << ": cannot create proposal for " << deviceId << ' ' << session.GetStringReport() << Endl;
                    return MakeAtomicShared<TFeedbackAggregationProcessState>(prevLastEventId);
                }
                deviceProposals[deviceId] = proposalId;
            } else {
                if (!deviceTagManager.AddTag(tag.GetData(), userId, deviceId, &server, session, EUniquePolicy::Rewrite)) {
                    ERROR_LOG << GetRobotId() << ": cannot add tag for " << deviceId << ' ' << session.GetStringReport() << Endl;
                    return MakeAtomicShared<TFeedbackAggregationProcessState>(prevLastEventId);
                }
            }
        }
        if (!session.Commit()) {
            ERROR_LOG << GetRobotId() << ": cannot commit transaction " << session.GetStringReport() << Endl;
            return MakeAtomicShared<TFeedbackAggregationProcessState>(prevLastEventId);
        }
    }

    return MakeAtomicShared<TFeedbackAggregationProcessState>(lastEventId);
}

NDrive::TScheme TFeedbackAggregationProcess::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSString>("device_tag_name", "Target device tag name").SetRequired(true);
    scheme.Add<TFSString>("trace_tag_name", "Source trace tag name").SetRequired(true);
    scheme.Add<TFSNumeric>("start_event_id", "Стартовая позиция").SetDefault(0);
    return scheme;
}

bool TFeedbackAggregationProcess::DoDeserializeFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DoDeserializeFromJson(value)) {
        return false;
    }
    return
        NJson::ParseField(value["start_event_id"], StartEventId, false) &&
        NJson::ParseField(value["device_tag_name"], DeviceTagName, true) &&
        NJson::ParseField(value["trace_tag_name"], TraceTagName, true);
}

NJson::TJsonValue TFeedbackAggregationProcess::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();
    result["device_tag_name"] = DeviceTagName;
    result["trace_tag_name"] = TraceTagName;
    result["start_event_id"] = StartEventId;
    return result;
}

TFeedbackAggregationProcess::TFactory::TRegistrator<TFeedbackAggregationProcess> TFeedbackAggregationProcess::Registrator(TFeedbackAggregationProcess::GetTypeName());
TFeedbackAggregationProcessState::TFactory::TRegistrator<TFeedbackAggregationProcessState> TFeedbackAggregationProcessState::Registrator(TFeedbackAggregationProcessState::GetTypeName());
