#include "attachment_processor_base.h"
#include "util/generic/algorithm.h"

#include <drive/backend/data/fine.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/fines/manager.h>
#include <drive/backend/fines/constructors/fine_attachment_constructor.h>

#include <rtline/util/algorithm/container.h>

IFineAttachmentCollectorContext::IFineAttachmentCollectorContext(
    const NDrive::IServer& server,
    const TAttachmentConstructor& fineAttachmentConstructor,
    const TString& fineId,
    const TString& rulingNumber,
    TEventsHandler::TEventPtr relatedEventPtr
)
    : Server(server)
    , FineAttachmentConstructor(fineAttachmentConstructor)
    , FineId(fineId)
    , RulingNumber(rulingNumber)
    , RelatedEventPtr(relatedEventPtr)
{
}

const NDrive::IServer& IFineAttachmentCollectorContext::GetServer() const {
    return Server;
}

const NDrive::NFine::TFineAttachmentConstructor& IFineAttachmentCollectorContext::GetFineAttachmentConstructor() const {
    return FineAttachmentConstructor;
}

TRTFineAttachmentCollectorBase::TRTFineAttachmentCollectorBase(TEventsHandler::TPtr tagEventsHandlerPtr)
    : TagEventsHandlerPtr(tagEventsHandlerPtr)
{
}

bool TRTFineAttachmentCollectorBase::DoStart(const TRTBackgroundProcessContainer& container) {
    if (!TBase::DoStart(container)) {
        return false;
    }
    if (!NotifyHandlers) {
        return false;
    }
    NotifyHandlers->SetSignalName(GetRTProcessName());
    return true;
}

void TRTFineAttachmentCollectorBase::InitNotifyHandlers() {
    if (!NotifyHandlers) {
        NotifyHandlers = MakeAtomicShared<TNotifyHandlers>(GetType());
    }
}

using TKnownAttachmentMap = NDrive::NFine::TFineAttachmentConstructor::TKnownAttachmentMap;

TExpectedState TRTFineAttachmentCollectorBase::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    CHECK_WITH_LOG(!!server.GetDriveAPI());

    if (!server.GetDriveAPI()->HasFinesManager() || !NotifyHandlers) {
        return nullptr;
    }

    auto nativeState = std::dynamic_pointer_cast<IState>(state);
    const ui64 lastProcessedEventId = (!!nativeState) ? nativeState->GetLastEventId() : std::numeric_limits<ui64>::max();

    const auto& finesManager = server.GetDriveAPI()->GetFinesManager();

    TFineIdToEventPtrMapping fineIdToEventPtrMapping;
    if (!PrepareEventsToProcess(server, lastProcessedEventId, fineIdToEventPtrMapping)) {
        return MakeUnexpected<TString>("Error fetching tag events");
    }

    auto filters = GetFilters(server, fineIdToEventPtrMapping);

    TMap<TString, TString> finesMap;
    NDrive::NFine::TFineAttachmentConstructor fineAttachmentConstructor(GetFineAttachmentConstructorConfig(), server);
    {
        auto tx = finesManager.BuildSession(/* readOnly = */ true);
        auto fines = finesManager.GetFines(filters, tx);
        R_ENSURE(fines, {}, "Fail to fetch fines", tx);
        Transform(fines->begin(), fines->end(), std::inserter(finesMap, finesMap.begin()), [](const auto& fine) {
            return std::pair<TString, TString>(fine.GetId(), fine.GetRulingNumber());
        });
        R_ENSURE(fineAttachmentConstructor.Init(finesManager, MakeSet(NContainer::Keys(finesMap)), tx), {}, "Fail to fetch attachments", tx);
    }

    ui32 totalProcessed = 0;

    TMessagesCollector errors;
    for (auto&& [fineId, fineRulingNumber] : finesMap) {
        if (!!IterationsLimit && totalProcessed >= IterationsLimit) {  // unlimited if zero
            WARNING_LOG << "Processed items limit exceeded: " << totalProcessed << " of " << IterationsLimit << Endl;
            break;
        }

        auto relatedTagPtr = fineIdToEventPtrMapping.Value(fineId, nullptr);
        IBaseContext::TPtr context = ConstructContext(server, fineAttachmentConstructor, fineId, fineRulingNumber, relatedTagPtr);

        if (!FilterFine(context, errors)) {
            if (errors.HasMessages()) {
                return MakeUnexpected<TString>("Got critical error from fine filter:" + errors.GetStringReport());
            }
            continue;
        }

        EProcessingResult result = ProcessFine(context);

        if (result == EProcessingResult::OK) {
            ++totalProcessed;
        } else if (result == EProcessingResult::SKIP) {
            continue;
        } else if (result == EProcessingResult::CRITICAL_ERROR) {
            // NB. Tags to process are fetched at once, therefore they must be processed again in case of error. Removed tags won't be processed.
            return MakeUnexpected<TString>("Got critical error from fine process id: " + fineId);
        } else {
            return MakeUnexpected<TString>("Got unknown status from fine process id: " + fineId);
        }
    }

    return (!!TagEventsHandlerPtr) ? BuildState(TagEventsHandlerPtr->GetLastProcessedEventId()) : BuildState(std::numeric_limits<ui64>::max());
}

bool TRTFineAttachmentCollectorBase::PrepareEventsToProcess(const NDrive::IServer& server, const ui64 lastProcessedEventId, TFineIdToEventPtrMapping& fineIdToEventPtrMapping) const {
    if (!!TagEventsHandlerPtr && GetHandleActionTags()) {
        const bool fetchResult = TagEventsHandlerPtr->Fetch(server, lastProcessedEventId, DataActuality);
        if (!fetchResult) {
            return false;
        }

        auto tx = TagEventsHandlerPtr->GetEntityTagsManager(server).BuildTx<NSQL::ReadOnly>();
        while (auto eventPtr = TagEventsHandlerPtr->Next(server)) {
            auto fineActionTagPtr = eventPtr->GetTagAs<TFineActionTag>();
            if (!!fineActionTagPtr && TagEventsHandlerPtr->DoesTagExist(eventPtr, server, tx)) {
                fineIdToEventPtrMapping.emplace(fineActionTagPtr->GetFineId(), eventPtr);
            }
        }
    }
    return true;
}

NDrive::NFine::TFineFilterGroup TRTFineAttachmentCollectorBase::GetFilters(const NDrive::IServer& /* server */, const TFineIdToEventPtrMapping& fineIdToEventPtrMapping) const {
    NDrive::NFine::TFineFilterGroup filters;

    if (GetHandleActionTags()) {
        if (!!fineIdToEventPtrMapping) {
            filters.Append(MakeAtomicShared<NDrive::NFine::TMultipleFineIdsFilter>(NContainer::Keys(fineIdToEventPtrMapping)));
        } else {
            filters.Append(MakeAtomicShared<NDrive::NFine::TRejectFilter>());  // skip all fines
        }
    }

    if (GetNeedsChargeOnlyFlag()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineChargeableFilter>());
    }

    if (!!SourceTypeFilter) {
        if (SourceTypeFilter.size() == 1) {
            filters.Append(MakeAtomicShared<NDrive::NFine::TFineSourceTypeFilter>(*SourceTypeFilter.begin()));
        } else {
            filters.Append(MakeAtomicShared<NDrive::NFine::TFineMultipleSourcesTypeFilter>(SourceTypeFilter));
        }
    }

    TVector<const NDrive::NFine::IDateTimeFilterConfig*> dateFilterConfigs = { &GetRulingDateFilterConfig(), &GetViolationTimeFilterConfig(), &GetFineReceiveTimeFilterConfig() };
    for (auto dateFilterConfig : dateFilterConfigs) {
        if (dateFilterConfig->HasRestrictions()) {
            filters.Append(dateFilterConfig->GetFilter());
        }
    }

    return filters;
}

bool TRTFineAttachmentCollectorBase::FilterFine(const IBaseContext::TPtr /* context */, TMessagesCollector& /* errors */) const {
    return true;
}

TRTFineAttachmentCollectorBase::EProcessingResult TRTFineAttachmentCollectorBase::ProcessFine(IBaseContext::TPtr context) const {
    TMessagesCollector errors;

    if (!CollectExternalAttachments(context, errors)) {
        ERROR_LOG << "Error collecting fine " << context->GetFineId() << " attachments: " << errors.GetStringReport() << Endl;
        NotifyHandlers->Handle(ECollectingStatus::CollectingFail, context->GetServer());
        return EProcessingResult::SKIP;
    }

    NotifyHandlers->Handle(ECollectingStatus::CollectingSuccess, context->GetServer());

    if (!context->GetCollectedTotal()) {
        return EProcessingResult::SKIP;
    }

    NotifyHandlers->Handle(ECollectingStatus::CollectedTotal, context->GetServer(), context->GetCollectedTotal());

    TVector<NDrive::NFine::TAutocodeFineAttachmentEntry> attachmentEntries;
    if (!ConstructNativeFineAttachments(context, errors)) {
        NotifyHandlers->Handle(ECollectingStatus::CollectedConstructingError, context->GetServer());
        ERROR_LOG << "Error constructing and uploading fine " << context->GetFineId() << " attachement: " << errors.GetStringReport() << Endl;
        return EProcessingResult::CRITICAL_ERROR;
    }

    if (!UpsertFineAttachments(context, errors)) {
        ERROR_LOG << errors.GetStringReport() << Endl;
        return EProcessingResult::CRITICAL_ERROR;
    }

    return EProcessingResult::OK;
}

bool TRTFineAttachmentCollectorBase::UpsertFineAttachments(IBaseContext::TPtr context, TMessagesCollector& errors) const {
    const auto& finesManager = context->GetServer().GetDriveAPI()->GetFinesManager();

    auto session = finesManager.BuildSession(/* readOnly = */ false);

    if (!HandleFineRelatedTags(context, session)) {
        errors.AddMessage(__LOCATION__, TStringBuilder() << "Error handle tags related to fine " << context->GetFineId() << " attachment: " << session.GetStringReport());
        NotifyHandlers->Handle(ECollectingStatus::TagUpsertError, context->GetServer());
        return false;
    }

    for (auto&& attachmentEntry : context->MutableAttachmentEntries()) {
        if (!finesManager.UpsertFineAttachment(attachmentEntry, session)) {
            errors.AddMessage(__LOCATION__, TStringBuilder() << "Error upserting fine " << context->GetFineId() << " attachment: " << session.GetStringReport());
            NotifyHandlers->Handle(ECollectingStatus::UpsertError, context->GetServer());
            return false;
        }
    }

    if (!session.Commit()) {
        errors.AddMessage(__LOCATION__, TStringBuilder() << "Error processing fine " << context->GetFineId() << " attachment: " << session.GetStringReport());
        NotifyHandlers->Handle(ECollectingStatus::CommitError, context->GetServer());
        return false;
    }

    return true;
}

bool TRTFineAttachmentCollectorBase::HandleFineRelatedTags(IBaseContext::TPtr context, NDrive::TEntitySession& session) const {
    if (GetHandleActionTags() && !!TagEventsHandlerPtr && !!context->GetRelatedEventPtr()) {
        return TagEventsHandlerPtr->RemoveTagIfExists(context->GetRelatedEventPtr(), GetRobotUserId(), context->GetServer(), session);
    }
    return true;
}

NDrive::TScheme TRTFineAttachmentCollectorBase::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSNumeric>("iterations_limit", "Ограничение на количество итераций; без ограничений если 0").SetMin(0).SetDefault(0);

    scheme.Add<TFSBoolean>("needs_charge_only", "Только для списания").SetDefault(false);
    scheme.Add<TFSBoolean>("handle_action_tags", "Обрабатывать теги действий").SetDefault(false);

    scheme.Add<TFSVariants>("source_type_filter", "Источник штрафов").SetVariants(GetEnumAllValues<NDrive::NFine::ESourceType>()).SetMultiSelect(true);

    scheme.Add<TFSStructure>("ruling_date_filter", "Диапазон дат постановлений").SetStructure(NDrive::NFine::TRulingDateFilterConfig::GetScheme());
    scheme.Add<TFSStructure>("violation_time_filter", "Диапазон времени нарушения").SetStructure(NDrive::NFine::TViolationTimeFilterConfig::GetScheme());
    scheme.Add<TFSStructure>("fine_receive_time_filter", "Диапазон времени получения информации").SetStructure(NDrive::NFine::TFineReceiveTimeFilterConfig::GetScheme());

    scheme.Add<TFSStructure>("fine_attachment_constructor_config", "Настройки конструктора").SetStructure(NDrive::NFine::TFineAttachmentConstructorConfig::GetScheme(server));

    scheme.Add<TFSStructure>("notify_handlers", "Настройки нотификаторов").SetStructure(TNotifyHandlers::GetScheme(server));

    return scheme;
}

bool TRTFineAttachmentCollectorBase::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    if (!TBase::DoDeserializeFromJson(data)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "iterations_limit", IterationsLimit)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "needs_charge_only", NeedsChargeOnlyFlag)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "handle_action_tags", HandleActionTags)) {
        return false;
    }

    if (!TJsonProcessor::ReadContainer(data, "source_type_filter", SourceTypeFilter)) {
        return false;
    }

    if (!RulingDateFilterConfig.DeserializeFromJson(data["ruling_date_filter"])) {
        return false;
    }
    if (!ViolationTimeFilterConfig.DeserializeFromJson(data["violation_time_filter"])) {
        return false;
    }
    if (!FineReceiveTimeFilterConfig.DeserializeFromJson(data["fine_receive_time_filter"])) {
        return false;
    }

    if (!FineAttachmentConstructorConfig.DeserializeFromJson(data["fine_attachment_constructor_config"])) {
        return false;
    }

    if (!!NotifyHandlers && !NotifyHandlers->DeserializeFromJson(data["notify_handlers"])) {
        return false;
    }

    return true;
}

NJson::TJsonValue TRTFineAttachmentCollectorBase::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "iterations_limit", IterationsLimit);

    TJsonProcessor::Write(result, "needs_charge_only", NeedsChargeOnlyFlag);
    TJsonProcessor::Write(result, "handle_action_tags", HandleActionTags);

    TJsonProcessor::WriteContainerArrayStrings(result, "source_type_filter", SourceTypeFilter);

    TJsonProcessor::WriteSerializable(result, "ruling_date_filter", RulingDateFilterConfig);
    TJsonProcessor::WriteSerializable(result, "violation_time_filter", ViolationTimeFilterConfig);
    TJsonProcessor::WriteSerializable(result, "fine_receive_time_filter", FineReceiveTimeFilterConfig);

    TJsonProcessor::WriteSerializable(result, "fine_attachment_constructor_config", FineAttachmentConstructorConfig);

    if (!!NotifyHandlers) {
        TJsonProcessor::WriteSerializable(result, "notify_handlers", *NotifyHandlers);
    }

    return result;
}
