#pragma once

#include "events.h"
#include "statuses.h"

#include <drive/backend/rt_background/common/config.h>
#include <drive/backend/rt_background/common/state.h>

#include <drive/backend/fines/config.h>
#include <drive/backend/fines/constants.h>
#include <drive/backend/fines/filters.h>
#include <drive/backend/fines/context_fetchers/fine_context.h>
#include <drive/backend/notifications/collection.h>
#include <drive/backend/proto/background.pb.h>

#include <rtline/util/types/accessor.h>

namespace NDrive::NFine {
    class TFineAttachmentConstructor;
}

class IFineAttachmentCollectorState : public TRTHistoryWatcherState {
};

class IFineAttachmentCollectorContext {
public:
    using TPtr = TAtomicSharedPtr<IFineAttachmentCollectorContext>;
    using TAttachmentConstructor = NDrive::NFine::TFineAttachmentConstructor;

public:
    IFineAttachmentCollectorContext(
        const NDrive::IServer& server,
        const TAttachmentConstructor& fineAttachmentConstructor,
        const TString& fineId,
        const TString& rulingNumber,
        TEventsHandler::TEventPtr relatedEventPtr
    );
    virtual ~IFineAttachmentCollectorContext() = default;

    const NDrive::IServer& GetServer() const;
    const NDrive::NFine::TFineAttachmentConstructor& GetFineAttachmentConstructor() const;

    virtual size_t GetCollectedTotal() const = 0;

private:
    const NDrive::IServer& Server;
    const TAttachmentConstructor& FineAttachmentConstructor;

    R_READONLY(TString, FineId);
    R_READONLY(TString, RulingNumber);
    R_READONLY(TEventsHandler::TEventPtr, RelatedEventPtr);

    R_FIELD(TVector<NDrive::NFine::TAutocodeFineAttachmentEntry>, AttachmentEntries);
};

class TRTFineAttachmentCollectorBase: public IRTRegularBackgroundProcess {
protected:
    using TBase = IRTRegularBackgroundProcess;

    using IState = IFineAttachmentCollectorState;
    using IBaseContext = IFineAttachmentCollectorContext;

    using ECollectingStatus = NDrive::NFine::EAttachmentCollectingStatus;
    using EProcessingResult = NDrive::NFine::EFineAttachmentProcessingResult;

    using TNotifyHandlers = TContextNotifyHandlerCollection<ECollectingStatus, NDrive::NFine::IFineContextFetcher>;

    using TFineIdToEventPtrMapping = TMap<TString, TEventsHandler::TEventPtr>;

public:
    explicit TRTFineAttachmentCollectorBase(TEventsHandler::TPtr tagEventsHandlerPtr);

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& data) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

protected:
    virtual bool DoStart(const TRTBackgroundProcessContainer& container) override;
    virtual void InitNotifyHandlers() final;

    bool PrepareEventsToProcess(const NDrive::IServer& server, const ui64 lastProcessedEventId, TFineIdToEventPtrMapping& fineIdToEventPtrMapping) const;

    virtual NDrive::NFine::TFineFilterGroup GetFilters(const NDrive::IServer& server, const TFineIdToEventPtrMapping& fineIdToEventPtrMapping) const;

    virtual TAtomicSharedPtr<IState> BuildState(const ui64 lastEventId) const = 0;
    virtual IBaseContext::TPtr ConstructContext(
        const NDrive::IServer& server,
        const NDrive::NFine::TFineAttachmentConstructor& fineAttachmentConstructor,
        const TString& fineId,
        const TString& rulingNumber,
        TEventsHandler::TEventPtr relatedEventPtr
    ) const = 0;

    virtual bool FilterFine(const IBaseContext::TPtr context, TMessagesCollector& errors) const;
    EProcessingResult ProcessFine(IBaseContext::TPtr context) const;

    virtual bool CollectExternalAttachments(IBaseContext::TPtr context, TMessagesCollector& errors) const = 0;
    virtual bool ConstructNativeFineAttachments(IBaseContext::TPtr context, TMessagesCollector& errors) const = 0;

    bool UpsertFineAttachments(IBaseContext::TPtr context, TMessagesCollector& errors) const;
    bool HandleFineRelatedTags(IBaseContext::TPtr context, NDrive::TEntitySession& session) const;

private:
    R_FIELD(ui32, IterationsLimit, 0);

    R_FIELD(bool, NeedsChargeOnlyFlag, false);
    R_FIELD(bool, HandleActionTags, false);

    R_FIELD(TSet<NDrive::NFine::ESourceType>, SourceTypeFilter);

    R_FIELD(NDrive::NFine::TRulingDateFilterConfig, RulingDateFilterConfig);
    R_FIELD(NDrive::NFine::TViolationTimeFilterConfig, ViolationTimeFilterConfig);
    R_FIELD(NDrive::NFine::TFineReceiveTimeFilterConfig, FineReceiveTimeFilterConfig);

    R_FIELD(NDrive::NFine::TFineAttachmentConstructorConfig, FineAttachmentConstructorConfig);

    R_FIELD(TNotifyHandlers::TPtr, NotifyHandlers, nullptr);

    R_READONLY(TEventsHandler::TPtr, TagEventsHandlerPtr, nullptr, mutable);
};
