#include "autocode_photo_processor.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/fines/manager.h>
#include <drive/backend/fines/constructors/fine_attachment_constructor.h>

#include <drive/library/cpp/autocode/client.h>

const TString TRTAutocodeFinePhotoCollector::TypeName = "fines_autocode_photo_processor";

TString TRTAutocodeFinePhotoCollectorState::GetTypeName() {
    return TRTAutocodeFinePhotoCollector::TypeName;
}

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTAutocodeFinePhotoCollector> TRTAutocodeFinePhotoCollector::Registrator(TRTAutocodeFinePhotoCollector::TypeName);
TRTHistoryWatcherState::TFactory::TRegistrator<TRTAutocodeFinePhotoCollectorState> TRTAutocodeFinePhotoCollectorState::Registrator(TRTAutocodeFinePhotoCollectorState::GetTypeName());

TAutocodeFinePhotoCollectorContext::TAutocodeFinePhotoCollectorContext(
    const NDrive::IServer& server,
    const TAttachmentConstructor& fineAttachmentConstructor,
    const TString& fineId,
    const TString& rulingNumber,
    TEventsHandler::TEventPtr relatedEventPtr
)
    : TBase(server, fineAttachmentConstructor, fineId, rulingNumber, relatedEventPtr)
{
}

size_t TAutocodeFinePhotoCollectorContext::GetCollectedTotal() const {
    return Photos.size();
}

TRTAutocodeFinePhotoCollector::TRTAutocodeFinePhotoCollector()
    : TBase(nullptr)
{
    InitNotifyHandlers();
}

TExpectedState TRTAutocodeFinePhotoCollector::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    CHECK_WITH_LOG(!!server.GetDriveAPI());

    if (!server.GetDriveAPI()->HasFinesManager() || !server.GetDriveAPI()->HasAutocodeClient()) {
        ERROR_LOG << "Fines manager or Autocode client is undefined" << Endl;
        return nullptr;
    }

    return TBase::DoExecute(state, context);
}

NDrive::NFine::TFineFilterGroup TRTAutocodeFinePhotoCollector::GetFilters(const NDrive::IServer& server, const TFineIdToEventPtrMapping& fineIdToEventPtrMapping) const {
    NDrive::NFine::TFineFilterGroup filters = TBase::GetFilters(server, fineIdToEventPtrMapping);

    if (GetExplicitPhotoOnlyFlag()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineExplicitPhotoFilter>());
    }

    if (GetCameraFixationOnlyFlag()) {
        filters.Append(MakeAtomicShared<NDrive::NFine::TFineCameraFixationFilter>());
    }

    return filters;
}

TAtomicSharedPtr<TRTAutocodeFinePhotoCollector::IState> TRTAutocodeFinePhotoCollector::BuildState(const ui64 lastEventId) const {
    auto state = MakeAtomicShared<TState>();
    state->SetLastEventId(lastEventId);
    return state;
}

TRTAutocodeFinePhotoCollector::IBaseContext::TPtr TRTAutocodeFinePhotoCollector::ConstructContext(
    const NDrive::IServer& server,
    const NDrive::NFine::TFineAttachmentConstructor& fineAttachmentConstructor,
    const TString& fineId,
    const TString& rulingNumber,
    TEventsHandler::TEventPtr relatedEventPtr
) const {
    return MakeAtomicShared<TContext>(server, fineAttachmentConstructor, fineId, rulingNumber, relatedEventPtr);
}

bool TRTAutocodeFinePhotoCollector::FilterFine(const IBaseContext::TPtr context, TMessagesCollector& errors) const {
    if (!TBase::FilterFine(context, errors)) {
        return false;
    }
    auto existingFinePhotosPtr = context->GetFineAttachmentConstructor().GetKnownPhotos().FindPtr(context->GetFineId());
    return !existingFinePhotosPtr || existingFinePhotosPtr->empty();
}

bool TRTAutocodeFinePhotoCollector::CollectExternalAttachments(IBaseContext::TPtr context, TMessagesCollector& errors) const {
    auto derivedContextPtr = VerifyDynamicCast<TContext*>(context.Get());

    const auto& autocodeClient = context->GetServer().GetDriveAPI()->GetAutocodeClient();
    if (!autocodeClient.GetViolationPhotos(context->GetRulingNumber(), derivedContextPtr->MutablePhotos(), errors)) {
        // some photos may be absent as well as API is not quite stable, therefore the process continues
        // consider to save information about trial for futher usage
        return false;
    }

    return true;
}

bool TRTAutocodeFinePhotoCollector::ConstructNativeFineAttachments(IBaseContext::TPtr context, TMessagesCollector& errors) const {
    auto derivedContextPtr = VerifyDynamicCast<TContext*>(context.Get());

    for (auto&& photo : derivedContextPtr->MutablePhotos()) {
        NDrive::NFine::TAutocodeFineAttachmentEntry attachmentEntry;

        if (!context->GetFineAttachmentConstructor().ConstructAutocodeFineAttachmentEntry(context->GetFineId(), photo, attachmentEntry, errors)) {
            return false;
        }

        context->MutableAttachmentEntries().push_back(std::move(attachmentEntry));
    }

    return true;
}

NDrive::TScheme TRTAutocodeFinePhotoCollector::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSBoolean>("explicit_photo_only", "Только с явным признаком наличия фото").SetDefault(true);
    scheme.Add<TFSBoolean>("camera_fixation_only", "Только с фотофиксацией").SetDefault(false);

    return scheme;
}

bool TRTAutocodeFinePhotoCollector::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    if (!TBase::DoDeserializeFromJson(data)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "explicit_photo_only", ExplicitPhotoOnlyFlag)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "camera_fixation_only", CameraFixationOnlyFlag)) {
        return false;
    }

    return true;
}

NJson::TJsonValue TRTAutocodeFinePhotoCollector::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "explicit_photo_only", ExplicitPhotoOnlyFlag);
    TJsonProcessor::Write(result, "camera_fixation_only", CameraFixationOnlyFlag);

    return result;
}
