#include "autocode_processor.h"

#include <drive/backend/database/transaction/assert.h>

#include <drive/library/cpp/autocode/client.h>

const ui32 TRTAutocodeFinesCollector::DefaultMaxRequestFinesCount = 1000000;

const TString TRTAutocodeFinesCollector::TypeName = "fines_autocode_processor";

IRTRegularBackgroundProcess::TFactory::TRegistrator<TRTAutocodeFinesCollector> TRTAutocodeFinesCollector::Registrator(TRTAutocodeFinesCollector::TypeName);

TAutocodeFinesProcessingContext::TAutocodeFinesProcessingContext(const NDrive::IServer& server
    , const NDrive::NFine::TFineConstructor& fineConstructor
    , const TPrefechedFinesContainer& prefechedFines
    , TVector<TExternalFine>&& externalFines
)
    : TBase(server, fineConstructor, prefechedFines)
    , ExternalFines(std::move(externalFines))
{
}

bool TAutocodeFinesProcessingContext::OnBeforeCommit(const NDrive::NFine::TAutocodeFineEntry& fine) {
    FineIdsToConfirm.push_back(::ToString(fine.GetAutocodeId()));
    return true;
}

TRTAutocodeFinesCollector::TRTAutocodeFinesCollector()
    : TBase()
{
    InitNotifyHandlers();
}

TExpectedState TRTAutocodeFinesCollector::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> /* state */, const TExecutionContext& context) const {
    const auto& server = context.GetServerAs<NDrive::IServer>();
    CHECK_WITH_LOG(!!server.GetDriveAPI());

    if (!server.GetDriveAPI()->HasFinesManager() || !server.GetDriveAPI()->HasAutocodeClient() || !GetNotifyHandlers()) {
        return MakeUnexpected<TString>({});
    }

    NDrive::NFine::TFineConstructor fineConstructor(GetFineConstructorConfig(), server);

    ui64 offset = 0;
    ui64 limit = (IsRequestSizeUnlimited()) ? DefaultMaxRequestFinesCount : MaxRequestFinesCount;

    TPrefechedFinesContainer prefechedFines;
    {
        const auto& finesManager = Yensured(server.GetDriveAPI())->GetFinesManager();
        auto tx = finesManager.BuildTx<NSQL::ReadOnly>();
        R_ENSURE(prefechedFines.Init(finesManager, GetFilters(), tx), {}, "Fail to fetch fines for AutocodeFinesCollector", tx);
    }

    bool hasMore = false;
    TVector<TAutocodeFine> finesChunk;
    TSet<TString> collectedRulingNumbers;
    do {
        hasMore = false;
        finesChunk.clear();

        if (!CollectFines(server, offset, limit, finesChunk)) {
            return MakeUnexpected<TString>({});
        }

        if (hasMore = (!finesChunk.empty())) {
            auto filteredFinesChunk = FilterCollectedFines(collectedRulingNumbers, std::move(finesChunk));

            IFineProcessingContextBase::TPtr contextPtr = ConstructProcessingContext(server, fineConstructor, prefechedFines, std::move(filteredFinesChunk));
            TMessagesCollector errors;

            if (!ProcessFines(contextPtr, errors)) {
                ERROR_LOG << errors.GetStringReport() << Endl;
                return MakeUnexpected<TString>({});
            }
        }

        offset += limit;
    } while (!IsRequestSizeUnlimited() && hasMore);

    return MakeAtomicShared<IRTBackgroundProcessState>();
}

TVector<NDrive::NAutocode::TAutocodeFine> TRTAutocodeFinesCollector::FilterCollectedFines(TSet<TString>& collectedRulingNumbers, TVector<TAutocodeFine>&& fines) const {
    TVector<TAutocodeFine> filteredFines;

    for (auto&& fine : fines) {
        TString rulingNumber = fine.GetRulingNumber();

        if (collectedRulingNumbers.contains(rulingNumber)) {
            continue;
        }
        collectedRulingNumbers.insert(rulingNumber);

        if (StsToCheck.contains(ToString(fine.GetViolationDocumentNumber())) ^ IsExcludeSts()) {
            filteredFines.push_back(std::move(fine));
        }
    }

    return filteredFines;
}

bool TRTAutocodeFinesCollector::IsRequestSizeUnlimited() const {
    return (MaxRequestFinesCount == 0 || MaxRequestFinesCount == DefaultMaxRequestFinesCount);
}

bool TRTAutocodeFinesCollector::CollectFines(const NDrive::IServer& server, ui64 offset, ui64 limit, TVector<TAutocodeFine>& fines) const {
    const auto& autocodeClient = server.GetDriveAPI()->GetAutocodeClient();

    TMessagesCollector errors;
    if (!autocodeClient.GetFines(fines, errors, GetPenaltyCheckPolicy(), limit, offset)) {
        ERROR_LOG << "Error collecting autocode fines: " << errors.GetStringReport() << Endl;
        GetNotifyHandlers()->Handle(ECollectingStatus::CollectingFail, server);
        return false;
    }

    GetNotifyHandlers()->Handle(ECollectingStatus::CollectingSuccess, server);
    GetNotifyHandlers()->Handle(ECollectingStatus::CollectedTotal, server, fines.size());

    return true;
}

IFineProcessingContextBase::TPtr TRTAutocodeFinesCollector::ConstructProcessingContext(const NDrive::IServer& server
    , const NDrive::NFine::TFineConstructor& fineConstructor
    , const TPrefechedFinesContainer& prefechedFines
    , TVector<TAutocodeFine>&& fines
) const {
    return MakeAtomicShared<TContext>(server, fineConstructor, prefechedFines, std::move(fines));
}

bool TRTAutocodeFinesCollector::ConstructFines(IFineProcessingContextBase::TPtr contextPtr, TVector<TAutocodeFineEntry>& nativeFines, TMessagesCollector& /* errors */) const {
    auto handlerContextPtr = VerifyDynamicCast<const TContext*>(contextPtr.Get());

    for (auto&& fine : handlerContextPtr->GetExternalFines()) {
        TMessagesCollector fineProcessingErrors;

        TAutocodeFineEntry nativeFine;
        if (!ProcessFine(contextPtr, fine, nativeFine, fineProcessingErrors)) {
            if (fineProcessingErrors.HasMessages()) {
                ERROR_LOG << fineProcessingErrors.GetStringReport() << Endl;
            }
            continue;
        }

        nativeFines.push_back(std::move(nativeFine));
    }

    return true;
}

bool TRTAutocodeFinesCollector::ProcessFine(const IFineProcessingContextBase::TPtr contextPtr, const TAutocodeFine& fine, TAutocodeFineEntry& nativeFine, TMessagesCollector& errors) const {
    if (!!MinRulingDate && fine.GetRulingDate() < MinRulingDate) {
        return false;
    }

    TString rulingNumber = fine.GetRulingNumber();
    if (!rulingNumber) {
        return false;
    }
    if (contextPtr->GetFineMapByRulingNumber().contains(rulingNumber)) {
        return false;
    }

    if (!contextPtr->GetFineConstructor().ConstructAutocodeFineEntry(fine, nativeFine)) {
        errors.AddMessage(__LOCATION__, TStringBuilder() << "Error converting fine to db entry: " << fine.GetRulingNumber());
        GetNotifyHandlers()->Handle(ECollectingStatus::CollectedConversionError, contextPtr->GetServer());
        return false;
    }

    return true;
}

bool TRTAutocodeFinesCollector::OnBeforeCommit(const IFineProcessingContextBase::TPtr contextPtr, const TVector<TAutocodeFineEntry>& fines, NDrive::TEntitySession& /* session */, TMessagesCollector& errors) const {
    auto autocodeContextPtr = dynamic_cast<TContext*>(contextPtr.Get());

    if (autocodeContextPtr == nullptr) {
        errors.AddMessage(__LOCATION__, "expected autocode fines processing context to process fines before commit");
        return false;
    }

    for (const auto& fine : fines) {
        if (!autocodeContextPtr->OnBeforeCommit(fine)) {
            errors.AddMessage(__LOCATION__, "error processing fine before commit: " + fine.GetId());
            return false;
        }
    }

    return true;
}

bool TRTAutocodeFinesCollector::OnAfterCommit(const IFineProcessingContextBase::TPtr contextPtr, TMessagesCollector& errors) const {
    auto autocodeContextPtr = dynamic_cast<TContext*>(contextPtr.Get());

    if (autocodeContextPtr == nullptr) {
        errors.AddMessage(__LOCATION__, "expected autocode fines processing context to process fines before commit");
        return false;
    }

    if (IsConfirmCollectingStatus()) {
        const auto& autocodeClient = autocodeContextPtr->GetServer().GetDriveAPI()->GetAutocodeClient();
        if (!autocodeClient.ConfirmFines(autocodeContextPtr->GetFineIdsToConfirm(), errors)) {
            return false;
        }
    }

    return true;
}

NDrive::TScheme TRTAutocodeFinesCollector::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);

    scheme.Add<TFSString>("min_ruling_date", "Дата, с которой забираются штрафы; все если не задано");

    scheme.Add<TFSArray>("sts_to_check", "STS автомобилей для проверки; все если не задано").SetElement<TFSString>();
    scheme.Add<TFSBoolean>("exclude_sts", "Для всех STS, кроме указанных").SetDefault(false);

    scheme.Add<TFSNumeric>("max_request_fines_count", "Ограничение на кол-во единовременно запрашиваемых штрафов; без ограничений если ноль")
          .SetMin(0)
          .SetMax(DefaultMaxRequestFinesCount)
          .SetDefault(0);
    scheme.Add<TFSVariants>("penalty_check_policy", "Способ сбора штрафов, по умолчанию все")
          .SetVariants(GetEnumAllValues<EPenaltyCheckPolicy>())
          .SetDefault(ToString(EPenaltyCheckPolicy::NotPaid));

    scheme.Add<TFSBoolean>("confirm_collecting_status", "Уведомлять об обработанных штрафах источник").SetDefault(true);

    return scheme;
}

bool TRTAutocodeFinesCollector::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    if (!TBase::DoDeserializeFromJson(data)) {
        return false;
    }

    JREAD_INSTANT_ISOFORMAT_NULLABLE_OPT(data, "min_ruling_date", MinRulingDate);

    if (!TJsonProcessor::ReadContainer(data, "sts_to_check", StsToCheck)) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "exclude_sts", ExcludeSts)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "max_request_fines_count", MaxRequestFinesCount)) {
        return false;
    }

    TString rawPenaltyCheckPolicy;
    if (!TJsonProcessor::Read(data, "penalty_check_policy", rawPenaltyCheckPolicy)) {
        return false;
    }
    if (!TryFromString(rawPenaltyCheckPolicy, PenaltyCheckPolicy)) {
        return false;
    }

    if (!TJsonProcessor::Read(data, "confirm_collecting_status", ConfirmCollectingStatus)) {
        return false;
    }

    return true;
}

NJson::TJsonValue TRTAutocodeFinesCollector::DoSerializeToJson() const {
    NJson::TJsonValue result = TBase::DoSerializeToJson();

    TJsonProcessor::Write(result, "min_ruling_date", (!!MinRulingDate) ? NJson::TJsonValue(MinRulingDate.ToString()) : NJson::JSON_NULL);

    TJsonProcessor::WriteContainerArray(result, "sts_to_check", StsToCheck);
    TJsonProcessor::Write(result, "exclude_sts", ExcludeSts);

    TJsonProcessor::Write(result, "max_request_fines_count", MaxRequestFinesCount);
    TJsonProcessor::WriteAsString(result, "penalty_check_policy", PenaltyCheckPolicy);

    TJsonProcessor::Write(result, "confirm_collecting_status", ConfirmCollectingStatus);

    return result;
}
