#include "charge_state_handler.h"

#include <drive/backend/data/billing_tags.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/fines/manager.h>

#include <rtline/util/algorithm/ptr.h>

#include <util/generic/algorithm.h>

TString TRTFineChargeHandler::GetTypeName() {
    return "fines_charge_state_hanlder";
}

TString TRTFineChargeHandlerState::GetTypeName() {
    return TRTFineChargeHandler::GetTypeName();
}

IRTBackgroundProcess::TFactory::TRegistrator<TRTFineChargeHandler> TRTFineChargeHandler::Registrator(TRTFineChargeHandler::GetTypeName());
TRTHistoryWatcherState::TFactory::TRegistrator<TRTFineChargeHandlerState> TRTFineChargeHandlerState::Registrator(TRTFineChargeHandlerState::GetTypeName());

TRTFineChargeHandler::TRTFineChargeHandler()
    : TBase()
    , TagEventsHandler(TEventsHandler(NEntityTagsManager::EEntityType::User))
{
    InitNotifyHandlers();
}

bool TRTFineChargeHandler::DoStart(const TRTBackgroundProcessContainer& container) {
    if (!TBase::DoStart(container)) {
        return false;
    }
    if (!NotifyHandlers) {
        return false;
    }
    NotifyHandlers->SetSignalName(GetRTProcessName());
    return true;
}

void TRTFineChargeHandler::InitNotifyHandlers() {
    if (!NotifyHandlers) {
        NotifyHandlers = MakeAtomicShared<TNotifyHandlers>(GetType());
    }
}

TExpectedState TRTFineChargeHandler::DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const {
    const NDrive::IServer& server = context.GetServerAs<NDrive::IServer>();

    auto nativeState = std::dynamic_pointer_cast<TState>(state);
    const ui64 lastProcessedEventId = (!!nativeState) ? nativeState->GetLastEventId() : std::numeric_limits<ui64>::max();

    const bool fetchResult = TagEventsHandler.Fetch(server, lastProcessedEventId, DataActuality);

    if (ExplicitChargeTimeCheckConfig.HasRestrictions()) {
        if (!HandleFines(server)) {
            return BuildState(TagEventsHandler.GetLastProcessedEventId());
        }
    }

    if (fetchResult) {
        HandleHistoryEvents(server);
    }

    return BuildState((RetryOnError) ? TagEventsHandler.GetLastProcessedEventId() : TagEventsHandler.GetLastProducedEventId());
}

TAtomicSharedPtr<TRTFineChargeHandler::TState> TRTFineChargeHandler::BuildState(const ui64 lastEventId) const {
    auto state = MakeAtomicShared<TState>();
    state->SetLastEventId(lastEventId);
    return state;
}

bool TRTFineChargeHandler::HandleFines(const NDrive::IServer& server) const {
    const auto& finesManager = server.GetDriveAPI()->GetFinesManager();
    const auto& userTagsManager = server.GetDriveAPI()->GetTagsManager().GetUserTags();

    NDrive::NFine::TFineFilterGroup filters = { MakeAtomicShared<NDrive::NFine::TFineChargeNotPassedFilter>() };

    if (ExplicitChargeTimeCheckConfig.HasRestrictions()) {
        filters.Append(ExplicitChargeTimeCheckConfig.GetFilter());
    }

    TVector<NDrive::NFine::TAutocodeFineEntry> fines;
    {
        auto tx = finesManager.BuildTx<NSQL::ReadOnly>();
        auto fines = finesManager.GetFines(filters, tx);
        R_ENSURE(fines, {}, "Fail to fetch fines for ElementFinesCollector", tx);
        fines = std::move(*fines);
    }

    for (auto&& fine : fines) {
        const auto relatedChargeTagIdJson = fine.GetMetaInfoProperty(NDrive::NFine::TAutocodeFineEntry::EMetaInfoProperty::ChargeTagId);

        const auto& relatedChargeTagId = relatedChargeTagIdJson.GetString();
        if (!relatedChargeTagId) {
            continue;
        }

        auto session = userTagsManager.BuildSession(/* readOnly = */ true);

        TVector<TDBTag> tags;
        if (userTagsManager.RestoreTags(relatedChargeTagId, tags, session) && !tags.empty()) {
            continue;
        }

        TMessagesCollector errors;

        auto status = MarkFineCharged(server, fine.GetId(), errors);
        NotifyHandlers->Handle(status, server);

        if (errors.HasMessages()) {
            ERROR_LOG << errors.GetStringReport() << Endl;
        }

        if (status != EHandlingStatus::AlreadyMarked && status != EHandlingStatus::MarkSuccessful) {
            return false;
        }
    }

    return true;
}

bool TRTFineChargeHandler::HandleHistoryEvents(const NDrive::IServer& server) const {
    while (auto event = TagEventsHandler.Next(server)) {
        TMessagesCollector errors;

        auto status = HandleEvent(server, event, errors);

        if (errors.HasMessages()) {
            ERROR_LOG << errors.GetStringReport() << Endl;
        }

        if (status != EHandlingStatus::SkipFine) {
            NotifyHandlers->Handle(status, server);

            if (status == EHandlingStatus::UpsertError) {
                return false;
            }
        }
    }
    return true;
}

TRTFineChargeHandler::EHandlingStatus TRTFineChargeHandler::HandleEvent(const NDrive::IServer& server, TAtomicSharedPtr<TUserTagHistoryEvent> ev, TMessagesCollector& errors) const {
    // actually a new tag type to be introduced to store all fine related data
    auto billingTag = ev->GetTagAs<TBillingTag>();
    if (!billingTag) {
        errors.AddMessage(__LOCATION__, "Billing tag expected: " + ev->GetTagId());
        return EHandlingStatus::InvalidTagData;
    }

    const bool isCharged = (billingTag->GetResolution() == "finished");
    if (!isCharged) {
        return EHandlingStatus::SkipFine;
    }

    NJson::TJsonValue relatedFineData(NJson::JSON_MAP);

    const TString& tagComment = billingTag->GetComment();
    if (!NJson::ReadJsonTree(tagComment, &relatedFineData)) {
        errors.AddMessage(__LOCATION__, "Cannot interpret tag data as json: " + ev->GetTagId());
        return EHandlingStatus::InvalidTagData;
    }

    TString relatedFineId;
    if (!TJsonProcessor::Read(relatedFineData, "fine_id", relatedFineId)) {
        errors.AddMessage(__LOCATION__, "Expected fine id property does not present in tag comment: " + ev->GetTagId());
        return EHandlingStatus::InvalidTagData;
    }

    return MarkFineCharged(server, relatedFineId, errors);
}

TRTFineChargeHandler::EHandlingStatus TRTFineChargeHandler::MarkFineCharged(const NDrive::IServer& server, const TString& fineId, TMessagesCollector& errors) const {
    const auto& finesManager = server.GetDriveAPI()->GetFinesManager();

    auto tx = finesManager.BuildTx<NSQL::Writable>();
    auto fine = finesManager.GetFineById(fineId, tx);
    if (!fine) {
        errors.AddMessage(__LOCATION__, "Error fetching fine " + fineId + ": " + tx.GetStringReport());
        return EHandlingStatus::UpsertError;
    }

    if (!!fine->GetChargePassedAt()) {
        errors.AddMessage(__LOCATION__, "Fine " + fine->GetId() + " is marked as charged already");
        return EHandlingStatus::AlreadyMarked;
    }

    fine->SetChargePassedAt(ModelingNow());

    if (!finesManager.UpsertFine(*fine, tx) || !tx.Commit()) {
        errors.AddMessage(__LOCATION__, "Error upserting fine " + fine->GetId() + ": " + tx.GetStringReport());
        return EHandlingStatus::UpsertError;
    }

    return EHandlingStatus::MarkSuccessful;
}

NDrive::TScheme TRTFineChargeHandler::DoGetScheme(const IServerBase& server) const {
    auto scheme = TBase::DoGetScheme(server);
    scheme.Add<TFSArray>("tag_name_patterns", "Регулярные выражения имени тега").SetElement<TFSString>();
    scheme.Add<TFSBoolean>("retry_on_error", "Повторять обработку тега в случае ошибки").SetDefault(true);
    scheme.Add<TFSStructure>("explicit_charge_time_check_config", "Явная проверка статуса с такого момента").SetStructure(decltype(ExplicitChargeTimeCheckConfig)::GetScheme());
    scheme.Add<TFSStructure>("notify_handlers", "Настройки нотификаторов").SetStructure(TNotifyHandlers::GetScheme(server));
    scheme.Add<TFSDuration>("fine_fetch_deep", "Период выборки штрафов из базы").SetDefault(FinesHistoryDeep);
    return scheme;
}

bool TRTFineChargeHandler::DoDeserializeFromJson(const NJson::TJsonValue& data) {
    if (!TJsonProcessor::ReadContainer(data, "tag_name_patterns", TagNamePatterns) || !InitTagEventsHandler(TagNamePatterns) || TagEventsHandler.Empty()) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "retry_on_error", RetryOnError)) {
        return false;
    }
    if (!ExplicitChargeTimeCheckConfig.DeserializeFromJson(data["explicit_charge_time_check_config"])) {
        return false;
    }
    if (!!NotifyHandlers && !NotifyHandlers->DeserializeFromJson(data["notify_handlers"])) {
        return false;
    }
    if (!TJsonProcessor::Read(data, "fine_fetch_deep", FinesHistoryDeep)) {
        return false;
    }
    return TBase::DoDeserializeFromJson(data);
}

NJson::TJsonValue TRTFineChargeHandler::DoSerializeToJson() const {
    auto result = TBase::DoSerializeToJson();
    TJsonProcessor::WriteContainerArray(result, "tag_name_patterns", TagNamePatterns);
    TJsonProcessor::Write(result, "retry_on_error", RetryOnError);
    TJsonProcessor::WriteSerializable(result, "explicit_charge_time_check_config", ExplicitChargeTimeCheckConfig);
    if (!!NotifyHandlers) {
        TJsonProcessor::WriteSerializable(result, "notify_handlers", *NotifyHandlers);
    }
    TJsonProcessor::WriteAsString(result, "fine_fetch_deep", FinesHistoryDeep);
    return result;
}

bool TRTFineChargeHandler::InitTagEventsHandler(const TSet<TString>& patterns) {
    TagEventsHandler.Clear();
    for (const auto& pattern : patterns) {
        TagEventsHandler.AddFilter(MakeAtomicShared<TagNameRegexpEventsFilter>(EObjectHistoryAction::Remove, pattern));
    }
    return TagEventsHandler.AreFiltersValid();
}
