#pragma once

#include "events.h"
#include "statuses.h"

#include <drive/backend/rt_background/common/config.h>
#include <drive/backend/rt_background/common/state.h>

#include <drive/backend/fines/filters.h>
#include <drive/backend/fines/autocode/entries.h>
#include <drive/backend/fines/context_fetchers/fine_context.h>
#include <drive/backend/notifications/collection.h>
#include <drive/backend/proto/background.pb.h>

#include <library/cpp/regex/pcre/regexp.h>

#include <rtline/library/unistat/cache.h>
#include <rtline/util/types/accessor.h>

// consider to generalize fine and tag interaction using common alert schemes

class TRTFineChargeHandlerState : public TRTHistoryWatcherState {
private:
    static TFactory::TRegistrator<TRTFineChargeHandlerState> Registrator;

public:
    static TString GetTypeName();

    virtual TString GetType() const override {
        return GetTypeName();
    }
};

class TRTFineChargeHandler: public IRTRegularBackgroundProcess {
    using TBase = IRTRegularBackgroundProcess;
    using TState = TRTFineChargeHandlerState;

    using EHandlingStatus = NDrive::NFine::EChargeStateHandlingStatus;

    using TNotifyHandlers = TContextNotifyHandlerCollection<EHandlingStatus, NDrive::NFine::IFineContextFetcher>;

public:
    TRTFineChargeHandler();

    static TString GetTypeName();

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual NDrive::TScheme DoGetScheme(const IServerBase& server) const override;
    virtual bool DoDeserializeFromJson(const NJson::TJsonValue& jsonInfo) override;
    virtual NJson::TJsonValue DoSerializeToJson() const override;

protected:
    virtual bool DoStart(const TRTBackgroundProcessContainer& container) override;
    virtual void InitNotifyHandlers() final;

    virtual TExpectedState DoExecute(TAtomicSharedPtr<IRTBackgroundProcessState> state, const TExecutionContext& context) const override;

    virtual TAtomicSharedPtr<TState> BuildState(const ui64 lastEventId) const;

    bool HandleFines(const NDrive::IServer& server) const;

    bool HandleHistoryEvents(const NDrive::IServer& server) const;
    EHandlingStatus HandleEvent(const NDrive::IServer& server, TAtomicSharedPtr<TUserTagHistoryEvent> ev, TMessagesCollector& errors) const;

    EHandlingStatus MarkFineCharged(const NDrive::IServer& server, const TString& fineId, TMessagesCollector& errors) const;

    bool InitTagEventsHandler(const TSet<TString>& patterns);

private:
    static TFactory::TRegistrator<TRTFineChargeHandler> Registrator;

    R_READONLY(TSet<TString>, TagNamePatterns, { "^fine_pdd_[\\d\\_]+$" });
    R_READONLY(bool, RetryOnError, true);
    R_FIELD(TDuration, FinesHistoryDeep, TDuration::Days(30));

    R_READONLY(NDrive::NFine::TFineChargeTimeFilterConfig, ExplicitChargeTimeCheckConfig);

    R_FIELD(TNotifyHandlers::TPtr, NotifyHandlers, nullptr);

    mutable TEventsHandler TagEventsHandler;
};
